package ru.yandex.calendar.logic.event.repetition;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.MutableDateTime;
import org.joda.time.ReadablePeriod;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.misc.lang.Validate;

/**
 * @author Stepan Koltsov
 */
public class RegularRepetition {
    private final Repetition repetition;
    private final DateTime firstInstanceStart;

    // caches
    private final RegularRepetitionRule rule;
    private final DateTimeZone tz;

    public RegularRepetition(Repetition repetition, DateTime firstInstanceStart) {
        Validate.notNull(repetition);
        Validate.notNull(firstInstanceStart);

        this.repetition = repetition;
        this.firstInstanceStart = firstInstanceStart;

        this.rule = RegularRepetitionRule.find(repetition);
        this.tz = firstInstanceStart.getZone();
    }

    public void alignToUnit(MutableDateTime dt) {
        dt.setRounding(rule.getAlignment(dt.getChronology()));
    }

    public void jumpToBoundByUnits(MutableDateTime dt, Instant bound) {
        final int unitsBetween = RepetitionUtils.unitsBetween(dt.toDateTime(), bound.toDateTime(dt.getZone()), rule);
        final int rEach = RepetitionUtils.calcREach(repetition);
        final int unitsMultiplier = unitsBetween / rEach * rEach;
        final ReadablePeriod unit = rule.getUnit();
        dt.add(unit, unitsMultiplier);
    }

    public void addUnit(MutableDateTime curUnitStartDt) {
        curUnitStartDt.add(rule.getUnit(), RepetitionUtils.calcREach(repetition));
    }

    public ListF<LocalDate> eventDatesInUnit(LocalDate unitStart) {
        final Long dueMs = RepetitionUtils.getDueMs(repetition);
        return RepetitionUtils.eventDatesInUnitCandidates(rule, unitStart, firstInstanceStart, repetition).filter(new Function1B<LocalDate>() {
            public boolean apply(LocalDate ri) {
                return ri.toDateTimeAtStartOfDay(tz).getMillis() > firstInstanceStart.getMillis()
                        && (dueMs == null || ri.toDateTimeAtStartOfDay(tz).getMillis() < dueMs);
            }
        });
    }

} //~
