package ru.yandex.calendar.logic.event.repetition;

import org.joda.time.Chronology;
import org.joda.time.DateTimeField;
import org.joda.time.DateTimeFieldType;
import org.joda.time.Days;
import org.joda.time.Months;
import org.joda.time.ReadablePeriod;
import org.joda.time.Weeks;
import org.joda.time.Years;

import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.Freq;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.IcsRecur;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartByDay;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.enums.StringEnumResolver;

/**
 * Everything that relates repetition regular rule specific data
 * (in other words, everything but common fields like 'r-each')
 * @author ssytnik
 */
public enum RegularRepetitionRule implements StringEnum {
    DAILY(DateTimeFieldType.dayOfMonth(), Days.ONE, Freq.DAILY),
    WEEKLY(DateTimeFieldType.weekOfWeekyear(), Weeks.ONE, Freq.WEEKLY),
    MONTHLY_NUMBER(DateTimeFieldType.monthOfYear(), Months.ONE, Freq.MONTHLY),
    MONTHLY_DAY_WEEKNO(DateTimeFieldType.monthOfYear(), Months.ONE, Freq.MONTHLY),
    YEARLY(DateTimeFieldType.year(), Years.ONE, Freq.YEARLY),
    NONE,
    ;

    private final DateTimeFieldType dateTimeFieldType;
    private final ReadablePeriod unit;
    private final Freq icalFreq;

    private RegularRepetitionRule() {
        this.dateTimeFieldType = null;
        this.unit = null;
        this.icalFreq = null;
    }

    private RegularRepetitionRule(DateTimeFieldType dateTimeFieldType, ReadablePeriod unit, Freq icalFreq) {
        this.dateTimeFieldType = dateTimeFieldType;
        this.unit = unit;
        this.icalFreq = icalFreq;
    }

    /////////////////////////////////////////////
    // Methods for calculating event instances //
    /////////////////////////////////////////////

    protected DateTimeField getAlignment(Chronology chrono) {
        return dateTimeFieldType.getField(chrono);
    }

    public ReadablePeriod getUnit() {
        return unit;
    }

    public Freq getICalFreq() {
        return icalFreq;
    }

    public String getRType() {
        return name().toLowerCase();
    }

    /**
     * Finds RRule by its database repetition "type" field.
     * Performed search is case-insensitive.
     * @param rType database repetition type field
     * @return enum instance (or runtime error is thrown)
     * @throws CommandRunException if instance not exists
     */
    public static RegularRepetitionRule find(Freq freq) {
        for (RegularRepetitionRule rule : values()) {
            if (rule.icalFreq == freq) {
                return rule;
            }
        }
        throw new IllegalArgumentException("unsuported freq: " + freq);
    }

    public static RegularRepetitionRule find(Repetition r) {
        return r.getType();
    }

    public static RegularRepetitionRule find(IcsRecur r) {
        // JIC (because freq. cannot be null and must come upper-case)
        final Freq freq = r.getFreq();
        if (Freq.MONTHLY == freq) {
            return r.getPartByType(IcsRecurRulePartByDay.class).isPresent() ? MONTHLY_DAY_WEEKNO : MONTHLY_NUMBER;
        } else {
            return find(freq);
        }
    }

    @Override
    public String value() {
        return name().toLowerCase();
    }

    public static final StringEnumResolver<RegularRepetitionRule> R = StringEnumResolver.r(RegularRepetitionRule.class);

}
