package ru.yandex.calendar.logic.event.repetition;

import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.Days;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.RepetitionConfirmation;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventInfo;
import ru.yandex.calendar.logic.event.EventInfoDbLoader;
import ru.yandex.calendar.logic.event.EventLoadLimits;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.web.EventWebManager;
import ru.yandex.calendar.logic.event.web.UserOrYaCalendar;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.sending.param.MessageParameters;
import ru.yandex.calendar.logic.sending.param.RepetitionConfirmationMessageParameters;
import ru.yandex.calendar.logic.sending.real.MailSender;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.svc.SvcRoutines;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.calendar.util.exception.ExceptionUtils;
import ru.yandex.calendar.util.resources.UStringLiteral;
import ru.yandex.commune.mail.MailAddress;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.TimeUtils;

public class RepetitionConfirmationManager {
    private static final Logger logger = LoggerFactory.getLogger(RepetitionConfirmationManager.class);

    private static final Duration UNCONFIRMED_REPETITION_DURATION = Days.days(180).toStandardDuration();
    private static final ListF<Duration> CONFIRMATION_EMAIL_OFFSETS = Cf.list(
            Days.days(30).toStandardDuration(),
            Days.days(14).toStandardDuration(),
            Days.days(5).toStandardDuration(),
            Days.days(1).toStandardDuration());

    @Autowired
    private RepetitionConfirmationDao repetitionConfirmationDao;
    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EventWebManager eventWebManager;
    @Autowired
    private SvcRoutines svcRoutines;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private MailSender mailSender;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private EventInfoDbLoader eventInfoDbLoader;
    @Autowired
    private ResourceDao resourceDao;


    public void confirm(long mainEventId, ActionInfo actionInfo) {
        Event master = eventRoutines.findMasterEventByMainId(mainEventId).get();

        EventWithRelations masterEvent = eventDbManager.getEventWithRelationsByEvent(master);
        RepetitionInstanceInfo repetitionInfo = repetitionRoutines.getRepetitionInstanceInfo(masterEvent);

        rescheduleConfirmation(masterEvent, repetitionInfo, actionInfo);
    }

    public boolean isItTimeToConfirm(long mainEventId, ActionInfo actionInfo) {
        return findWhereTimeToConfirm(Cf.list(mainEventId), actionInfo).isNotEmpty();
    }

    public ListF<Long> findWhereTimeToConfirm(ListF<Long> mainEventIds, ActionInfo actionInfo) {
        ListF<RepetitionConfirmation> confirmations =
                repetitionConfirmationDao.findRepetitionConfirmationsByMainEventIds(mainEventIds);

        return confirmations.filterMap(rc -> Option.when(
                rc.getDeadline().isBefore(actionInfo.getNow().plus(CONFIRMATION_EMAIL_OFFSETS.first())),
                rc.getMainEventId()));
    }

    public void rescheduleConfirmation(
            EventWithRelations event, RepetitionInstanceInfo repetitionInfo, ActionInfo actionInfo)
    {
        rescheduleConfirmation(event, repetitionInfo, UNCONFIRMED_REPETITION_DURATION, actionInfo);
    }

    private void rescheduleConfirmation(
            EventWithRelations event, RepetitionInstanceInfo repetitionInfo,
            Duration deadlineOffset, ActionInfo actionInfo)
    {
        if (!passportAuthDomainsHolder.containsYandexTeamRu()) return;

        if (event.getRecurrenceId().isPresent()) return;

        Instant deadline = ObjectUtils.max(
                event.getEvent().getStartTs().plus(UNCONFIRMED_REPETITION_DURATION),
                actionInfo.getNow().plus(deadlineOffset));

        if (event.getResourceIds().isNotEmpty()
            && event.getParticipants().isMeeting()
            && event.getParticipants().getOrganizer().isYandexUser()
            && RepetitionUtils.hasInstanceAfter(repetitionInfo, deadline))
        {
            RepetitionConfirmation data = new RepetitionConfirmation();
            data.setMainEventId(event.getMainEventId());
            data.setDeadline(deadline);
            data.setNextQueryTs(deadline.minus(CONFIRMATION_EMAIL_OFFSETS.first()));
            repetitionConfirmationDao.replaceRepetitionConfirmation(data);

        } else {
            repetitionConfirmationDao.deleteRepetitionConfirmationByMainEventId(event.getMainEventId());
        }
    }

    public void deleteConfirmationsByEventIds(ListF<Long> eventIds) {
        ListF<Event> events = eventDbManager.getEventsByIds(eventIds);
        ListF<Event> masterEvents = events.filter(Event.getRecurrenceIdF().andThen(Cf2.f1B(r -> !r.isPresent())));

        repetitionConfirmationDao.deleteRepetitionConfirmationsByMainEventIds(masterEvents.map(Event.getMainEventIdF()));
    }

    public void sendConfirmationEmailsAndKillUnconfirmed(final ActionInfo actionInfo) {
        Function1V<RepetitionConfirmation> callback = new Function1V<RepetitionConfirmation>() {
            public void apply(RepetitionConfirmation confirmation) {
                try {
                    sendConfirmationEmailOrKillUnconfirmed(confirmation, actionInfo);
                } catch (Exception e) {
                    ExceptionUtils.rethrowIfTlt(e);
                    logger.error("Error occurred while processing confirmation " + confirmation, e);
                }
            }
        };
        repetitionConfirmationDao.findRepetitionConfirmationsQueriedBefore(actionInfo.getNow(), callback);
    }

    public void rescheduleAllConfirmations() {
        ActionInfo actionInfo = ActionInfo.adminManager();

        Check.isTrue(passportAuthDomainsHolder.containsYandexTeamRu());

        for (ListF<Long> resourceIds : resourceDao.findResources().map(Resource.getIdF()).paginate(5)) {
            InfiniteInterval interval = new InfiniteInterval(actionInfo.getNow(), Option.<Instant>empty());
            EventLoadLimits limits = EventLoadLimits.intersectsInterval(interval);

            ListF<EventAndRepetition> eventsWithRepetitions = eventInfoDbLoader
                    .getEventsOnResources(resourceIds, limits)
                    .filter(er -> er.getRepetitionInfo().getRepetition().isPresent());

            for (EventInfo event : eventInfoDbLoader.getEventInfosByEventsAndRepetitions(
                    Option.empty(), eventsWithRepetitions, actionInfo.getActionSource()))
            {
                rescheduleConfirmation(
                        event.getEventWithRelations(), event.getRepetitionInstanceInfo(),
                        CONFIRMATION_EMAIL_OFFSETS.first(), actionInfo);
            }
        }
    }

    private void sendConfirmationEmailOrKillUnconfirmed(RepetitionConfirmation confirmation, ActionInfo actionInfo) {
        Instant deadline = confirmation.getDeadline();
        Duration currentOffset = new Duration(confirmation.getNextQueryTs(), deadline);
        Option<Duration> nextOffset = CONFIRMATION_EMAIL_OFFSETS.find(TimeUtils.duration.isShorterThan(currentOffset));

        if (!confirmation.getNextQueryTs().isBefore(deadline)) {
            removeFutureInstances(confirmation, actionInfo);
            repetitionConfirmationDao.deleteRepetitionConfirmationByMainEventId(confirmation.getMainEventId());

        } else {
            sendConfirmationEmail(confirmation, actionInfo);
            repetitionConfirmationDao.updateRepetitionConfirmationNextQueryTsByIdAndNextQueryTs(
                    deadline.minus(nextOffset.getOrElse(Duration.ZERO)),
                    confirmation.getId(), confirmation.getNextQueryTs());
        }
    }

    private void sendConfirmationEmail(RepetitionConfirmation confirmation, ActionInfo actionInfo) {
        long masterEventId = eventRoutines.findMasterEventByMainId(confirmation.getMainEventId()).get().getId();

        EventWithRelations event = eventDbManager.getEventWithRelationsById(masterEventId);
        RepetitionInstanceInfo repetitionInfo = repetitionRoutines.getRepetitionInstanceInfo(event);

        Participants participants = event.getParticipants();
        ParticipantInfo organizer = participants.getOrganizer();

        Language language = settingsRoutines.getLanguage(organizer.getUid().get());
        DateTimeZone tz = dateTimeManager.getTimeZoneForUid(organizer.getUid().get());

        String eventName = event.getEvent().getName();
        String repetition = RepetitionToStringConverter.convert(repetitionInfo, language);
        String roomName = event.getResources().filterMap(ResourceInfo.getNameI18nF(language)).first();
        LocalDateTime deadline = new LocalDateTime(confirmation.getDeadline(), tz);

        Email senderEmail = svcRoutines.getCalendarInfoEmail(PassportAuthDomain.byUid(organizer.getUid().get()));
        MailAddress sender = new MailAddress(senderEmail, UStringLiteral.YA_CALENDAR);
        MailAddress recipient = new MailAddress(organizer.getEmail(), organizer.getName());

        String calendarUrl = svcRoutines.getCalendarUrlForUid(organizer.getUid().get());
        MessageParameters mail = new RepetitionConfirmationMessageParameters(
                language, masterEventId, eventName, repetition, roomName, deadline, sender, recipient, calendarUrl);

        mailSender.sendEmailsViaTask(Cf.list(mail), actionInfo);
    }

    private void removeFutureInstances(RepetitionConfirmation confirmation, ActionInfo actionInfo) {
        val masterEventId = eventRoutines.findMasterEventByMainId(confirmation.getMainEventId()).get().getId();
        eventWebManager.deleteFutureEvents(UserOrYaCalendar.yaCalendar(), masterEventId, actionInfo);
    }
}
