package ru.yandex.calendar.logic.event.repetition;

import org.jdom.Element;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.util.dates.WeekdayConv;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

/**
 * Should replace RepetitionRoutines.addElms
 * @author ssytnik
 */
@BenderBindAllFields
public class RepetitionHints extends DefaultObject {
    private final RegularRepetitionRule type;
    private final int each;
    private final ListF<Integer> weekdays; // WEEKLY, MONTHLY_DAY_WEEKNO
    private final int day; // MONTHLY_NUMBER, YEARLY
    private final int weekNoOrLastWeek; // MONTHLY_DAY_WEEKNO; -1 <=> last
    private final int month; // YEARLY
    private final Option<LocalDate> dueO;


    public RepetitionHints(
            RegularRepetitionRule type, int each, ListF<Integer> weekdays,
            int day, int weekNoOrLastWeek, int month, Option<LocalDate> dueO)
    {
        String diagErrorMessage = "Wrong weekends for type '" + type.value() + "': " + weekdays.mkString(", ");
        Validate.isTrue(type != RegularRepetitionRule.WEEKLY || weekdays.size() >= 1, diagErrorMessage);
        Validate.isTrue(type != RegularRepetitionRule.MONTHLY_DAY_WEEKNO || weekdays.size() == 1, diagErrorMessage);

        this.type = type;
        this.each = each;
        this.weekdays = weekdays;
        this.day = day;
        this.weekNoOrLastWeek = weekNoOrLastWeek;
        this.month = month;
        // XXX ssytnik: due is not inclusive in calendar.
        // Replace this temporary hack with true last instance date.
        this.dueO = dueO.map(new Function<LocalDate, LocalDate>() {
            public LocalDate apply(LocalDate a) {
                return a.minusDays(1);
            }
        });
    }


    public static RepetitionHints createDefaults(DateTime d) {
        return new RepetitionHints(
                RegularRepetitionRule.DAILY, 1,
                Cf.list(d.getDayOfWeek()), d.getDayOfMonth(),
                getWeekNoOrLastWeek(d, RepetitionUtils.isLastWeekDayInMonth(d)),
                d.getMonthOfYear(), Option.<LocalDate>empty()
                );
    }

    public static RepetitionHints createByRepetition(DateTime d, Repetition r) {
        Validate.isTrue(r.getType() != RegularRepetitionRule.NONE);

        return new RepetitionHints(
                r.getType(), RepetitionUtils.calcREach(r),
                getWeekdays(d, r), d.getDayOfMonth(),
                getWeekNoOrLastWeek(d, r.getRMonthlyLastweek().getOrElse(false)),
                d.getMonthOfYear(), getDueO(r, d.getZone())
                );
    }

    public static RepetitionHints create(DateTime d, Option<Repetition> rO) {
        return rO.isPresent() ? createByRepetition(d, rO.get()) : createDefaults(d);
    }


    private static ListF<Integer> getWeekdays(DateTime d, Repetition r) {
        if (r.getType() == RegularRepetitionRule.MONTHLY_DAY_WEEKNO) {
            return Cf.list(d.getDayOfWeek());
        }

        String str = r.getRWeeklyDays().getOrElse("");
        return Cf.x(StringUtils.split(str, ",")).map(new Function<String, Integer>() {
            public Integer apply(String calsWeekday) {
                return WeekdayConv.calsToJoda(calsWeekday);
            }
        });
    }

    private static int getWeekNoOrLastWeek(DateTime d, boolean isLastWeek) {
        int weekNo = RepetitionUtils.getNumWeekDaysForDate(d);
        return isLastWeek || weekNo > 4 ? -1 : weekNo;
    }

    private static Option<LocalDate> getDueO(Repetition r, final DateTimeZone tz) {
        return r.getDueTs().map(new Function<Instant, LocalDate>() {
            public LocalDate apply(Instant a) {
                return new LocalDate(a.getMillis(), tz);
            }
        });
    }


    public Element toXml(String name) {
        Element res = new Element(name);

        CalendarXmlizer.appendElm(res, "type", type.value());
        CalendarXmlizer.appendElm(res, "each", each);
        if (type == RegularRepetitionRule.WEEKLY || type == RegularRepetitionRule.MONTHLY_DAY_WEEKNO) {
            CalendarXmlizer.appendElmColl(res, null, "weekday", weekdays);
        }
        if (type == RegularRepetitionRule.MONTHLY_NUMBER || type == RegularRepetitionRule.YEARLY) {
            CalendarXmlizer.appendElm(res, "day", day);
        }
        if (type == RegularRepetitionRule.MONTHLY_DAY_WEEKNO) {
            CalendarXmlizer.appendElm(res, "week-no", weekNoOrLastWeek);
        }
        if (type == RegularRepetitionRule.YEARLY) {
            CalendarXmlizer.appendElm(res, "month", month);
        }
        if (dueO.isPresent()) {
            CalendarXmlizer.appendElm(res, "due", dueO.get(), null);
        }

        return res;
    }

}
