package ru.yandex.calendar.logic.event.repetition;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.Interval;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.Period;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.RdateFields;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

/**
 * Stores all beans, which are used for calculation repeating event instances
 * @author akirakozov
 */
public class RepetitionInstanceInfo {
    private final InstantInterval eventInterval;
    private final DateTimeZone tz;
    private final Option<Repetition> repetition;
    private final ListF<Rdate> rdates;
    private final ListF<Rdate> exdates;
    private final ListF<RecurrenceTimeInfo> recurrences;

    public RepetitionInstanceInfo(
            InstantInterval eventInterval, DateTimeZone tz, Option<Repetition> repetition)
    {
        this(eventInterval, tz, repetition, Cf.list(), Cf.list(), Cf.list());
    }

    public RepetitionInstanceInfo(
            InstantInterval eventInterval, DateTimeZone tz,
            Option<Repetition> repetition, ListF<Rdate> rdates,
            ListF<Rdate> exdates, ListF<RecurrenceTimeInfo> recurrences)
    {
        Validate.notNull(eventInterval);
        Validate.notNull(tz);
        Validate.notNull(rdates);
        Validate.notNull(repetition);
        Validate.notNull(exdates);
        Validate.notNull(recurrences);

        this.eventInterval = eventInterval;
        this.tz = tz;
        this.repetition = repetition;
        this.rdates = rdates;
        this.exdates = exdates;
        this.recurrences = recurrences;
    }

    public static RepetitionInstanceInfo noRepetition(InstantInterval eventInterval, DateTimeZone tz) {
        return new RepetitionInstanceInfo(eventInterval, tz, Option.empty(), Cf.list(), Cf.list(), Cf.list());
    }

    public static RepetitionInstanceInfo noRepetition(InstantInterval eventInterval) {
        return noRepetition(eventInterval, DateTimeZone.UTC);
    }

    public InstantInterval getEventInterval() {
        return eventInterval;
    }

    public Instant getEventStart() {
        return getEventInterval().getStart();
    }

    public Interval getEventIntervalWithTz() {
        return eventInterval.toInterval(tz);
    }

    public DateTimeZone getTz() {
        return tz;
    }

    public Period getEventPeriod() {
        return eventInterval.toPeriod(tz);
    }

    public Option<Repetition> getRepetition() {
        return repetition;
    }
    public ListF<Rdate> getRdates() {
        return rdates;
    }
    public ListF<Rdate> getExdates() {
        return exdates;
    }
    public ListF<Instant> getRecurIds() {
        return recurrences.map(RecurrenceTimeInfo::getRecurrenceId);
    }

    public Option<LocalDate> getUntilDate() {
        return repetition.isPresent() && repetition.get().getDueTs().isPresent()
                ? Option.of(EventRoutines.convertDueTsToUntilDate(repetition.get().getDueTs().get(), tz))
                : Option.<LocalDate>empty();
    }

    public ListF<Instant> getExdateStarts() {
        return exdates.map(Rdate.getStartTsF());
    }

    public boolean isEmpty() {
        // exdates and recurIds can not exist without
        // repetition or rdates
        return !repetition.isPresent() && rdates.isEmpty();
    }

    public Option<Instant> getRepetitionEndOrElseEventEnd() {
        return RepetitionUtils.calcCommonInstsInterval(this).getEnd();
    }

    public boolean goesOnAfter(Instant ts) {
        return !getRepetitionEndOrElseEventEnd().exists(ts::isAfter);
    }

    public static RepetitionInstanceInfo create(InstantInterval eventInterval, DateTimeZone tz, Repetition repetition) {
        return new RepetitionInstanceInfo(
                eventInterval, tz, Option.of(repetition), Cf.list(), Cf.list(), Cf.list());
    }

    public static RepetitionInstanceInfo create(InstantInterval eventInterval, DateTimeZone tz, Option<Repetition> repetition) {
        return new RepetitionInstanceInfo(eventInterval, tz, repetition, Cf.list(), Cf.list(), Cf.list());
    }

    public static RepetitionInstanceInfo create(Interval eventInterval, Option<Repetition> repetition) {
        return create(InstantInterval.valueOf(eventInterval), eventInterval.getChronology().getZone(), repetition);
    }

    public SetF<Instant> getExcludeInstants(boolean ignoreExdates) {
        ListF<Instant> exdateInstants = ignoreExdates ? Cf.list() : exdates.map(RdateFields.START_TS.getF());
        return getRecurIds().plus(exdateInstants).unique();
    }

    public RepetitionInstanceInfo withExdates(ListF<Rdate> newExdates) {
        return new RepetitionInstanceInfo(eventInterval, tz, repetition, rdates, newExdates, recurrences);
    }

    public RepetitionInstanceInfo excludeExdates(ListF<Instant> starts) {
        return withExdates(getExdates().filterNot(Rdate.getStartTsF().andThen(starts.containsF())));
    }

    public RepetitionInstanceInfo plusExdates(ListF<Instant> starts) {
        return withExdates(getExdates().plus(starts.map(RepetitionUtils::consExdate)));
    }

    public RepetitionInstanceInfo withoutRecurrence(Instant recurrenceId) {
        return withRecurrences(recurrences.filter(r -> !r.getRecurrenceId().equals(recurrenceId)));
    }

    public RepetitionInstanceInfo withInstancesInsteadOfTimeUnchangedRecurrences() {
        return withRecurrences(recurrences.filterNot(r ->
                r.getRecurrenceId().isEqual(r.getInterval().getStart()) && getEventPeriod().equals(r.getPeriod(tz))));
    }

    public RepetitionInstanceInfo withoutPastRecurrencesAndExdates(Instant now) {
        ListF<Rdate> exdates = getExdates().filterNot(e -> e.getStartTs().isBefore(now));
        ListF<RecurrenceTimeInfo> recurrences = getRecurrences().filterNot(e -> e.getInterval().getEnd().isBefore(now));

        return new RepetitionInstanceInfo(eventInterval, tz, repetition, rdates, exdates, recurrences);
    }

    public RepetitionInstanceInfo withoutRecurrences() {
        return withRecurrences(Cf.list());
    }

    public RepetitionInstanceInfo withoutRdatesAndRecurrences() {
        return new RepetitionInstanceInfo(eventInterval, tz, repetition, Cf.list(), exdates, Cf.list());
    }

    public RepetitionInstanceInfo withoutExdatesAndRdatesAndRecurrences() {
        return new RepetitionInstanceInfo(eventInterval, tz, repetition, Cf.list(), Cf.list(), Cf.list());
    }

    public RepetitionInstanceInfo withRecurrences(ListF<RecurrenceTimeInfo> recurrences) {
        return new RepetitionInstanceInfo(eventInterval, tz, repetition, rdates, exdates, recurrences);
    }

    public RepetitionInstanceInfo withoutAnyRepeating() {
        return noRepetition(eventInterval, tz);
    }

    public RepetitionInstanceInfo withDueTs(Option<Instant> dueTs) {
        return new RepetitionInstanceInfo(eventInterval, tz,
                repetition.map(r -> {
                    Repetition copy = r.copy();
                    copy.setDueTs(dueTs);
                    return copy;
                }),
                rdates.filterNot(r -> dueTs.exists(due -> !r.getStartTs().isBefore(due))),
                exdates.filterNot(r -> dueTs.exists(due -> !r.getStartTs().isBefore(due))),
                recurrences.filterNot(r -> dueTs.exists(due -> !r.getRecurrenceId().isBefore(due))));
    }

    public RepetitionInstanceInfo changeTz(final DateTimeZone tz) {
        final Function<Instant, Instant> changeTzF = new Function<Instant, Instant>() {
            public Instant apply(Instant instant) {
                return AuxDateTime.toDateTimeIgnoreGap(new LocalDateTime(instant, getTz()), tz).toInstant();
            }
        };

        final Function<Rdate, Rdate> changeRdateTzF = new Function<Rdate, Rdate>() {
            public Rdate apply(Rdate rdate) {
                Rdate copy = rdate.copy();
                copy.setStartTs(changeTzF.apply(rdate.getStartTs()));
                copy.setEndTs(rdate.getEndTs().map(changeTzF));
                return copy;
            }
        };

        final Function<Repetition, Repetition> changeRepetitionTzF = new Function<Repetition, Repetition>() {
            public Repetition apply(Repetition repetition) {
                Repetition copy = repetition.copy();
                copy.setDueTs(repetition.getDueTs().map(changeTzF));
                return copy;
            }
        };

        ListF<RecurrenceTimeInfo> newRecurrences = recurrences.map(r -> r.mapTz(changeTzF));

        return new RepetitionInstanceInfo(
                new InstantInterval(changeTzF.apply(getEventStart()), changeTzF.apply(getEventInterval().getEnd())),
                tz, repetition.map(changeRepetitionTzF),
                rdates.map(changeRdateTzF), exdates.map(changeRdateTzF), newRecurrences);
    }

    public RepetitionInstanceInfo withInterval(InstantInterval interval) {
        return new RepetitionInstanceInfo(interval, tz, repetition, rdates, exdates, recurrences);
    }

    public static Function<RepetitionInstanceInfo, Option<Repetition>> getRepetitionF() {
        return new Function<RepetitionInstanceInfo, Option<Repetition>>() {
            public Option<Repetition> apply(RepetitionInstanceInfo repetitionInstanceInfo) {
                return repetitionInstanceInfo.getRepetition();
            }
        };
    }

    public ListF<RecurrenceTimeInfo> getRecurrences() {
        return recurrences;
    }

    public Option<Instant> findRecurrenceIdByStart(Instant start) {
        return getRecurrences()
                .find(r -> r.getInterval().getStart().equals(start))
                .map(RecurrenceTimeInfo::getRecurrenceId);
    }

    public boolean containsInterval(InstantInterval interval) {
        return RepetitionUtils.getClosestInstanceInterval(
                this.withoutRecurrences(), interval.getStart()).getOrElse(eventInterval).equals(interval);
    }

    public Repetition getRepetitionOrNone() {
        return repetition.getOrElse(RepetitionRoutines::createNoneRepetition);
    }

    public boolean hasActualOccurrenceIn(Instant start, Instant end) {
        if (end.isAfter(start)) {
            if (RepetitionUtils.getIntervals(this, start, Option.of(end), true, 1).isNotEmpty()) {
                return true;
            }
            if (getRecurrences().exists(r -> r.getRecurrenceId().isBefore(end) && r.getInterval().getEnd().isAfter(start))) {
                return true;
            }
        }
        return false;
    }

    public Option<Instant> getRdatesMinTs() {
        return rdates.iterator()
                .filter(Rdate::getIsRdate)
                .map(Rdate::getStartTs)
                .minO();
    }

    public Option<Instant> getRdatesMaxTs() {
        Interval interval = getEventIntervalWithTz();

        return rdates.iterator()
                .filter(Rdate::getIsRdate)
                .map(rd -> RepetitionUtils.getRdateEndTs(rd, interval))
                .maxO();
    }

    public Option<RepetitionHints> createHints() {
        return repetition.map(r -> RepetitionHints.createByRepetition(getEventIntervalWithTz().getStart(), r));
    }
}
