package ru.yandex.calendar.logic.event.repetition;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.Months;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.suggest.IntervalSet;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.time.InstantInterval;

public class RepetitionInstanceSet {
    private static final Months MAX_CHECK_PERIOD = Months.months(12);
    private static final Months MAX_CHECK_PERIOD_FOR_USERS = Months.months(6);

    private final IntervalSet instances;

    public RepetitionInstanceSet(RepetitionInstanceInfo repetition, Instant lowerBound, Instant upperBound) {
        this(IntervalSet.fromSuccessiveIntervals(mergeSortedIntervals(
                RepetitionUtils.getIntervals(repetition, lowerBound, Option.of(upperBound), true, 0))));
    }

    private RepetitionInstanceSet(IntervalSet instances) {
        this.instances = instances;
    }

    public static RepetitionInstanceSet boundedByMaxCheckPeriod(RepetitionInstanceInfo repetition, Instant lowerBound) {
        return new RepetitionInstanceSet(repetition, lowerBound, boundedByMaxCheckPeriod(Option.<Instant>empty()));
    }

    public static RepetitionInstanceSet boundedForUsersAvailability(RepetitionInstanceInfo repetition, Instant lowerBound) {
        return new RepetitionInstanceSet(repetition, lowerBound, boundedForUsersAvailabilities(Option.empty()));
    }

    public int countInstancesEndingBefore(Instant before) {
        return instances.countEndingBefore(before);
    }

    public boolean isEmpty() {
        return instances.isEmpty();
    }

    public Instant getFirstStart() {
        return instances.getStart();
    }

    public Instant getLastEnd() {
        return instances.getEnd();
    }

    public boolean overlaps(InstantInterval interval) {
        return instances.overlaps(interval);
    }

    public boolean intersects(RepetitionInstanceSet other) {
        return findFirstOverlap(other).isPresent();
    }

    public Option<IntersectingIntervals> findFirstOverlap(RepetitionInstanceSet other) {
        return overlap(other, true).singleO();
    }

    public ListF<IntersectingIntervals> overlap(RepetitionInstanceInfo other) {
        return isEmpty() ? Cf.list() : overlap(new RepetitionInstanceSet(other, getFirstStart(), getLastEnd()));
    }

    public ListF<IntersectingIntervals> overlap(RepetitionInstanceSet other) {
        return overlap(other, false);
    }

    private ListF<IntersectingIntervals> overlap(RepetitionInstanceSet other, boolean stopOnFirst) {
        if (other.instances.isEmpty() || instances.isEmpty()) return Cf.list();

        ListF<IntersectingIntervals> result = Cf.arrayList();
        ListF<InstantInterval> otherInstances = other.instances.getIntervals();

        int i = 0;
        for (InstantInterval instance : instances.getIntervals().
                subList(instances.countEndingBefore(other.instances.getStart()), instances.getIntervals().size()))
        {
            while (i < otherInstances.size()
                    && !otherInstances.get(i).getEnd().isAfter(instance.getStart())
                    && !otherInstances.get(i).getStart().isEqual(instance.getStart()))
            {
                ++i;
            }

            if (i >= otherInstances.size()) break;

            int j = i;
            while (j < otherInstances.size() && RepetitionUtils.overlapsOrSameStart(otherInstances.get(j), instance)) {
                result.add(new IntersectingIntervals(instance, otherInstances.get(j++)));

                if (stopOnFirst) return result;
            }
        }
        return result;
    }

    public static Instant boundedByMaxCheckPeriod(Option<Instant> ms) {
        return minBoundedByMaxCheckPeriod(ms, Option.<Instant>empty());
    }

    public static Instant boundedForUsersAvailabilities(Option<Instant> ms) {
        return minBoundedByPeriod(ms, Option.empty(), MAX_CHECK_PERIOD_FOR_USERS);
    }

    public static Instant minBoundedByMaxCheckPeriod(Option<Instant> ms1, Option<Instant> ms2) {
        return minBoundedByPeriod(ms1, ms2, MAX_CHECK_PERIOD);
    }

    private static Instant minBoundedByPeriod(Option<Instant> ms1, Option<Instant> ms2, Months period) {
        Instant max = Instant.now().toDateTime(DateTimeZone.UTC).plus(period).toInstant();
        return Option.of(max).plus(ms1).plus(ms2).min();
    }

    public static RepetitionInstanceSet single(InstantInterval interval) {
        return fromSuccessiveIntervals(Cf.list(interval));
    }

    public static RepetitionInstanceSet fromSuccessiveIntervals(ListF<InstantInterval> intervals) {
        return new RepetitionInstanceSet(IntervalSet.fromSuccessiveIntervals(intervals));
    }

    public RepetitionInstanceSet plusIntervals(ListF<InstantInterval> intervals) {
        return new RepetitionInstanceSet(IntervalSet.cons(instances.getIntervals().plus(intervals)));
    }

    private static ListF<InstantInterval> mergeSortedIntervals(ListF<InstantInterval> intervals) {
        return Cf2.merge(intervals, (one, two) -> one.getEnd().isAfter(two.getStart())
                ? Cf.list(one.withEnd(ObjectUtils.max(one.getEnd(), two.getEnd())))
                : Cf.list(one, two));
    }

    ListF<InstantInterval> getInstances() {
        return instances.getIntervals();
    }
}
