package ru.yandex.calendar.logic.event.repetition;

import lombok.val;
import org.jdom.Element;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.Interval;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.RdateFields;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.RepetitionFields;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.util.base.AuxColl;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.base.MultiMap;
import ru.yandex.calendar.util.dates.WeekdayConv;
import ru.yandex.calendar.util.db.CalendarJdbcTemplate;
import ru.yandex.calendar.util.db.DbUtils;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.misc.lang.CamelWords;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author ssytnik
 */
public class RepetitionRoutines {

    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private EventDao eventDao;
    /** @deprecated */
    @Autowired
    @Deprecated
    private CalendarJdbcTemplate jdbcTemplate;

    /** @deprecated */
    protected CalendarJdbcTemplate getJdbcTemplate() {
        return jdbcTemplate;
    }

    public static Repetition createNoneRepetition() {
        Repetition repetition = new Repetition();
        repetition.setType(RegularRepetitionRule.NONE);
        return repetition;
    }

    public static boolean isNoneRepetition(Repetition repetition) {
        return repetition.getType() == RegularRepetitionRule.NONE;
    }

    /**
     * By given repetition data provider, creates or updates a repetition rule
     * @param idO current that indicates repetition
     * @param newRepetition repetition data
     * @return new repetition id
     * @throws CommandRunException if non-regular rule encountered, unless it is allowed,
     * or other underlying error occurs
     */
    public long createOrUpdateRepetition(Option<Long> idO, Repetition newRepetition) {
        newRepetition = clearUnusedFields(newRepetition);

        if (idO.isPresent()) {
            newRepetition.setId(idO.get());
            eventDao.updateRepetition(newRepetition);
            return idO.get();
        } else {
            return eventDao.saveRepetition(newRepetition);
        }
    }

    public long createRepetition(Repetition repetitionData) {
        Validate.isTrue(!repetitionData.isFieldSet(RepetitionFields.ID));
        return eventDao.saveRepetition(clearUnusedFields(repetitionData));
    }

    private Repetition clearUnusedFields(Repetition repetition) {
        Repetition r = repetition.copy();
        if (repetition.isFieldSet(RepetitionFields.TYPE)) {
            if (repetition.getType() != RegularRepetitionRule.MONTHLY_DAY_WEEKNO) r.setRMonthlyLastweekNull();
            if (repetition.getType() != RegularRepetitionRule.WEEKLY) r.setRWeeklyDaysNull();
        }
        return r;
    }

    /////////////////////////
    // REGULAR REPETITIONS //
    /////////////////////////

    public Option<Repetition> getRepetition(Event e) {
        Option<Long> repetitionId = e.getRepetitionId();
        if (repetitionId.isPresent()) {
            return Option.of(eventDao.findRepetitionById(repetitionId.get()));
        } else {
            return Option.empty();
        }
    }

    public Repetition getRepetitionById(long repetitionId) {
        return eventDao.findRepetitionById(repetitionId);
    }

    // Returns single interval (if no instance found, returns null)
    public Option<InstantInterval> getSingleInterval(
            Event e, RepetitionInstanceInfo rii, Instant iStartMs,
            boolean overlap)
    {
        ListF<InstantInterval> intervals = RepetitionUtils.getIntervals(
            rii, iStartMs, Option.<Instant>empty(),
            overlap,
            1
        );
        return intervals.singleO();
    }


//    private MultiMap<String, Instant> getEventRecurIds(ListF<Event> eList) {
//        MultiMap<String, Instant> res = AuxColl.newMMap();
//        if (!eList.isEmpty()) {
//            ListF<Object> extIds = Cf.arrayList();
//            for (Event e : eList) {
//                extIds.add(EventRoutines.getPublicExtId(e));
//           }
//           SqlCondition condition = EventFields.RECURRENCE_ID.column().isNotNull();
//           condition = condition.and(EventFields.EXTERNAL_ID.column().inSet(extIds));
//           //condition = condition.and(EventFields.CREATOR_UID.column());
//           String sql =
//                   "SELECT recurrence_id, external_id FROM event WHERE " +
//                   condition.sql();
//           // Recurrence instances always should have external_id, so
//           // we may not use here common ER.createSqlConditionByExtId(...)
//           ListF<Map<String, Object>> rows = DbUtils.getRows(getJdbcTemplate(), sql, condition.args());
//           for (Map<String, Object> row : rows) {
//               String extId = (String) row.get(EventFields.EXTERNAL_ID.column().name());
//               Instant recurId = AuxDateTime.toInstant(row.get(EventFields.RECURRENCE_ID.column().name()));
//               res.append(extId, recurId);
//           }
//        }
//        return res;
//    }

    public RepetitionInstanceInfo getRepetitionInstanceInfo(EventWithRelations event) {
        return getRepetitionInstanceInfos(Cf.list(event)).getOrThrow(event.getId());
    }

    public RepetitionInstanceInfo getRepetitionInstanceInfoByEvent(Event event) {
        Validate.notNull(event);
        return getRepetitionInstanceInfosByEvents(Cf.list(event)).getTs(event.getId());
    }

    public RepetitionInstanceInfo getRepetitionInstanceInfoByEventId(long eventId) {
        return getRepetitionInstanceInfoByEvent(eventDbManager.getEventById(eventId));
    }

    public MapF<Long, RepetitionInstanceInfo> getRepetitionInstanceInfosByEventIds(ListF<Long> eventIds) {
        return getRepetitionInstanceInfosByEvents(eventDbManager.getEventsByIds(eventIds));
    }

    public boolean hasInstancesAfter(long eventId, Instant instant) {
        RepetitionInstanceInfo repetitionInstanceInfo = getRepetitionInstanceInfoByEventId(eventId);
        return hasInstancesAfter(repetitionInstanceInfo, instant);
    }

    public static boolean hasInstancesAfter(RepetitionInstanceInfo repetitionInstanceInfo, Instant instant) {
        InfiniteInterval interval = RepetitionUtils.calcCommonInstsInterval(repetitionInstanceInfo);
        return interval.getStart().isAfter(instant) || interval.contains(instant);
    }

    public MapF<Long, RepetitionInstanceInfo> getRepetitionInstanceInfos(ListF<EventWithRelations> events) {
        Tuple2List<Event, DateTimeZone> eventsAndTimezones = events.toTuple2List(
                EventWithRelations::getEvent, EventWithRelations::getTimezone);
        return getRepetitionInstanceInfos(eventsAndTimezones, true);
    }

    public MapF<Long, RepetitionInstanceInfo> getRepetitionInstanceInfosByEventsAndTimezones(
            Tuple2List<Event, DateTimeZone> events)
    {
        return getRepetitionInstanceInfos(events, true);
    }

    public MapF<Long, RepetitionInstanceInfo> getRepetitionInstanceInfosByEvents(ListF<Event> events) {
        MapF<Long, DateTimeZone> timezoneById = eventRoutines.getEventsTimeZones(events).toMap();
        return getRepetitionInstanceInfos(events.zipWith(e -> timezoneById.getOrThrow(e.getId())), true);
    }

    public RepetitionInstanceInfo getRepetitionInstanceInfoByEventAndTimezone(Event event, DateTimeZone tz) {
        return getRepetitionInstanceInfos(Tuple2List.fromPairs(event, tz), true).values().single();
    }

    public MapF<Long, RepetitionInstanceInfo> getRepetitionInstanceInfosByMainEventId(long mainEventId) {
        return getRepetitionInstanceInfosAmongEvents(eventDao.findEventsByMainId(mainEventId));
    }

    public MapF<Long, RepetitionInstanceInfo> getRepetitionInstanceInfosAmongEvents(ListF<Event> events) {
        MapF<Long, DateTimeZone> timezoneById = eventRoutines.getEventsTimeZones(events).toMap();
        // XXX: upyachka for performance,
        // use this method only if events list contains all possible
        // recurrence events, otherwise use getRepetitionInstanceInfos(...)
        return getRepetitionInstanceInfos(events.zipWith(e -> timezoneById.getOrThrow(e.getId())), false);
    }

    private MapF<Long, RepetitionInstanceInfo> getRepetitionInstanceInfos(
            Tuple2List<Event, DateTimeZone> eventsAndTimezones, boolean doQueryForRecurrence)
    {
        MapF<Long, RepetitionInstanceInfo> res = Cf.hashMap();

        Tuple2<Tuple2List<Event, DateTimeZone>, Tuple2List<Event, DateTimeZone>> partitioned =
                eventsAndTimezones.partitionBy1(EventFields.RECURRENCE_ID.getF().andThen(Function1B.notNullF()));

        for (Tuple2<Event, DateTimeZone> eTz : partitioned.get1()) {
            res.put(eTz.get1().getId(), RepetitionInstanceInfo.noRepetition(
                    EventRoutines.getInstantInterval(eTz.get1()), eTz.get2()));
        }

        ListF<Event> recurrenceEventInstances = partitioned.get1().map(Tuple2::get1);
        ListF<Event> nonRecurrenceEventInstances = partitioned.get2().map(Tuple2::get1);

        if (nonRecurrenceEventInstances.isNotEmpty()) {
            // Exdates & Rdates
            ListF<Long> eventIds = nonRecurrenceEventInstances.map(EventFields.ID.getF());
            ListF<Rdate> rdateList = eventDao.findRdatesByEventIds(eventIds);
            Tuple2<ListF<Rdate>, ListF<Rdate>> rdatesExdates = rdateList.partition(Rdate::getIsRdate);
            MapF<Long, ListF<Rdate>> rdatesByEventId = rdatesExdates._1.groupBy(RdateFields.EVENT_ID.getF());
            MapF<Long, ListF<Rdate>> exdatesByEventId = rdatesExdates._2.groupBy(RdateFields.EVENT_ID.getF());
            // Repetitions
            ListF<Long> repetitionIds = nonRecurrenceEventInstances
                .map(EventFields.REPETITION_ID.getF())
                .filterNotNull()
                ;
            ListF<Repetition> repetitionList = eventDao.findRepetitionsByIds(repetitionIds);
            MapF<Long, Repetition> repetitionMap = repetitionList.toMapMappingToKey(RepetitionFields.ID.getF());
            // Recurrence ids
            MultiMap<Long, RecurrenceTimeInfo> recurrencesMMap = doQueryForRecurrence ?
                    getEventsRecurrenceIdsFromDb(nonRecurrenceEventInstances, rdatesByEventId) :
                    getEventsRecurrenceIdsDirectly(recurrenceEventInstances);

            MapF<Long, DateTimeZone> timezoneByEventId = eventsAndTimezones.map1(Event.getIdF()).toMap();

            for (Event e : nonRecurrenceEventInstances) {
                ListF<Rdate> rdates = rdatesByEventId.getTs(e.getId());
                ListF<Rdate> exdates = exdatesByEventId.getTs(e.getId());
                ListF<RecurrenceTimeInfo> recurrences = recurrencesMMap.get(e.getMainEventId());
                RepetitionInstanceInfo rInstInfo = new RepetitionInstanceInfo(
                        EventRoutines.getInstantInterval(e),
                        timezoneByEventId.getTs(e.getId()),
                        repetitionMap.getO(e.getRepetitionId().getOrNull()),
                        rdates != null ? rdates : Cf.<Rdate>list(),
                        exdates != null ? exdates : Cf.<Rdate>list(),
                        recurrences != null ? recurrences : Cf.list()
                );
                res.put(e.getId(), rInstInfo);
            }
        }
        return res;
    }

    private MultiMap<Long, RecurrenceTimeInfo> getEventsRecurrenceIdsDirectly(ListF<Event> recurrenceEventInstances) {
        MultiMap<Long, RecurrenceTimeInfo> recurrenceIdsMap = AuxColl.newMMap();
        for (Event event : recurrenceEventInstances) {
            recurrenceIdsMap.append(event.getMainEventId(), RecurrenceTimeInfo.fromEvent(event));
        }
        return recurrenceIdsMap;
    }

    private MultiMap<Long, RecurrenceTimeInfo> getEventsRecurrenceIdsFromDb(
            ListF<Event> nonRecurenceEventInstances, final MapF<Long, ListF<Rdate>> rdatesByEventId)
    {
        MultiMap<Long, RecurrenceTimeInfo> recurrenceIdsMap = AuxColl.newMMap();
        ListF<Event> repeatingEvents = nonRecurenceEventInstances.partition(new Function1B<Event>() {
            public boolean apply(Event event) {
                if (event.getRepetitionId().isPresent()) {
                    return true;
                } else {
                    return rdatesByEventId.getTs(event.getId()) != null;
                }
            }
        }).get1();
        if (repeatingEvents.isNotEmpty()) {
            eventDao.findRecurrenceInstantInfosByMainEventIds(repeatingEvents.map(Event::getMainEventId), Option.empty())
                    .forEach(r -> recurrenceIdsMap.append(r.get1(), r.get2()));
        }
        return recurrenceIdsMap;
    }

    public Instant calcEndTs(Event event, Instant eInstStartMs) {
        Interval i = eventRoutines.getIntervalByEvent(event);
        DateTime startDt = new DateTime(eInstStartMs, i.getChronology());
        return startDt.plus(i.toPeriod()).toInstant();
    }

    /**
     * This routine appends eParent xml element with additional tags which represent
     * repetition information. The rules are as follows:
     * 1. Repetition hints for date are calculated.
     * 2. Tag is added indicating whether repetition is currently ON.
     * 3. Repetition tag itself is added.
     * Tag is named 'repetition' and consists of the following contents:
     * default repetition set for the day (specified by the 'dateMs' and 'dtf' parameters)
     * that can be overridden if 'r' is set. 'r' overrides common fields such as
     * 'id', 'type' and 'r_each'. Depending on the 'type', subsequent fields such as
     * 'r_weekly_days' or 'r_monthly_last_week' can be overridden.
     * This guarantees that user will see current repetition settings correctly,
     * and will see correct default repetition values for other rule types for chosen day.
     * @param eParent parent xml element which new data to be put
     * @param dateMs milliseconds for date (date time should not necessarily
     * be cut) for calculating default repetition rule
     * @param r event repetition (can be null, cannot be non-regular)
     * @throws CommandRunException if r has non-regular rule or if underlying error occurs
     */
    public static void addElms(Element eParent, long dateMs, Repetition r, DateTimeZone tz)
    {
        boolean rSet = r != null;

        // Always calculate rep-hints
        // OLD: DateMidnight dm = new DateMidnight(dateMs, dtf.getChrono());
        // OLD: DateTime dt = dm.toDateTime(dm.getChronology());
        DateTime dt = new DateTime(dateMs, tz);
        // For daily             : nothing
        // For weekly            : weekday
        // For monthly_number    : day
        // For monthly_day_weekno: weekday, weekday-no, is-last-weekday
        // For yearly            : day, month
        int day = dt.getDayOfMonth();
        int month = dt.getMonthOfYear();
        int weekDayNo = RepetitionUtils.getNumWeekDaysForDate(dt);
        String calWeekDay = WeekdayConv.jodaToCals(dt.getDayOfWeek());
        boolean isLastWeekday = RepetitionUtils.isLastWeekDayInMonth(dt) || weekDayNo > 4;

        Element eHints = new Element("rep-hints");
        CalendarXmlizer.appendElm(eHints, "day", day);
        CalendarXmlizer.appendElm(eHints, "month", month);
        CalendarXmlizer.appendElm(eHints, "weekday", calWeekDay);
        CalendarXmlizer.appendElm(eHints, "weekday-no", weekDayNo);
        CalendarXmlizer.appendElm(eHints, "is-last-weekday", isLastWeekday);
        eParent.addContent(eHints);

        // rRes := default repetition based on initialized hints variables
        MapF<String, Object> rMap = Cf.hashMap();
        rMap.put("id", -1L); // non-existing id, in order to put sth valid to xml element
        rMap.put("type", RegularRepetitionRule.DAILY); // does not matter
        rMap.put("r_each", 1);
        rMap.put("r_weekly_days", calWeekDay);
        rMap.put("r_monthly_lastweek", isLastWeekday);
        for (MapField<?> field : RepetitionFields.OBJECT_DESCRIPTION.getFields()) {
            String columnName = CamelWords.parse(field.getName()).toDbName();
            if (!rMap.containsKeyTs(columnName)) {
                // XXX: remove
                rMap.put(columnName, null);
            } // if
        } // for
        Repetition rRes = new Repetition();
        rRes.setFields(rMap.entries().map1(RepetitionFields.OBJECT_DESCRIPTION.getFieldByNameF()));
        if (rSet) { // change default repetition data - only for current r.type!
            rRes.setId(r.getId());
            rRes.setType(r.getType());
            rRes.setDueTs(r.getDueTs().getOrNull());
            rRes.setREach(RepetitionUtils.calcREach(r));
            RepetitionUtils.copyFields(RegularRepetitionRule.find(r), r, rRes);
        }
        CalendarXmlizer.appendElm(eParent, "is-rep-on", rSet);
        rRes.addXmlElementTo(eParent, "repetition", tz);
    }

    public void createExdates(final long eventId, ListF<Instant> exdates, ActionInfo actionInfo) {
        createRdates(exdates.map(Cf2.f2(RepetitionUtils::consExdateEventId).bind1(eventId)), actionInfo);
    }

    public void createRdates(ListF<Rdate> newRdates, ActionInfo actionInfo) {
        for (Rdate rdate : newRdates) {
            rdate.setCreationReqId(actionInfo.getRequestIdWithHostId());
            rdate.setCreationTs(actionInfo.getNow());
            eventDao.saveRdate(rdate);
        }
    }

    /**
     * Returns rdates/exdates changes
     * @param rdList new rdates, exdates data
     * @param rii old info about event recurrences
     */
    public RdateChangesInfo rdatesChanges(ListF<Rdate> rdList, RepetitionInstanceInfo rii) {
        ListF<Rdate> newRdates = Cf.arrayList();
        ListF<Rdate> rdates = Cf.toArrayList(rii.getRdates());
        rdates.addAll(rii.getExdates());
        final SetF<Long> ids = Cf.toHashSet(rdates.map(RdateFields.ID.getF()));
        for (final Rdate newRdate : rdList) {
            boolean exist = rdates.exists(new Function1B<Rdate>() {
                @Override
                public boolean apply(Rdate rdate) {
                    // optional field
                    Instant endTs = newRdate.getFieldValueO(RdateFields.END_TS).getOrNull();
                    boolean equals =
                        ObjectUtils.equals(rdate.getStartTs(), newRdate.getStartTs()) &&
                        ObjectUtils.equals(rdate.getEndTs().getOrNull(), endTs) &&
                        ObjectUtils.equals(rdate.getIsRdate(), newRdate.getIsRdate());
                    if (equals) {
                        ids.removeTs(rdate.getId());
                    }
                    return equals;
                }
            });
            if (!exist) {
                newRdates.add(newRdate);
            }
        }
        return new RdateChangesInfo(newRdates, ids);
    }

    public void updateRdates(long eId, RdateChangesInfo rdateChangeInfo, ActionInfo actionInfo) {
        val remRdateIds = Cf.toSet(rdateChangeInfo.getRemRdateIds());
        if (AuxColl.isSet(remRdateIds)) {
            String sql = "DELETE FROM rdate WHERE " + DbUtils.getIdInClause(remRdateIds);
            getJdbcTemplate().update(sql);
        }

        val newRdates = rdateChangeInfo.getNewRdates();
        for (Rdate newRdate : newRdates) {
            newRdate.setEventId(eId);
        }

        createRdates(Cf.toList(newRdates), actionInfo);
    }
} //~
