package ru.yandex.calendar.logic.event.repetition;

import java.util.Optional;

import lombok.val;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.util.dates.DayOfWeek;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.format.MessageFormatter;
import ru.yandex.misc.log.mlf.format.Slf4jLikeMessageFormatter;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author Daniel Brylev
 */
public class RepetitionToStringConverter {

    public static String convert(InstantInterval interval, Repetition repetition, DateTimeZone tz, Language lang) {
        return convert(new RepetitionInstanceInfo(
                interval, tz, Option.of(repetition), Cf.list(), Cf.list(), Cf.list()), lang);
    }

    public static String convert(RepetitionInstanceInfo info, Language lang) {
        return convert(info, false, lang);
    }

    @Deprecated
    public static String convert(RepetitionInstanceInfo info, boolean sinceInsteadDue, Language lang) {
        if (info.isEmpty() || info.getRepetition().isEmpty()) {
            return "";
        }

        switch (info.getRepetition().get().getType()) {
            case DAILY: return convertDaily(info, sinceInsteadDue, lang);
            case WEEKLY: return convertWeekly(info, sinceInsteadDue, lang);
            case MONTHLY_NUMBER: return convertMonthlyNumber(info, sinceInsteadDue, lang);
            case MONTHLY_DAY_WEEKNO: return convertMonthlyDayWeekNo(info, sinceInsteadDue, lang);
            case YEARLY: return convertYearly(info, sinceInsteadDue, lang);
        }
        return "";
    }

    public static Optional<String> tryConvert(RepetitionInstanceInfo info, boolean sinceInsteadDue, Language lang) {
        val value = convert(info, sinceInsteadDue, lang);
        return StringUtils.notEmptyO(value).toOptional();
    }

    private static String convertDaily(RepetitionInstanceInfo info, boolean sinceInsteadDue, Language lang) {
        Repetition repetition = info.getRepetition().get();
        Validate.isTrue(repetition.getType() == RegularRepetitionRule.DAILY);

        StringBuilder result = new StringBuilder();
        int each = RepetitionUtils.calcREach(repetition);

        if (lang == Language.RUSSIAN) {
            result.append(formatREachRu(each, "каждый {}-й день", "ежедневно"));
        } else {
            result.append(formatREachEn(each, "every {} day", "every day"));
        }

        if (repetition.getDueTs().isPresent()) {
            LocalDate dueDate = EventRoutines.convertDueTsToUntilDate(repetition.getDueTs().get(), info.getTz());
            LocalDate nowDate = LocalDate.now(info.getTz());

            result.append(" ").append(convertDueDate(
                    lang, dueDate, sinceInsteadDue, true, true, dueDate.getYear() != nowDate.getYear()));
        }
        return result.toString();
    }

    private static final SetF<DayOfWeek> weekdaySet = Cf.set(
            DayOfWeek.MONDAY, DayOfWeek.TUESDAY, DayOfWeek.WEDNESDAY, DayOfWeek.THURSDAY, DayOfWeek.FRIDAY);
    private static final SetF<DayOfWeek> weekendSet = Cf.set(
            DayOfWeek.SATURDAY, DayOfWeek.SUNDAY);
    private static final SetF<DayOfWeek> everydaySet = Cf.set(DayOfWeek.values());


    private static String convertWeekly(RepetitionInstanceInfo info, boolean sinceInsteadDue, final Language lang) {
        Repetition repetition = info.getRepetition().get();
        Validate.isTrue(repetition.getType() == RegularRepetitionRule.WEEKLY);

        if (StringUtils.isEmpty(repetition.getRWeeklyDays().get())) {
            return "";
        }

        final String days;

        ListF<DayOfWeek> daysOfWeek = Cf.x(repetition.getRWeeklyDays().get().split(",")).map(DayOfWeek.byNameF());
        SetF<DayOfWeek> daysOfWeekSet = Cf.toSet(daysOfWeek);

        if (daysOfWeekSet.equals(weekdaySet)) {
            days = choose(lang, "с понедельника по пятницу", "Monday through Friday");

        } else if (daysOfWeekSet.equals(weekendSet)) {
            days = choose(lang, "по субботам и воскресеньям", "on Saturdays and Sundays");

        } else if (daysOfWeekSet.equals(everydaySet)) {
            days = choose(lang, "каждый день", "every day");

        } else if (daysOfWeek.size() > 2) {
            ListF<String> ds = daysOfWeek.map(new Function<DayOfWeek, String>() {
                public String apply(DayOfWeek d) {
                    switch (d) {
                        case MONDAY: return choose(lang, "пн", "Mon");
                        case TUESDAY: return choose(lang, "вт", "Tue");
                        case WEDNESDAY: return choose(lang, "ср", "Wed");
                        case THURSDAY: return choose(lang, "чт", "Thu");
                        case FRIDAY: return choose(lang, "пт", "Fri");
                        case SATURDAY: return choose(lang, "сб", "Sat");
                        default: return choose(lang, "вс", "Sun");
                    }
                }
            });
            days = choose(lang, "по " + ds.rdrop(1).mkString(", ") + " и " + ds.last(), "on " + ds.mkString(", "));
        } else {
            ListF<String> ds = daysOfWeek.map(new Function<DayOfWeek, String>() {
                public String apply(DayOfWeek d) {
                    switch (d) {
                        case MONDAY: return choose(lang, "понедельникам", "Mondays");
                        case TUESDAY: return choose(lang, "вторникам", "Tuesdays");
                        case WEDNESDAY: return choose(lang, "средам", "Wednesdays");
                        case THURSDAY: return choose(lang, "четвергам", "Thursdays");
                        case FRIDAY: return choose(lang, "пятницам", "Fridays");
                        case SATURDAY: return choose(lang, "субботам", "Saturdays");
                        default: return choose(lang, "воскресеньям", "Sundays");
                    }
                }
            });
            days = choose(lang, "по ", "on ") + ds.mkString(", ");
        }

        StringBuilder result = new StringBuilder();

        int each = RepetitionUtils.calcREach(repetition);
        if (lang == Language.RUSSIAN) {
            result.append(days).append(formatREachRu(each, " каждой {}-й недели", ""));
        } else {
            result.append(formatREachEn(each, "every {} week ", "")).append(days);
        }
        if (repetition.getDueTs().isPresent()) {
            LocalDate dueDate = EventRoutines.convertDueTsToUntilDate(repetition.getDueTs().get(), info.getTz());
            LocalDate nowDate = LocalDate.now(info.getTz());

            result.append(" ").append(convertDueDate(
                    lang, dueDate, sinceInsteadDue, true, true, dueDate.getYear() != nowDate.getYear()));
        }
        return result.toString();
    }

    private static String convertMonthlyNumber(RepetitionInstanceInfo info, boolean sinceInsteadDue, Language lang) {
        Repetition repetition = info.getRepetition().get();
        Validate.isTrue(repetition.getType() == RegularRepetitionRule.MONTHLY_NUMBER);

        StringBuilder result = new StringBuilder();
        DateTime start = info.getEventIntervalWithTz().getStart();

        int each = RepetitionUtils.calcREach(repetition), day = start.getDayOfMonth();

        if (lang == Language.RUSSIAN) {
            result.append(day).append(" числа каждого ").append(formatREachRu(each, "{}-го месяца", "месяца"));
        } else {
            result.append(formatREachEn(each, "every {} month", "monthly")).append(" on day ").append(day);
        }

        if (repetition.getDueTs().isPresent()) {
            LocalDate dueDate = EventRoutines.convertDueTsToUntilDate(repetition.getDueTs().get(), info.getTz());
            LocalDate nowDate = LocalDate.now(info.getTz());

            result.append(" ").append(convertDueDate(
                    lang, dueDate, sinceInsteadDue, false, true, dueDate.getYear() != nowDate.getYear()));
        }
        return result.toString();
    }

    private static String convertMonthlyDayWeekNo(RepetitionInstanceInfo info, boolean sinceInsteadDue, Language lang) {
        Repetition repetition = info.getRepetition().get();
        Validate.isTrue(repetition.getType() == RegularRepetitionRule.MONTHLY_DAY_WEEKNO);

        DateTime start = info.getEventIntervalWithTz().getStart();

        int weekNum = RepetitionUtils.getNumWeekDaysForDate(start);
        boolean lastWeekOfMonth = repetition.getRMonthlyLastweek().get() || weekNum > 4;

        StringBuilder dayWeekNo = new StringBuilder();

        if (lang == Language.RUSSIAN) {
            dayWeekNo.append(weekNum == 2 && !lastWeekOfMonth ? "во " : "в ");
            dayWeekNo.append((Cf.set(1, 2, 4).containsTs(start.getDayOfWeek())
                    ? (!lastWeekOfMonth ? weekNum + "-й" : "последний")
                    : Cf.set(3, 5, 6).containsTs(start.getDayOfWeek())
                            ? (!lastWeekOfMonth ? weekNum + "-ю" : "последнюю")
                            : (!lastWeekOfMonth ? weekNum + "-е" : "последнее")));

        } else {
            dayWeekNo.append("on ").append(!lastWeekOfMonth ? suffixedNumberEn(weekNum) : "last");
        }
        dayWeekNo.append(" ");

        switch (start.getDayOfWeek()) {
            case 1: dayWeekNo.append(choose(lang, "понедельник", "Monday")); break;
            case 2: dayWeekNo.append(choose(lang, "вторник", "Tuesday")); break;
            case 3: dayWeekNo.append(choose(lang, "среду", "Wednesday")); break;
            case 4: dayWeekNo.append(choose(lang, "четверг", "Thursday")); break;
            case 5: dayWeekNo.append(choose(lang, "пятницу", "Friday")); break;
            case 6: dayWeekNo.append(choose(lang, "субботу", "Saturday")); break;
            default: dayWeekNo.append(choose(lang, "воскресенье", "Sunday"));
        }

        StringBuilder result = new StringBuilder();

        int each = RepetitionUtils.calcREach(repetition);
        if (each == 12) {
            result.append(dayWeekNo).append(' ');

            switch (start.getMonthOfYear()) {
                case 1: result.append(choose(lang, "января", "of January")); break;
                case 2: result.append(choose(lang, "февраля", "of February")); break;
                case 3: result.append(choose(lang, "марта", "of March")); break;
                case 4: result.append(choose(lang, "апреля", "of April")); break;
                case 5: result.append(choose(lang, "мая", "of May")); break;
                case 6: result.append(choose(lang, "июня", "of June")); break;
                case 7: result.append(choose(lang, "июля", "of July")); break;
                case 8: result.append(choose(lang, "августа", "of August")); break;
                case 9: result.append(choose(lang, "сентября", "of September")); break;
                case 10: result.append(choose(lang, "октября", "of October")); break;
                case 11: result.append(choose(lang, "ноября", "of November")); break;
                default: result.append(choose(lang, "декабря", "of December"));
            }
        } else {
            if (lang == Language.RUSSIAN) {
                result.append(dayWeekNo).append(" каждого ").append(formatREachRu(each, "{}-го месяца", "месяца"));
            } else {
                result.append(formatREachEn(each, "every {} month ", "monthly ")).append(dayWeekNo);
            }
        }

        if (repetition.getDueTs().isPresent()) {
            LocalDate dueDate = EventRoutines.convertDueTsToUntilDate(repetition.getDueTs().get(), info.getTz());
            LocalDate nowDate = LocalDate.now(info.getTz());

            result.append(" ").append(convertDueDate(
                    lang, dueDate, sinceInsteadDue,
                    false, each != 12, each == 12 || dueDate.getYear() != nowDate.getYear()));
        }
        return result.toString();
    }

    private static String convertYearly(RepetitionInstanceInfo info, boolean sinceInsteadDue, Language lang) {
        Repetition repetition = info.getRepetition().get();
        Validate.isTrue(repetition.getType() == RegularRepetitionRule.YEARLY);

        DateTime start = info.getEventIntervalWithTz().getStart();
        String date = convertDate(lang, start.toLocalDate(), true, true, false);

        StringBuilder result = new StringBuilder();
        int each = RepetitionUtils.calcREach(repetition);
        if (lang == Language.RUSSIAN) {
            result.append(date).append(formatREachRu(each, " каждый {}-й год", ""));
        } else {
            result.append(formatREachEn(each, "every {} year on ", "on ")).append(date);
        }

        if (repetition.getDueTs().isPresent()) {
            LocalDate dueDate = EventRoutines.convertDueTsToUntilDate(repetition.getDueTs().get(), info.getTz());
            result.append(" ").append(convertDueDate(lang, dueDate, sinceInsteadDue, false, false, true));
        }
        return result.toString();
    }

    private static String convertDueDate(
            Language lang, LocalDate date, boolean sinceInsteadDue, boolean withDay, boolean withMonth, boolean withYear)
    {
        return (sinceInsteadDue ? choose(lang, "с ", "since ") : choose(lang, "до ", "till "))
                + convertDate(lang, date.plusDays(sinceInsteadDue ? 1 : 0), withDay, withMonth, withYear);
    }

    private static String convertDate(
            Language lang, LocalDate date, boolean withDay, boolean withMonth, boolean withYear)
    {
        ListF<String> parts = Cf.arrayList();

        if (withDay) {
            parts.add(lang == Language.ENGLISH ? suffixedNumberEn(date.getDayOfMonth()) : "" + date.getDayOfMonth());
        }
        if (withMonth) {
            switch (date.getMonthOfYear()) {
                case 1: parts.add(choose(lang, "января", "Jan")); break;
                case 2: parts.add(choose(lang, "февраля", "Feb")); break;
                case 3: parts.add(choose(lang, "марта", "Mat")); break;
                case 4: parts.add(choose(lang, "апреля", "Apr")); break;
                case 5: parts.add(choose(lang, "мая", "May")); break;
                case 6: parts.add(choose(lang, "июня", "Jun")); break;
                case 7: parts.add(choose(lang, "июля", "Jul")); break;
                case 8: parts.add(choose(lang, "августа", "Aug")); break;
                case 9: parts.add(choose(lang, "сентября", "Sep")); break;
                case 10: parts.add(choose(lang, "октября", "Oct")); break;
                case 11: parts.add(choose(lang, "ноября", "Nov")); break;
                default: parts.add(choose(lang, "декабря", "Dec"));
            }
        }
        if (withYear) parts.add(date.getYear() + choose(lang, " года", ""));

        return parts.mkString(" ");
    }

    private static String suffixedNumberEn(int number) {
        if (number % 10 == 1 && number % 100 != 11) return number + "st";
        if (number % 10 == 2 && number % 100 != 12) return number + "nd";
        if (number % 10 == 3 && number % 100 != 13) return number + "rd";

        return number + "th";
    }

    private static String choose(Language lang, String ruMessage, String enMessage) {
        return lang == Language.RUSSIAN ? ruMessage : enMessage;
    }

    private static String formatREachRu(int each, String moreThanOneMessage, String lessThanTwoMessage) {
        return format(each > 1 ? moreThanOneMessage : lessThanTwoMessage, each);
    }

    private static String formatREachEn(int each, String moreThanOneMessage, String lessThanTwoMessage) {
        return format(each > 1 ? moreThanOneMessage : lessThanTwoMessage, suffixedNumberEn(each));
    }

    private static final MessageFormatter formatter = new Slf4jLikeMessageFormatter();

    private static String format(String message, Object... args) {
        return formatter.formatMessage(message, args);
    }
}
