package ru.yandex.calendar.logic.event.repetition;

import java.util.Comparator;
import java.util.Optional;
import java.util.SortedSet;

import javax.annotation.Nullable;

import lombok.val;
import net.fortuna.ical4j.model.WeekDay;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Days;
import org.joda.time.Duration;
import org.joda.time.IllegalFieldValueException;
import org.joda.time.Instant;
import org.joda.time.Interval;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;
import org.joda.time.Months;
import org.joda.time.MutableDateTime;
import org.joda.time.Weeks;
import org.joda.time.Years;
import org.joda.time.chrono.ISOChronology;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.RepetitionFields;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.IntervalComparator;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.Freq;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.IcsRecur;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.IcsRecurUntil;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartByDay;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartByMonth;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartByMonthDay;
import ru.yandex.calendar.util.base.AuxColl;
import ru.yandex.calendar.util.base.Binary;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DayOfWeek;
import ru.yandex.calendar.util.dates.WeekdayConv;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;

import static java.util.stream.Stream.concat;

/**
 * @author Stepan Koltsov
 */
public class RepetitionUtils {

    private static final Logger logger = LoggerFactory.getLogger(RepetitionUtils.class);

    public static final Instant END_OF_EVERYTHING_TS = new LocalDate(2100, 1, 1)
            .toDateTimeAtStartOfDay(DateTimeZone.UTC).toInstant();

    public static final Instant START_OF_EVERYTHING_TS = new LocalDate(1899, 1, 1)
            .toDateTimeAtStartOfDay(DateTimeZone.UTC).toInstant();

    public static Long getDueMs(Repetition r) {
        if (r == null || !r.isFieldSet(RepetitionFields.DUE_TS) || r.getDueTs().getOrNull() == null) {
            return null;
        } else {
            return r.getDueTs().get().getMillis();
        }
    }

    public static InstantInterval getRdateInterval(Rdate rdate, Interval eventInterval) {
        return new InstantInterval(rdate.getStartTs(), getRdateEndTs(rdate, eventInterval));
    }

    public static Instant getRdateEndTs(Rdate rdate, Interval eventInterval) {
        return rdate.getEndTs().orElseGet(() -> rdate.getStartTs().plus(eventInterval.toDurationMillis()));
    }

    /**
     * Calculates minimal interval, in which lies all instances of
     * the event. Such interval could have infinity right bound
     * @param rii stores information about event instances
     * @return calculated common minimal interval
     */
    public static InfiniteInterval calcCommonInstsInterval(RepetitionInstanceInfo rii) {
        if (rii.isEmpty()) {
            return new InfiniteInterval(rii.getEventInterval().getStart(), Option.of(rii.getEventInterval().getEnd()));
        }
        Long dueMs = getDueMs(rii.getRepetition().getOrNull());
        Long stMs = rii.getEventInterval().getStartMillis();
        Long endMs = dueMs;
        ListF<Rdate> rdates = rii.getRdates();
        for (Rdate rdate : rdates) {
            InstantInterval rdInt = getRdateInterval(rdate, rii.getEventInterval().toInterval(rii.getTz()));
            stMs = Math.min(stMs, rdInt.getStartMillis());
            if (endMs != null) {
                endMs = Math.max(endMs, rdInt.getEndMillis());
            }
        }
        return new InfiniteInterval(new Instant(stMs), Option.ofNullable(endMs).map(Instant::new));
    }

    public static Optional<Instant> calcLatestInstanceEnd(RepetitionInstanceInfo info) {
        if (info.isEmpty()) {
            return Optional.of(info.getEventInterval().getEnd());
        }

        if (info.getRepetition().exists(r -> r.getDueTs().isEmpty())) {
            return Optional.empty();
        }

        val rDatesEnds = info.getRdates().stream().flatMap(d -> d.getEndTs().stream());
        val recurrencesEnds = info.getRecurrences().stream().map(RecurrenceTimeInfo::getEnd);
        val occurrencesEnds = info.getRepetition().stream().flatMap(r ->
                getIntervals(info, info.getEventStart(), r.getDueTs(), false, 0)
                        .stream()
                        .map(InstantInterval::getEnd)
        );

        return concat(concat(rDatesEnds, recurrencesEnds), occurrencesEnds).max(Comparator.naturalOrder());
    }

    public static Option<Instant> dueTsForEventIndent(RepetitionInstanceInfo info) {
        Function<Instant, Instant> stretchDueF = due -> { // CAL-7586
            Option<InstantInterval> overlap = RepetitionUtils.getIntervals(info, due, Option.empty(), true, 1).singleO();
            return overlap.map(InstantInterval::getEnd).getOrElse(due);
        };
        return info.getRepetition().map(r -> r.getDueTs().map(stretchDueF).getOrElse(END_OF_EVERYTHING_TS));
    }

    public static Instant instanceStart(RepetitionInstanceInfo repetitionInstanceInfo, int count) {
        Validate.isTrue(count > 0);

        InstantInterval firstInstanceInterval = repetitionInstanceInfo.getEventInterval();

        ListF<InstantInterval> intervals = RepetitionUtils.getIntervals(repetitionInstanceInfo, firstInstanceInterval.getStart(),
                Option.empty(),
                false, count);

        Validate.isTrue(intervals.size() == count);

        return intervals.last().getStart();
    }

    /**
     * For given event, repetition and interval, obtains intervals
     * (bounded to event's chronology) which either overlap given
     * interval or begin in it (depending on the <code>overlap</code> parameter)
     * @param repetitionInstanceInfo repetition rule for event
     * @param iStartMs start of checking interval, cannot be 0
     * @param iEndMsO end of checking interval, can be 0 if sorted and
     * small-limited (e.g. limit == 1) query is performing
     * @param overlap if true, then event should only overlap checking interval;
     * otherwise, event should begin within checking interval
     * (the latter is more strict)
     * @param limit if &gt; 0, sets the limit for the result array size
     * (for consistency, this is assumed and assured that
     * limited queries set 'sorted' parameter to true)
     * @return event intervals that satisfy overlapping condition
     */
    public static ListF<InstantInterval> getIntervals(
            RepetitionInstanceInfo repetitionInstanceInfo, Instant iStartMs,
            Option<Instant> iEndMsO, boolean overlap, int limit)
    {
         return getIntervals(repetitionInstanceInfo, iStartMs, iEndMsO, overlap, limit, false, false);
    }

    public static ListF<InstantInterval> getIntervals(
            RepetitionInstanceInfo repetitionInstanceInfo, Instant iStartMs,
            Option<Instant> iEndMsO, boolean overlap, int limit, boolean ignoreExdates, boolean includeRecurrences)
    {
        Instant iEndMs;
        if (!iEndMsO.isPresent()) {
            if (limit <= 0) {
                throw new IllegalArgumentException("unlimited query without endMs requested");
            }
            iEndMs = END_OF_EVERYTHING_TS;
        } else {
            iEndMs = iEndMsO.get();
        }
        if (iEndMs.isBefore(iStartMs)) return Cf.list();

        repetitionInstanceInfo = moveEventIntervalToDayOfRepetition(repetitionInstanceInfo);

        // Take end date into attention.
        // Note that this is just optimization, but not enough.
        // Example: for 'weekly', we need to check whether each calculated instant < due ts
        InfiniteInterval commInt = calcCommonInstsInterval(repetitionInstanceInfo);
        if (commInt.getEnd().isSome(commInt.getStart())) {
            InstantInterval i = new InstantInterval(iStartMs, iEndMs);
            InstantInterval ei = repetitionInstanceInfo.getEventInterval();
            SetF<Instant> exs = repetitionInstanceInfo.getExcludeInstants(ignoreExdates);

            return Option.when(eventIntervalSuitable(ei, Option.empty(), i, overlap, exs), ei);
        }
        if (commInt.getEnd().isPresent()) {
            iEndMs = ObjectUtils.max(iStartMs, ObjectUtils.min(commInt.getEnd().get(), iEndMs));
        }
        if (iEndMs.getMillis() < commInt.getStart().getMillis()) {
            return Cf.list();
        }

        final Interval iInterval = new Interval(iStartMs.getMillis(), iEndMs.getMillis(), repetitionInstanceInfo.getTz());
        // In resSet, only proper intervals lie
        final SetF<InstantInterval> resSet = Cf.x(AuxColl.newTSet(IntervalComparator.INSTANCE));

        SetF<Instant> exTimestamps = repetitionInstanceInfo.getExcludeInstants(ignoreExdates);
        if (!repetitionInstanceInfo.getRepetition().isPresent() ||
                repetitionInstanceInfo.getRepetition().get().getType() == RegularRepetitionRule.NONE)
        {
            InstantInterval ei = repetitionInstanceInfo.getEventInterval();
            if (RepetitionUtils.eventIntervalSuitable(ei, Option.empty(),
                    InstantInterval.valueOf(iInterval), overlap, exTimestamps))
            {
                resSet.add(ei);
            }
        } else {
            Option<Instant> dueTs = repetitionInstanceInfo.getRepetition().get()
                    .getFieldValueO(RepetitionFields.DUE_TS)
                    .filterNotNull();

            final DateTime movedIntStartDt = (overlap ?
                iInterval.getStart().minus(repetitionInstanceInfo.getEventPeriod()) : // event should overlap given interval
                iInterval.getStart() // event should begin in given interval, so do nothing
            );
            MutableDateTime curUnitStartDt = repetitionInstanceInfo.getEventInterval().getStart().toMutableDateTime(
                    repetitionInstanceInfo.getTz());

            RegularRepetition regularRepetition = new RegularRepetition(repetitionInstanceInfo.getRepetition().get(), repetitionInstanceInfo.getEventIntervalWithTz().getStart());

            // NOTE: joda time seems to assign MONDAY as the first day of the week
            //       (see org.joda.time.chrono.BasicWeekOfWeekyearDateTimeField.roundFloor()
            //        and org.joda.time.DateTimeConstants.MONDAY).
            //       This means that the week rounding will align to Monday and,
            //       in other words, our offsets addition principle works
            //       (MONDAY = +0 days, ... SUNDAY = +6 days)

            regularRepetition.alignToUnit(curUnitStartDt);

            final LocalTime eventLocalTime = repetitionInstanceInfo.getEventIntervalWithTz().getStart().toLocalTime();

            if (repetitionInstanceInfo.getEventInterval().getStart().isBefore(movedIntStartDt)) {
                regularRepetition.jumpToBoundByUnits(curUnitStartDt, movedIntStartDt.toInstant());
            } else {
                // try to add event interval itself because
                // intervals that we get through repetition
                // may not contain event interval by occasion
                // (e.g. Wednesday event repeats on Fridays only)
                if (RepetitionUtils.eventIntervalSuitable(
                        repetitionInstanceInfo.getEventInterval(), dueTs,
                        InstantInterval.valueOf(iInterval), overlap, exTimestamps))
                {
                    resSet.add(repetitionInstanceInfo.getEventInterval());
                }
            }

            while (
                curUnitStartDt.isBefore(iInterval.getEnd()) &&
                (limit <= 0 || resSet.size() < limit)
            )
            {
                // NOTE: eventDatesInUnit() must care of ascending order of the result timestamps
                ListF<LocalDate> ris = regularRepetition.eventDatesInUnit(curUnitStartDt.toDateTime(repetitionInstanceInfo.getTz()).toLocalDate());
                for (LocalDate ri : ris) {
                    if (limit <= 0 || resSet.size() < limit) {
                        try {
                            DateTimeZone tz = repetitionInstanceInfo.getTz();
                            LocalDateTime startLocal = ri.toLocalDateTime(eventLocalTime);
                            LocalDateTime endLocal = startLocal.plus(repetitionInstanceInfo.getEventPeriod());

                            final DateTime newEIStart = AuxDateTime.toDateTimeIgnoreGap(startLocal, tz);
                            final DateTime newEIEnd = AuxDateTime.toDateTimeIgnoreGap(endLocal, tz);

                            InstantInterval eventInterval = new InstantInterval(newEIStart, newEIEnd);
                            if (eventIntervalSuitable(eventInterval, dueTs,
                                    InstantInterval.valueOf(iInterval), overlap, exTimestamps))
                            {
                                resSet.add(eventInterval);
                            }
                        } catch (IllegalFieldValueException e) {
                            // XXX: explain why it is necessary? // stepancheg@
                            final String msg =
                                "!!! Cannot create event instance for " +
                                "'ri' = " + ri + ", 'eLt' = " + eventLocalTime;
                            logger.info(msg);
                        }
                    }
                }
                regularRepetition.addUnit(curUnitStartDt);
            }
        }

        Function1B<InstantInterval> isSuitableF = interval -> eventIntervalSuitable(interval, Option.empty(),
                InstantInterval.valueOf(iInterval), overlap, exTimestamps);

        for (Rdate rdate : repetitionInstanceInfo.getRdates()) {
            InstantInterval rdateInterval = getRdateInterval(
                    rdate, repetitionInstanceInfo.getEventInterval().toInterval(repetitionInstanceInfo.getTz()));
            if (isSuitableF.apply(rdateInterval)) {
                resSet.add(rdateInterval);
            }
        }

        if (includeRecurrences) {
            resSet.addAll(repetitionInstanceInfo.getRecurrences().map(i -> i.getInterval()).filter(isSuitableF));
        }

        if (limit <= 0) {
            return resSet.toList();
        } else {
            return resSet.toList().take(limit);
        }
    }

    public static RepetitionInstanceInfo moveEventIntervalToDayOfRepetition(RepetitionInstanceInfo repetitionInfo) {
        if (!repetitionInfo.getRepetition().isPresent()) return repetitionInfo;

        InstantInterval moved = moveEventIntervalToDayOfRepetition(
                repetitionInfo.getEventInterval(), repetitionInfo.getRepetition().get(), repetitionInfo.getTz());

        return new RepetitionInstanceInfo(
                moved, repetitionInfo.getTz(), repetitionInfo.getRepetition(),
                repetitionInfo.getRdates(), repetitionInfo.getExdates(), repetitionInfo.getRecurrences());
    }

    public static InstantInterval moveEventIntervalToDayOfRepetition(
            InstantInterval eventInterval, Repetition repetition, DateTimeZone eventTz)
    {
        if (repetition.getType() == RegularRepetitionRule.WEEKLY) {
            ListF<Integer> days = Cf.x(repetition.getRWeeklyDays().get().split(",")).map(WeekdayConv.calsToJodaF());

            DateTime start = eventInterval.getStart().toDateTime(eventTz);

            if (!days.containsTs(start.getDayOfWeek())) {
                Option<Integer> nextDayO = days.filter(Cf.Integer.gtF(start.getDayOfWeek())).minO();

                start = nextDayO.isPresent() ?
                        start.withDayOfWeek(nextDayO.get()) :
                        start.plusWeeks(1).withDayOfWeek(days.min());

                return new InstantInterval(start.toInstant(), start.plus(eventInterval.toPeriod(eventTz)).toInstant());
            }
        }
        return eventInterval;
    }

    // if event start != 1st repetition interval, it just adds another event instance // ssytnik@
    public static boolean isValidStart(RepetitionInstanceInfo repetitionInstanceInfo, Instant start) {
        return getInstanceIntervalStartingAt(repetitionInstanceInfo, start).isPresent();
    }

    public static Option<InstantInterval> getInstanceIntervalStartingAt(
            RepetitionInstanceInfo repetitionInfo, Instant start)
    {
        return getIntervals(repetitionInfo, start, Option.empty(), false, 1)
                .singleO().filter(i -> i.getStart().isEqual(start));
    }

    public static Option<InstantInterval> getFirstInstanceInterval(RepetitionInstanceInfo repetitionInfo) {
        Instant searchStart = repetitionInfo.getEventInterval().getStart();
        return getIntervals(repetitionInfo, searchStart, Option.empty(), false, 1).singleO();
    }

    public static Option<InstantInterval> getInstanceIntervalStartingAfter(
            RepetitionInstanceInfo repetitionInfo, Instant after)
    {
        return getIntervals(repetitionInfo, after, Option.empty(), false, 1).singleO();
    }

    public static Option<InstantInterval> getInstanceIntervalEndingAfter(
            RepetitionInstanceInfo repetitionInfo, Instant after)
    {
        return getIntervals(repetitionInfo, after, Option.empty(), true, 1).singleO();
    }

    public static ListF<InstantInterval> getInstanceIntervalsStartingIn(
            RepetitionInstanceInfo repetitionInfo, InstantInterval interval)
    {
        return getIntervals(repetitionInfo, interval.getStart(), Option.of(interval.getEnd()), false, 0).singleO();
    }

    public static Option<InstantInterval> getClosestInstanceInterval(
            RepetitionInstanceInfo repetitionInfo, Instant instant)
    {
        Option<InstantInterval> after = getIntervals(repetitionInfo, instant, Option.empty(), true, 1).singleO();

        if (after.isPresent()) return after;

        return getIntervals(repetitionInfo, repetitionInfo.getEventStart(), Option.of(instant), false, 0).lastO();
    }

    public static Option<InstantInterval> getClosestInstanceIntervalWithRecurrence(
            RepetitionInstanceInfo repetitionInfo, Instant instant)
    {
        Option<InstantInterval> after = getIntervals(
                repetitionInfo, instant, Option.empty(),
                true, 1, false, true).singleO();

        if (after.isPresent()) return after;

        return getIntervals(repetitionInfo, repetitionInfo.getEventStart(),
                Option.of(instant), false, 0, false, true).lastO();
    }

    public static ListF<InstantInterval> getInstancesInInterval(
            RepetitionInstanceInfo repetitionInfo, InfiniteInterval interval)
    {
        return getIntervals(repetitionInfo, interval.getStart(), interval.getEnd(), true, -1);
    }

    public static ListF<InstantInterval> getInstancesInInterval(
            RepetitionInstanceInfo repetitionInfo, InstantInterval interval)
    {
        return getIntervals(repetitionInfo, interval.getStart(), Option.of(interval.getEnd()), true, -1);
    }

    public static boolean hasInstanceAfter(RepetitionInstanceInfo repetitionInfo, Instant after) {
        return getInstanceIntervalStartingAfter(repetitionInfo, after).isNotEmpty();
    }

    private static boolean eventIntervalSuitable(
            InstantInterval ei, Option<Instant> dueTs,
            InstantInterval i, boolean overlap, SetF<Instant> exTimestamps)
    {
        return !exTimestamps.containsTs(ei.getStart())
                && (!dueTs.isPresent() || dueTs.get().isAfter(ei.getStart()))
                && suitable(ei, i, overlap);
    }

    public static boolean suitable(InstantInterval ei, InstantInterval i, boolean overlap) {
        return (overlap && RepetitionUtils.overlapsOrSameStart(ei, i)) || (!overlap && RepetitionUtils.beginsIn(ei, i));
    }

    static boolean beginsIn(InstantInterval ei, InstantInterval i) { // does ei begin in i?
        return i.contains(ei.getStart()) || ei.getStartMillis() == i.getStartMillis();
    }

    public static boolean overlapsOrSameStart(InstantInterval ei, InstantInterval i) { // does ei overlap i?
        return ei.overlaps(i) || ei.getStartMillis() == i.getStartMillis();
    }

    public static Option<InstantInterval> overlapStartsAbutInclusive(InstantInterval i1, InstantInterval i2) {
        if (i1.overlaps(i2)) {
            return Option.of(i1.overlap(i2));
        }
        if (i1.getStartMillis() == i2.getStartMillis()) {
            return Option.of(new InstantInterval(i1.getStart(), i1.getStart()));
        }
        return Option.empty();
    }

    public static int calcREach(Repetition r) {
        return calcREach(r.getREach().getOrNull()); // '0'
    }

    public static int calcREach(@Nullable Integer rEach) {
        return rEach == null || rEach == 0 ? 1 : rEach;
    }

    // [ooooooooooo ?? dt o]
    static boolean isLastWeekDayInMonth(DateTime dt) {
        return (
            dt.dayOfMonth().getMaximumValue() - dt.getDayOfMonth() <
            dt.dayOfWeek().getMaximumValueOverall()
        );
    }

    // [ooooooooooo <- dt o]
    // dt is not supposed to be aligned
    static int getTotalWeekDaysInMonth(LocalDate monthDt, int testWeekDay)
    {
        final int daysInWeek = monthDt.dayOfWeek().getMaximumValueOverall(); // 7
        final int daysInMonth = monthDt.dayOfMonth().getMaximumValue(); // 28 .. 31
        final int commonCount = daysInMonth / daysInWeek; // 4 (only for days from interval 28 .. 31)
        final int remainderLength = daysInMonth % daysInWeek; // 0 .. 6
        final int  weekDayLastFullWeek = monthDt.withDayOfMonth(daysInMonth - remainderLength).getDayOfWeek(); // 1.. 7
        final int offs1 = WeekdayConv.jodaToOffs(weekDayLastFullWeek); // 0 .. 6
        final int testOffs = WeekdayConv.jodaToOffs(testWeekDay); // 0 .. 6
        final int offs2 = (offs1 + remainderLength) % daysInWeek; // 0 .. 6
        boolean isTestWeekDayInRemainder;
        if (offs1 <= offs2)
            isTestWeekDayInRemainder = offs1 < testOffs && testOffs <= offs2;
        else
            isTestWeekDayInRemainder = offs1 < testOffs || testOffs <= offs2;
        return commonCount + (isTestWeekDayInRemainder ? 1 : 0);
    }

    // [oooooo -> dt ......]
    // monthDt assumed to be month-aligned (and no check for 'i')
    static LocalDate setIthWeekDayInMonth(LocalDate monthDt, int weekDay, int i) {

        int
            daysInWeek = monthDt.dayOfWeek().getMaximumValue(), // getMaximumValueOverall?
            firstDayOffs = WeekdayConv.jodaToOffs(monthDt.getDayOfWeek()),
            weekDayOffs = WeekdayConv.jodaToOffs(weekDay);
        int dayNum =
            ((weekDayOffs - firstDayOffs + daysInWeek) % daysInWeek + 1) + (i - 1) * daysInWeek;
        return monthDt.withDayOfMonth(dayNum);
    }

    // [oooooo <- dt ......]
    static int getNumWeekDaysForDate(DateTime dt) {
        return (dt.getDayOfMonth() - 1) / 7 + 1;
    }

    public static int getApproxMinLengthDays(Repetition repetition) {
        final Integer rEach = calcREach(repetition);
        switch (RegularRepetitionRule.find(repetition)) {
        case DAILY:
            return rEach;
        case WEEKLY:
            int res = Integer.MAX_VALUE;
            String str = repetition.getRWeeklyDays().getOrElse("");
            ListF<String> rWeeklyDays = Cf.x(StringUtils.split(str, ","));
            if (rWeeklyDays.length() == 0) {
                return res;
            }
            SortedSet<Integer> wdOffsets = AuxColl.newTSet();
            for (String wd : rWeeklyDays) { wdOffsets.add(WeekdayConv.calsToOffs(wd)); }
            int offs1 = -1, offs2 = -1;
            for (int offs : wdOffsets) {
                if (offs1 == -1) { offs1 = offs; } else { res = Math.min(res, offs - offs2); }
                offs2 = offs;
            }
            res = Math.min(res, rEach * 7 + offs1 - offs2);
            return res;
        case MONTHLY_NUMBER:
        case MONTHLY_DAY_WEEKNO:
            return rEach * 28;
        case YEARLY:
            return rEach * 365;
        default:
            throw new IllegalArgumentException();
        }
    }

    public static int unitsBetween(DateTime start, DateTime end, RegularRepetitionRule rule) {
        switch (rule) {
        case DAILY:
            return Days.daysBetween(start, end).getDays();
        case WEEKLY:
            return Weeks.weeksBetween(start, end).getWeeks();
        case MONTHLY_NUMBER:
            return Months.monthsBetween(start, end).getMonths();
        case MONTHLY_DAY_WEEKNO:
            return Months.monthsBetween(start, end).getMonths();
        case YEARLY:
            return Years.yearsBetween(start, end).getYears();
        default:
            throw new IllegalArgumentException("unsupported rule: " + rule);
        }
    }

    public static String outInfo(Repetition repetition) {
        switch (RegularRepetitionRule.find(repetition)) {
        case DAILY:
            return "day";
        case WEEKLY:
            return "week on day(s): " + repetition.getRWeeklyDays().getOrNull();
        case MONTHLY_NUMBER:
            return "month (given number)";
        case MONTHLY_DAY_WEEKNO:
            String isLastWeekStr = (Binary.toBoolean(repetition.getRMonthlyLastweek().getOrNull()) ? "LAST" : "Ith");
            return "month (given " + isLastWeekStr + " week day)";
        case YEARLY:
            return "year";
        default:
            return repetition.toString();
        }
    }

    public static void copyFields(RegularRepetitionRule rule, Repetition from, Repetition to) {
        switch (rule) {
        case DAILY:
            return;
        case WEEKLY:
            to.setRWeeklyDays(from.getRWeeklyDays().getOrNull());
            return;
        case MONTHLY_NUMBER:
            return;
        case MONTHLY_DAY_WEEKNO:
            to.setRMonthlyLastweek(from.getRMonthlyLastweek().getOrNull());
            return;
        case YEARLY:
            return;
        default:
            throw new IllegalArgumentException();
        }
    }

    public static Option<Repetition> getRepetitionSafe(IcsRecur recur, Instant startTs, DateTimeZone tz) {
        try {
            return Option.of(getRepetition(recur, startTs, tz));

        } catch (Exception e) {
            logger.error("Failed to convert " + recur + ": " + e, e);
            return Option.empty();
        }
    }

    public static Repetition getRepetition(IcsRecur recur, Instant startTs, DateTimeZone tz) {
        Freq freq = recur.getFreq();
        if (Freq.HOURLY == freq || Freq.MINUTELY == freq || Freq.SECONDLY == freq) {
            throw new UnsupportedRepetitionException("Unsupported frequency encountered: " + freq);
        }
        Repetition repetition = new Repetition();
        RegularRepetitionRule rule = RegularRepetitionRule.find(recur);

        ListF<WeekDay> weekDays = recur.getDayList();

        if (RegularRepetitionRule.DAILY == rule && weekDays.isNotEmpty() || RegularRepetitionRule.WEEKLY == rule) {
            rule = RegularRepetitionRule.WEEKLY;

            if (weekDays.isNotEmpty()) {
                repetition.setRWeeklyDays(weekDays
                        .map(DayOfWeek.byWeekDayF()).unique().sorted().map(DayOfWeek.getDbValueF()).mkString(","));
            } else {
                int day = ISOChronology.getInstance(tz).dayOfWeek().get(startTs.getMillis());
                repetition.setRWeeklyDays(DayOfWeek.byJodaDayOfWeek(day).getDbValue());
            }

        } else if (RegularRepetitionRule.MONTHLY_DAY_WEEKNO == rule && weekDays.isNotEmpty()) {
            WeekDay wd = weekDays.first();
            repetition.setRMonthlyLastweek(wd.getOffset() == -1);
        }
        repetition.setType(rule);

        if (recur.getWeekStartDay().isPresent()) {
            logger.debug("WKST is ignored");
        }
        repetition.setREach(recur.getInterval().getOrElse(1));

        Option<Integer> recurCount = recur.getCount();
        Option<IcsRecurUntil> recurUntil = recur.getUntil();

        if (recurUntil.isPresent()) {
            Instant dueTs;
            if (recurUntil.get().isDateTime()) {
                Instant untilTs = recurUntil.get().getDateTime();
                dueTs = EventRoutines.calculateDueTsFromIcsUntilTs(startTs, untilTs, tz);
            } else {
                LocalDate untilDate = recurUntil.get().getDate();
                dueTs = EventRoutines.calculateDueTsFromUntilDate(startTs, untilDate, tz);
            }
            repetition.setDueTs(dueTs);

        } else if (recurCount.isPresent()) {
            RepetitionInstanceInfo r = RepetitionInstanceInfo.create(new InstantInterval(startTs, startTs), tz, Option.of(repetition));
            repetition.setDueTs(RepetitionUtils.instanceStart(r, recurCount.get()).plus(Duration.standardDays(1)));

        } else {
            repetition.setDueTsNull();
        }
        return repetition;
    }

    public static IcsRecur getRecur(EventWithRelations event, Repetition repetition) {
        RegularRepetitionRule rule = RegularRepetitionRule.find(repetition);
        DateTime startDt = event.getEvent().getStartTs().toDateTime(event.getTimezone());

        IcsRecur recur = getBaseRecur(rule, startDt, repetition);
        if (repetition.getDueTs().isPresent()) {
            LocalDate dueDate = EventRoutines.convertDueTsToUntilDate(repetition.getDueTs().get(), event.getTimezone());
            if (!event.getEvent().getIsAllDay()) {
                Instant untilTs = startDt.withFields(dueDate).toInstant();
                // Hack to prevent lost last instance by clients with invalid time zone
                Instant incrementedUntilTs = untilTs.plus(Duration.standardHours(1));
                recur = recur.withUntilTs(incrementedUntilTs);
            } else {
                recur = recur.withUntilDate(dueDate);
            }
        }
        return recur.withInterval(RepetitionUtils.calcREach(repetition));
    }

    public static Option<IcsRecur> getRecurWithExcludesAndNoDue(RepetitionInstanceInfo repetitionInfo) {
        if (!repetitionInfo.getRepetition().isPresent()) return Option.empty();

        Repetition repetition = repetitionInfo.getRepetition().get();

        RegularRepetitionRule rule = RegularRepetitionRule.find(repetition);

        return Option.of(getBaseRecur(rule, repetitionInfo.getEventIntervalWithTz().getStart(), repetition)
                .withInterval(calcREach(repetition))
                .withXExclude(repetitionInfo.getExcludeInstants(false)));
    }

    private static IcsRecur getBaseRecur(RegularRepetitionRule rule, DateTime eStartDt, Repetition r) {
        IcsRecur recur = new IcsRecur(rule.getICalFreq());
        switch (rule) {
        case DAILY:
            return recur;
        case WEEKLY:
            String str = r.getRWeeklyDays().getOrElse("");
            ListF<String> result = Cf.x(StringUtils.split(str, ","));
            ListF<WeekDay> days = result.map(new Function<String, WeekDay>() {
                public WeekDay apply(String calsWd) {
                    return WeekdayConv.calsToIcal(calsWd);
                }
            });
            return recur.withPart(new IcsRecurRulePartByDay(days));
        case MONTHLY_NUMBER:
            // BYMONTHDAY is actually required to distinguish between this and MONTHLY_DAY_WEEKNO
            // 1 .. 31 -> 1.. 31
            return recur.withPart(new IcsRecurRulePartByMonthDay(Cf.list(eStartDt.getDayOfMonth())));
        case MONTHLY_DAY_WEEKNO:
            // BYDAY is actually required to distinguish between this and MONTHLY_NUMBER
            int numWeekDays = getNumWeekDaysForDate(eStartDt);
            WeekDay wd = new WeekDay(
                WeekdayConv.jodaToIcal(eStartDt.getDayOfWeek()),
                (Binary.toBoolean(r.getRMonthlyLastweek().getOrNull()) || numWeekDays > 4 ? -1 : numWeekDays)
            );
            return recur.withPart(new IcsRecurRulePartByDay(Cf.list(wd)));
        case YEARLY:
            // BYMONTH and BYMONTHDAY are optional, but let's add them anyway
            recur = recur.withPart(new IcsRecurRulePartByMonth(Cf.list(eStartDt.getMonthOfYear())));
            recur = recur.withPart(new IcsRecurRulePartByMonthDay(Cf.list(eStartDt.getDayOfMonth())));
            return recur;
        default:
            throw new IllegalArgumentException();
        }
    }

    // all events in this unit that satisfy repetition rules
    // but may violate time bounds (first event start and due ts)
    public static ListF<LocalDate> eventDatesInUnitCandidates(
            RegularRepetitionRule rule,
            LocalDate unitStart, DateTime firstInstanceStart, Repetition r)
    {
        switch (rule) {
        case DAILY:
            return Cf.list(unitStart);
        case WEEKLY: {
            String rWeekDaysStr = r.getRWeeklyDays().getOrElse("");

            ListF<String> rWeekDays = Cf.x(StringUtils.split(rWeekDaysStr, ","));
            ListF<LocalDate> res = Cf.arrayList();
            for (String rWeekDay : rWeekDays) {
                int offset = WeekdayConv.calsToOffs(rWeekDay);
                res.add(unitStart.plusDays(offset));
            }
            // Bug CAL-687 caused error because returned timestamps must be sorted
            // (placing them in the order 'mon' ... 'sun' would be enough, if the
            //  order of weekdays in 'rWeekDaysStr' was guaranteed; but it's not).
            return res.sorted();
        }
        case MONTHLY_NUMBER: {
            if (firstInstanceStart.getDayOfMonth() <= unitStart.dayOfMonth().getMaximumValue()) {
                return Cf.list(unitStart.withDayOfMonth(firstInstanceStart.getDayOfMonth()));
            } else {
                return Cf.list(); // do not offer 'moved' date, just skip
            }
        }
        case MONTHLY_DAY_WEEKNO: {
            int eDayOfWeek = firstInstanceStart.getDayOfWeek();
            // dtTotalWeekDays can vary from 1 to 5
            int dtTotalWeekDays = RepetitionUtils.getTotalWeekDaysInMonth(unitStart, eDayOfWeek); // in unitStart's month
            int eWeekDay = RepetitionUtils.getNumWeekDaysForDate(firstInstanceStart); // for event's date
            int numToSet;
            if ( // if last-weekday
                RepetitionUtils.isLastWeekDayInMonth(firstInstanceStart) && // event allows last-week-day-of-month usage
                Binary.toBoolean(r.getRMonthlyLastweek().getOrNull()) // and this option is used in rep.
                    || eWeekDay > 4
            ) {
                numToSet = dtTotalWeekDays;
            } else { // if just i-th week ('i' specified by event start date)
                if (dtTotalWeekDays < eWeekDay) {
                    return Cf.list();
                }
                numToSet = eWeekDay;
            }
            return Cf.list(
                    RepetitionUtils.setIthWeekDayInMonth(unitStart, eDayOfWeek, numToSet));
        }
        case YEARLY: {
            // Always repeat event in a year, even if there is no such number in a month
            LocalDate dtWithMonth = unitStart.withMonthOfYear(firstInstanceStart.getMonthOfYear());
//                    return new Instant[] { dtWithMonth.withDayOfMonth(Math.min(
//                        dtWithMonth.dayOfMonth().getMaximumValue(), eStart.getDayOfMonth()
//                    )) };
            // NOW: just do not create event
            if (dtWithMonth.dayOfMonth().getMaximumValue() < firstInstanceStart.getDayOfMonth()) {
                return Cf.list();
            }
            return Cf.list(dtWithMonth.withDayOfMonth(firstInstanceStart.getDayOfMonth()));
        }
        default:
            throw new IllegalStateException();
        }
    }

    public static Function<RepetitionInstanceInfo, Option<InstantInterval>> getClosestInstanceIntervalF(
            final Instant instant)
    {
        return new Function<RepetitionInstanceInfo, Option<InstantInterval>>() {
            public Option<InstantInterval> apply(RepetitionInstanceInfo r) {
                return getClosestInstanceInterval(r, instant);
            }
        };
    }

    public static Rdate consExdate(Instant date) {
        Rdate rdate = new Rdate();
        rdate.setIsRdate(false);
        rdate.setStartTs(date);
        rdate.setEndTsNull();

        return rdate;
    }

    public static Rdate consExdateEventId(long eventId, Instant date) {
        Rdate exdate = consExdate(date);
        exdate.setEventId(eventId);

        return exdate;
    }

    public static boolean repetitionRuleChanged(Repetition changes) {
        return changes.isAnyFieldSet(
                RepetitionFields.TYPE, RepetitionFields.R_EACH,
                RepetitionFields.R_MONTHLY_LASTWEEK, RepetitionFields.R_WEEKLY_DAYS);
    }

} //~
