package ru.yandex.calendar.logic.event.web;

import java.util.Optional;

import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.CalendarUtils;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.frontend.web.cmd.run.ui.event.SpecialNames;
import ru.yandex.calendar.frontend.webNew.dto.out.MoveResourceEventsIds;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.XivaNotificationManager;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.ActorId;
import ru.yandex.calendar.logic.event.CreateInfo;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventInstanceForUpdate;
import ru.yandex.calendar.logic.event.EventInstanceInfo;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventLayerId;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.EventsAttachedLayerIds;
import ru.yandex.calendar.logic.event.EventsAttachedUser;
import ru.yandex.calendar.logic.event.EventsOnLayerChangeHandler;
import ru.yandex.calendar.logic.event.ModificationInfo;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.EventLayerDao;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.model.EventInvitationUpdateData;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.event.model.WebReplyData;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.log.EventIdLogDataJson;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.log.change.EventChangeLogEvents;
import ru.yandex.calendar.logic.log.change.EventChangesJson;
import ru.yandex.calendar.logic.log.change.changes.UserRelatedChangesJson;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.notification.xiva.notify.XivaMobileApiNotifier;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sending.param.EventMessageParameters;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.InvitationProcessingMode;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.UserParticipantInfo;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.update.LockResource;
import ru.yandex.calendar.logic.update.LockTransactionManager;
import ru.yandex.calendar.logic.update.UpdateLock2;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class EventWebManager {
    private static final Logger logger = LoggerFactory.getLogger(EventWebManager.class);

    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private EventWebRemover eventRemover;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private EventWebUpdater eventUpdater;
    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private EventLayerDao eventLayerDao;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private UpdateLock2 updateLock2;
    @Autowired
    private LockTransactionManager lockTransactionManager;
    @Autowired
    private UserManager userManager;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private EventsOnLayerChangeHandler eventsOnLayerChangeHandler;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private XivaNotificationManager xivaNotificationManager;
    @Autowired
    private XivaMobileApiNotifier xivaMobileNotifier;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private EventsLogger eventsLogger;

    public EventInstanceInfo getEvent(
            Option<PassportUid> uid, long eventId, Option<Instant> eventStartO,
            Option<Integer> sequence, ActionInfo actionInfo)
    {
        EventInstanceInfo eventInfo = eventRoutines.getSingleInstance(
                uid, eventStartO, eventId, actionInfo.getActionSource());

        checkEventExistsAndNotModified(eventId, sequence, eventStartO);

        return eventInfo;
    }

    public ModificationInfo deleteEvent(UserInfo user, long eventId, Option<Instant> eventStartO, boolean applyToFuture,
                                        ActionInfo actionInfo) {
        return deleteEvent(user, eventId, eventStartO, applyToFuture, true, actionInfo);
    }

    public ModificationInfo deleteEvent(UserInfo user, long eventId, Option<Instant> eventStartO, boolean applyToFuture,
                                        boolean sendMails, ActionInfo actionInfo) {
        return deleteEvent(user, IdOrExternalId.id(eventId), Option.empty(), eventStartO, applyToFuture, sendMails, actionInfo);
    }

    public ModificationInfo deleteEvent(UserInfo user, IdOrExternalId eventId, Option<Integer> sequence,
                                        Option<Instant> eventStartO, boolean applyToFuture, ActionInfo actionInfo) {
        return deleteEvent(user, eventId, sequence, eventStartO, applyToFuture, true, actionInfo);
    }

    public ModificationInfo deleteEvent(UserInfo user, IdOrExternalId eventId, Option<Integer> sequence,
                                        Option<Instant> eventStartO, boolean applyToFuture, boolean sendMails,
                                        ActionInfo actionInfo) {
        return deleteEvent(UserOrYaCalendar.user(user), eventId, sequence, eventStartO, applyToFuture, sendMails, actionInfo);
    }

    private ModificationInfo deleteEvent(UserOrYaCalendar userOrYaCalendar, IdOrExternalId eventId, Option<Integer> sequence,
                                         Option<Instant> eventStartO, boolean applyToFuture, ActionInfo ai) {
        return deleteEvent(userOrYaCalendar, eventId, sequence, eventStartO, applyToFuture, true, ai);
    }

    private ModificationInfo deleteEvent(UserOrYaCalendar userOrYaCalendar, IdOrExternalId eventId, Option<Integer> sequence,
                                         Option<Instant> eventStartO, boolean applyToFuture, boolean sendMails, ActionInfo ai) {
        Event event = getModificationEvent(userOrYaCalendar.getUidO(), eventId, eventStartO);
        EventAndRepetition eventAndRepetition = eventDbManager.getEventAndRepetitionByEvent(event);

        logEventIds(event.getId(), eventId.isExternal()
                ? eventId.getExternalId()
                : mainEventDao.findMainEventByEventId(eventId.getId()).getExternalId());

        Tuple2<ModificationInfo, ListF<Long>> result = lockTransactionManager.lockAndDoInTransactionWithPropagation(
                () -> updateLock2.lockEvent(event.getId()),
                () -> {
                    ListF<Long> affectedLayerIds = findMasterAndSingleEventsLayerIds(eventId);

                    ActionInfo actionInfo = ai.withNow(Instant.now());

                    Event e = getEventForUpdateCheckNotModified(event.getId(), sequence, eventStartO, false);
                    logger.info("Deleting event: " + e); // diag, e.g. for create details

                    if (e.getName().startsWith(SpecialNames.FAIL_ON_DELETE)) {
                        throw new RuntimeException(SpecialNames.FAIL_ON_DELETE);
                    }

                    Validate.isTrue(e.getType() != EventType.FEED,
                            "Ics feed event should not be deleted! Event id: " + eventId);
                    ModificationInfo info;

                    if (e.getType() == EventType.SERVICE) {
                        info = eventRoutines.deleteServiceEvent(
                                userOrYaCalendar.getUserInfo(), e, actionInfo.getActionSource());
                    } else {
                        info = eventRemover.remove(
                                userOrYaCalendar, event.getId(), eventStartO, applyToFuture, sendMails, actionInfo);
                    }
                    return Tuple2.tuple(info, affectedLayerIds);
                });

        xivaNotificationManager.notifyLayersUsersAboutEventsChangeAsynchronously(result.get2(), ai);
        if (userOrYaCalendar.isUser()) {
            xivaMobileNotifier
                    .notify(eventAndRepetition, userOrYaCalendar.getUserInfo().getUid(), Optional.of(result.get1()), ai);
        }
        return result.get1();
    }

    public ModificationInfo deleteEventResource(
            UserInfo user, String exchangeName, long eventId, int sequence,
            Option<Instant> eventStart, boolean applyToFuture, ActionInfo actionInfo)
    {
        Event event = getModificationEvent(Option.of(user.getUid()), IdOrExternalId.id(eventId), eventStart);
        EventAndRepetition eventAndRepetition = eventDbManager.getEventAndRepetitionByEvent(event);

        Option<Email> email = resourceRoutines.findByExchangeName(exchangeName).map(ResourceRoutines::getResourceEmail);

        EventData data = new EventData();
        data.setInstanceStartTs(eventStart);

        data.setEvent(event.copy());
        eventAndRepetition.getRepetitionInfo().getRepetition().forEach(r -> data.setRepetition(r.copy()));

        data.setInvData(new EventInvitationUpdateData(Cf.list(), Cf.list(email.get())));

        return update(user, data, NotificationsData.notChanged(),
                Option.empty(), Option.of(sequence), applyToFuture, Option.empty(), actionInfo);
    }

    public ModificationInfo deleteFutureEvents(UserOrYaCalendar userOrYaCalendar, long eventId, ActionInfo actionInfo) {
        ListF<Long> eventIds = eventRoutines.findMasterAndSingleEventIds(eventId);
        ListF<EventAndRepetition> events = eventDbManager.getEventsAndRepetitionsByEventIds(eventIds);

        Option<Tuple2<EventAndRepetition, Instant>> futureEventAndStart = events
                .zipWithFlatMapO(EventAndRepetition.getFirstInstanceStartAfterF(actionInfo.getNow()))
                .minO(Tuple2.<EventAndRepetition, Instant>get2F().andThenNaturalComparator());

        if (futureEventAndStart.isPresent()) {
            long futureEventId = futureEventAndStart.get().get1().getEventId();
            Instant firstStart = futureEventAndStart.get().get2();

            return deleteEvent(
                    userOrYaCalendar, IdOrExternalId.id(futureEventId), Option.empty(),
                    Option.of(firstStart), true, actionInfo);
        } else {
            return ModificationInfo.removed(ModificationScope.NONE, Cf.list(), Cf.list(), Cf.list());
        }
    }

    // Delete user, feed and ext_meeting events
    public ModificationInfo deleteUserEvent(
            UserInfo user, long eventId, Option<Instant> eventInstanceStartO, boolean applyToFuture, ActionInfo actionInfo)
    {
        checkEventExists(eventId);
        return eventRemover.remove(user, eventId, eventInstanceStartO, applyToFuture, actionInfo);
    }

    public void handleEventInvitationDecision(
            PassportUid uid, long eventId, WebReplyData reply, ActionInfo actionInfo)
    {
        handleEventInvitationDecision(
                uid, getEventInvitation(eventId, uid),
                Option.empty(), false, reply, actionInfo);
    }

    public void handleEventInvitationDecision(
            PassportUid uid, long eventId, Instant eventInstanceStart, WebReplyData reply, ActionInfo actionInfo)
    {
        handleEventInvitationDecision(
                uid, getEventInvitation(eventId, uid),
                Option.of(eventInstanceStart), false, reply, actionInfo);
    }

    public void handleEventInvitationDecision(
            PassportUid uid, long eventId, Option<String> privateToken,
            Option<Instant> eventInstanceStart, boolean forceApplyToAll,
            WebReplyData reply, ActionInfo actionInfo)
    {
        handleEventInvitationDecision(
                uid, getEventInvitation(eventId, uid, privateToken),
                eventInstanceStart, forceApplyToAll, reply, actionInfo);
    }

    public void handleEventInvitationDecision(
            final PassportUid uid, final UserParticipantInfo invitation,
            final Option<Instant> eventInstanceStart, final boolean forceApplyToAll,
            final WebReplyData reply, final ActionInfo ai)
    {
        ListF<Long> affectedLayerIds = lockTransactionManager.lockAndDoInTransaction(
                () -> updateLock2.lockEvent(invitation.getEventId()),
                () -> {
                    ListF<Long> affLayerIds = eventRoutines.findMasterAndSingleEventsLayerIds(invitation.getEventId());

                    ActionInfo actionInfo = ai.withNow(Instant.now());

                    Event event = getEventForUpdateCheckNotModified(
                            invitation.getEventId(), Option.<Integer>empty(), eventInstanceStart, false);

                    final EventsAttachedUser attach;

                    if (invitation.getId().isYandexUser() && reply.getDecision() == Decision.NO
                        && !event.getRecurrenceId().isPresent() && event.getRepetitionId().isPresent()
                        && eventInstanceStart.isPresent() && !forceApplyToAll)
                    {
                        long recurrenceEventId = eventRoutines.createNotChangedRecurrence(
                                invitation.getUid().get(), invitation.getEventId(),
                                eventInstanceStart.get(), actionInfo);

                        UserParticipantInfo recurrenceInvitation = eventInvitationManager
                                .getParticipantByEventIdAndParticipantId(recurrenceEventId, invitation.getId())
                                .filterByType(UserParticipantInfo.class).single();

                        attach = eventInvitationManager.handleEventInvitationDecision(
                                uid, recurrenceInvitation, reply, false, actionInfo);
                    } else {
                        boolean applyToAll = event.getRepetitionId().isPresent() || forceApplyToAll;

                        attach = eventInvitationManager.handleEventInvitationDecision(
                                uid, invitation, reply, applyToAll, actionInfo);
                    }
                    logAttach(userManager.getUserInfo(uid),
                            mainEventDao.findExternalIdByMainEventId(event.getMainEventId()), attach, actionInfo);

                    sendEventsAttachNotifyMails(uid, attach.getLayers(), actionInfo);

                    return affLayerIds;
                });

        affectedLayerIds = affectedLayerIds.plus(eventRoutines.findMasterAndSingleEventsLayerIds(invitation.getEventId()));
        xivaNotificationManager.notifyLayersUsersAboutEventsChangeAsynchronously(affectedLayerIds, ai);
    }

    public UserParticipantInfo getEventInvitation(long eventId, PassportUid uid) {
        return getEventInvitation(eventId, uid, Option.<String>empty());
    }

    public UserParticipantInfo getEventInvitation(long eventId, PassportUid uid, Option<String> privateToken) {
        checkEventExists(eventId);

        if (privateToken.isPresent()) {
            EventWithRelations event = eventDbManager.getEventWithRelationsById(eventId);

            return findEventInvitation(event, privateToken.get())
                    .getOrThrow(CommandRunException.createSituationF(
                            "invitation not found by token " + privateToken.get(), Situation.INV_IS_MISSING));
        } else {
            return findEventInvitation(eventId, uid)
                    .getOrThrow(CommandRunException.createSituationF(
                            "invitation not found by event id " + eventId + " and uid : " + uid,
                            Situation.INV_IS_MISSING));
        }
    }

    public Option<UserParticipantInfo> findEventInvitation(long eventId, PassportUid uid) {
        if (uid.isYandexTeamRu()) {
            return eventInvitationManager
                    .getParticipantByEventIdAndParticipantId(eventId, ParticipantId.yandexUid(uid))
                    .uncheckedCast();
        } else {
            return eventInvitationManager.getUserEventSharing(eventId, ParticipantId.yandexUid(uid));
        }
    }

    public Option<UserParticipantInfo> findEventInvitation(EventWithRelations event, String privateToken) {
        Option<UserParticipantInfo> participant = event.getParticipants().getByPrivateTokenSafe(privateToken);

        if (participant.isPresent()) return Option.of(participant.get());

        participant = eventInvitationManager.getUserParticipantByPrivateToken(privateToken);

        if (!participant.isPresent()) return Option.empty();

        long tokenMainEventId = eventDao.findMainEventIdByEventId(participant.get().getEventId()).get();
        ParticipantId participantId = participant.get().getId();

        if (tokenMainEventId == event.getMainEventId()) {
            return event.getParticipants().getByIdSafeWithInconsistent(participantId).uncheckedCast();
        }
        return Option.empty();
    }

    private void checkEventExists(long eventId) {
        if (!eventDao.findEventExistsById(eventId)) {
            throw CommandRunException.createSituation("event not found by id " + eventId, Situation.EVENT_NOT_FOUND);
        }
    }

    private Event getEvent(long eventId) {
        return eventDao.findEventsByIdsSafe(Cf.list(eventId)).singleO().getOrThrow(() ->
                CommandRunException.createSituation("event not found by id " + eventId, Situation.EVENT_NOT_FOUND));
    }

    private void checkEventExistsAndNotModified(long eventId, Option<Integer> sequence, Option<Instant> instanceStart) {
        getEventCheckNotModified(eventId, sequence, instanceStart, false, false);
    }

    private Event getEventForUpdateCheckNotModified(
            long eventId, Option<Integer> sequence, Option<Instant> instanceStart, boolean preferMaster)
    {
        return getEventCheckNotModified(eventId, sequence, instanceStart, preferMaster, true);
    }

    private Event getEventCheckNotModified(
            long eventId, Option<Integer> sequence, Option<Instant> instanceStart,
            boolean preferMaster, boolean lockForUpdate)
    {
        Option<Event> event = eventDao.findEventsByIdsSafe(Cf.list(eventId), lockForUpdate).singleO();
        if (!event.isPresent()) {
            throw CommandRunException.createSituation("event not found by id " + eventId, Situation.EVENT_NOT_FOUND);
        }
        if (event.get().getRepetitionId().isPresent() && instanceStart.isPresent()) {
            checkRepeatingEventNotModified(event.get(), instanceStart.get());
        }
        if (event.get().getRecurrenceId().isPresent() && preferMaster) {
            long mainEventId = event.get().getMainEventId();

            event = Option.of(eventDao.findMasterEventByMainId(mainEventId, lockForUpdate).singleO()
                    .getOrThrow(CommandRunException.createSituationF(
                            "master event not found for main id " + mainEventId, Situation.EVENT_NOT_FOUND)));
        }
        if (sequence.isPresent() && sequence.get() < event.get().getSequence()) {
            throw CommandRunException.createSituation(
                    "event already modified " + event.get().getId(), Situation.EVENT_MODIFIED);
        }
        return event.get();
    }

    private void checkRepeatingEventNotModified(Event event, Instant instanceStart) {
        long eventId = event.getId();
        RepetitionInstanceInfo repetition = repetitionRoutines.getRepetitionInstanceInfoByEvent(event);

        if (repetition.getRecurIds().containsTs(instanceStart)) {
            throw CommandRunException.createSituation("event occurrence already modified: " +
                    "id " + eventId + ", recurrence id " + instanceStart, Situation.EVENT_MODIFIED);
        }
        if (!RepetitionUtils.isValidStart(repetition, instanceStart)) {
            throw CommandRunException.createSituation("not a valid start of event instance: " +
                    "id " + eventId + ", start " + instanceStart, Situation.EVENT_NOT_FOUND);
        }
    }

    public Event getModificationEvent(
            Option<PassportUid> uid, IdOrExternalId eventId, Option<Instant> instanceStart)
    {
        if (!eventId.isExternal()) {
            return eventDao.findEventsByIdsSafe(Cf.list(eventId.getId())).singleO()
                    .getOrThrow(CommandRunException.createSituationF(
                            "event not found by id " + eventId.getId(), Situation.EVENT_NOT_FOUND));
        }
        Validate.some(uid);

        String externalId = eventId.getExternalId();
        Option<MainEvent> main = eventRoutines.getMainEventBySubjectAndExternalId(
                UidOrResourceId.user(uid.get()), externalId);

        if (!main.isPresent()) {
            throw CommandRunException.createSituation(
                    "main event with external id " + externalId + " not found for user " + uid,
                    Situation.EVENT_NOT_FOUND);
        }
        Option<Event> event = Option.empty();

        if (instanceStart.isPresent()) {
            ListF<Event> events = eventDao.findEvents(SqlCondition.all(
                    EventFields.MAIN_EVENT_ID.eq(main.get().getId()),
                    EventFields.RECURRENCE_ID.column().isNotNull(),
                    EventFields.START_TS.column().eq(instanceStart.get())));

            event = events.singleO();
        }
        if (!event.isPresent()) {
            event = eventRoutines.findMasterEventByMainId(main.get().getId());

            if (event.isPresent() && event.get().getRepetitionId().isPresent() && instanceStart.isPresent()) {
                checkRepeatingEventNotModified(event.get(), instanceStart.get());
            }
        }
        return event.getOrThrow(CommandRunException.createSituationF(
                "event with external id " + externalId + " not found for user " + uid, Situation.EVENT_NOT_FOUND));
    }

    public ModificationItem getModificationItem(
            long eventId, Option<Instant> startO, Option<PassportUid> clientUid,
            Option<Long> layerIdO, boolean applyToFuture, ActionInfo actionInfo)
    {
        Option<Event> requestedEvent = Option.empty();
        Option<Event> masterEvent = Option.empty();

        if (applyToFuture) {
            requestedEvent = Option.of(eventDao.findEventById(eventId));

            if (requestedEvent.get().getRecurrenceId().isPresent()) {
                startO = Option.empty();
                masterEvent = eventRoutines.getMainInstOfEvent(requestedEvent.get());
            } else {
                masterEvent = requestedEvent;
            }
            eventId = masterEvent.orElse(requestedEvent).get().getId();
        }

        eventInvitationManager.fixParticipants(eventId, actionInfo);

        EventInstanceForUpdate instance = eventRoutines.getEventInstanceForModifier(
                clientUid, startO, eventId, layerIdO, actionInfo);

        if (requestedEvent.exists(e -> e.getRecurrenceId().isPresent())) {
            instance = instance.withIntervalInsteadOfRecurrence(requestedEvent.get().getRecurrenceId().get());
        }
        if (!masterEvent.isPresent()) {
            masterEvent = eventRoutines.getMainInstOfEvent(instance.getEvent());
        }
        return new ModificationItem(instance, masterEvent);
    }

    public ModificationSituation chooseModificationSituation(
            ModificationItem item, boolean applyToFuture, boolean wasCuttingChange, ActionInfo actionInfo)
    {
        if (!item.isRepeated()) {
            return ModificationSituation.SINGLE_EVENT;

        } else if (applyToFuture && wasCuttingChange && item.instanceStart().exists(actionInfo.getNow()::isBefore)) {
            RepetitionInstanceInfo repetition = item.getInstance().getRepetitionInstanceInfo();

            return repetition.hasActualOccurrenceIn(actionInfo.getNow(), item.instanceStart().get())
                    ? ModificationSituation.THIS_AND_FUTURE
                    : ModificationSituation.MAIN_INST_AND_FUTURE;

        } else if (applyToFuture) {
            return ModificationSituation.MAIN_INST_AND_FUTURE;
        } else {
            if (item.isRecurrence()) {
                return ModificationSituation.RECURRENCE_INST;
            } else if (!item.getInstance().getInterval().isPresent()) {
                return ModificationSituation.MAIN_INST_AND_FUTURE;
            } else {
                return ModificationSituation.SINGLE_INST;
            }
        }
    }

    // TODO: merge updateLite and update method in one
    // Same as update method, but operates time fields only (and related changes)
    public ModificationInfo updateLite(
            final UserInfo user, final EventData eventData,
            final Option<Integer> sequence, final ActionInfo ai)
    {
        final long eventId = eventData.getEvent().getId();
        EventAndRepetition eventAndRepetition = eventDbManager.getEventAndRepetitionByEvent(eventData.getEvent());

        authorizer.ensurePermittedOrganizerSetting(user, Optional.of(getEvent(eventId)), eventData.getInvData(), ai.getActionSource());

        logEventIds(eventId, mainEventDao.findMainEventByEventId(eventId).getExternalId());

        Tuple2<ModificationInfo, ListF<Long>> result = lockTransactionManager.lockAndDoInTransaction(
                () -> updateLock2.lockEvent(eventId),
                () -> {
                    ListF<Long> affectedLayerIds = eventRoutines.findMasterAndSingleEventsLayerIds(eventId);
                    ActionInfo actionInfo = ai.withNow(Instant.now());
                    Event e = getEventForUpdateCheckNotModified(eventId, sequence, eventData.getInstanceStartTs(), false);

                    return Tuple2.tuple(eventUpdater.update(
                        user, eventData, NotificationsData.notChanged(), false, actionInfo), affectedLayerIds);
                }
        );

        xivaNotificationManager.notifyLayersUsersAboutEventsChangeAsynchronously(result.get2(), ai);
        xivaMobileNotifier.notify(eventAndRepetition, user.getUid(), Optional.of(result.get1()), ai);
        return result.get1();
    }

    public void ensureCommonPerms(ModificationSituation modificationSituation, UserInfo clientInfo, ModificationItem item,
                                  ActionSource actionSource) {
        if (ModificationSituation.THIS_AND_FUTURE == modificationSituation) {
            val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(Optional.of(clientInfo), item.getInstance().getEventWithRelations(),
                Optional.empty(), Optional.empty());
            authorizer.ensureCanSplitEvent(clientInfo, eventAuthInfo, actionSource);
        }
    }

    public Option<Long> update(UserInfo user, EventData eventData, boolean applyToFuture, ActionInfo actionInfo) {
        return update(user, eventData, Option.empty(), applyToFuture, actionInfo).getNewEventId();
    }

    public ModificationInfo update(UserInfo user, EventData eventData, Option<Integer> sequence, boolean applyToFuture,
                                   ActionInfo actionInfo) {
        return update(user, eventData,
                NotificationsData.updateFromWeb(eventData.getEventUserData().getNotifications()),
                Option.empty(), sequence, applyToFuture, Option.empty(), actionInfo);
    }

    public ModificationInfo update(UserInfo user, EventData eventData, NotificationsData.Update notifications,
                                   Option<String> privateToken, Option<Integer> sequence, boolean applyToFuture,
                                   Option<Boolean> mailToAll, ActionInfo ai) {
        return update(user, eventData, notifications, privateToken, sequence, applyToFuture, mailToAll, false, ai);
    }

    public ModificationInfo update(UserInfo user, EventData eventData, NotificationsData.Update notifications,
                                   Option<String> privateToken, Option<Integer> sequence, boolean applyToFuture,
                                   Option<Boolean> mailToAll, boolean disableAllMails, ActionInfo ai) {
        IdOrExternalId eventId = IdOrExternalId.fromOptions(
                eventData.getEvent().getFieldValueO(EventFields.ID), eventData.getExternalId());

        Event event = getModificationEvent(Option.of(user.getUid()), eventId, eventData.getInstanceStartTs());
        EventAndRepetition eventAndRepetition = eventDbManager.getEventAndRepetitionByEvent(event);

        String externalId = eventId.isExternal()
                ? eventId.getExternalId()
                : mainEventDao.findMainEventByEventId(event.getId()).getExternalId();

        logEventIds(event.getId(), externalId);

        Tuple2<ModificationInfo, ListF<Long>> result = lockTransactionManager.lockAndDoInTransactionWithPropagation(
                () -> updateLock2.lockEvent(Option.of(externalId)),
                () -> {
                    val affectedLayerIds = findMasterAndSingleEventsLayerIds(eventId);

                    val actionInfo = ai.withNow(Instant.now());

                    val maybeMaster = getEventForUpdateCheckNotModified(
                            event.getId(), sequence, eventData.getInstanceStartTs(), applyToFuture);

                    authorizer.ensurePermittedOrganizerSetting(user, Optional.of(maybeMaster), eventData.getInvData(), actionInfo.getActionSource());

                    final var absence = eventData.getEvent().getFieldValueO(EventFields.TYPE)
                            .filterMap(t -> Option.when(t.isAbsence(), t::toAbsenceType));
                    if (absence.isPresent() && !eventData.getLayerId().isPresent()) {
                        eventData.setLayerId(layerRoutines.getOrCreateAbsenceLayer(user.getUid(), absence.get()));
                    }
                    val invitationMode = absence.isPresent()
                            ? InvitationProcessingMode.SAVE_ATTACH
                            : InvitationProcessingMode.SAVE_ATTACH_SEND;

                    EventsAttachedUser attach = EventsAttachedUser.empty();

                    if (privateToken.isPresent()) {
                        attach = addUserToParticipantsIfNeeded(
                                user.getUid(), maybeMaster, eventData, applyToFuture, privateToken.get(), actionInfo);

                        logAttach(user, externalId, attach, actionInfo);
                    }
                    eventData.getEvent().setId(event.getId());

                    return Tuple2.tuple(eventUpdater.update(
                            user, eventData, notifications, attach.getLayers(), applyToFuture,
                            invitationMode, mailToAll, disableAllMails, actionInfo, true), affectedLayerIds);
                });

        val affectedLayerIds = result.get2().plus(eventRoutines.findMasterAndSingleEventsLayerIds(event.getId()));
        xivaNotificationManager.notifyLayersUsersAboutEventsChangeAsynchronously(affectedLayerIds, ai);
        xivaMobileNotifier.notify(eventAndRepetition, user.getUid(), Optional.of(result.get1()), ai);

        return result.get1();
    }

    public MoveResourceEventsIds moveResource(UserInfo user, EventData sourceEventData, EventData targetEventData, ActionInfo ai) {
        val sourceEventId = sourceEventData.getEvent().getId();
        val targetEventId = targetEventData.getEvent().getId();

        val sourceExternalId = mainEventDao.findMainEventByEventId(sourceEventId).getExternalId();
        logEventIds(sourceEventId, sourceExternalId);

        val targetExternalId = mainEventDao.findMainEventByEventId(targetEventId).getExternalId();
        logEventIds(targetEventId, targetExternalId);

        val affectedLayerIds = Cf.<Long>arrayList();

        final var result = lockTransactionManager.lockAndDoInTransaction(
                () -> updateLock2.lockForUpdate(Cf.list(sourceExternalId, targetExternalId).map(LockResource::event)),
                () -> {
                    affectedLayerIds.addAll(findMasterAndSingleEventsLayerIds(IdOrExternalId.id(sourceEventId)));
                    affectedLayerIds.addAll(findMasterAndSingleEventsLayerIds(IdOrExternalId.id(targetEventId)));

                    val actionInfo = ai.withNow(Instant.now());

                    val sourceId = updateEventAndGetId(user, sourceEventData, sourceEventId, actionInfo);
                    val targetId = updateEventAndGetId(user, targetEventData, targetEventId, actionInfo);

                    return new MoveResourceEventsIds(sourceId, targetId);
                });

        xivaNotificationManager.notifyLayersUsersAboutEventsChangeAsynchronously(affectedLayerIds, ai);

        return result;
    }

    private void logEventIds(Long eventId, String sourceExternalId) {
        logger.info(LogMarker.EVENT_ID.format(eventId));
        logger.info(LogMarker.EXT_ID.format(sourceExternalId));
    }

    private Long updateEventAndGetId(UserInfo user, EventData eventData, Long eventId, ActionInfo actionInfo) {
        val maybeMaster = getEventForUpdateCheckNotModified(eventId, Option.empty(), eventData.getInstanceStartTs(), false);
        authorizer.ensurePermittedOrganizerSetting(user, Optional.of(maybeMaster), eventData.getInvData(), actionInfo.getActionSource());
        val modificationResult = eventUpdater.update(
                user, eventData, NotificationsData.notChanged(), EventsAttachedUser.empty().getLayers(), false,
                InvitationProcessingMode.SAVE_ATTACH_SEND, Option.empty(), actionInfo, false);
        val eventAndRepetition = modificationResult.getNewEvent().getOrElse(modificationResult.getUpdatedEvent()::get);
        return eventAndRepetition.getEvent().getId();
    }

    private EventsAttachedUser addUserToParticipantsIfNeeded(
            PassportUid uid, Event event, EventData eventData, boolean applyToFuture,
            String privateToken, ActionInfo actionInfo)
    {
        boolean applyToAll = event.getRepetitionId().isPresent() || applyToFuture;

        long masterId = event.getRecurrenceId().isPresent()
                ? eventRoutines.findMasterEventByMainId(event.getMainEventId()).get().getId()
                : event.getId();

        long eventIdToAttach = event.getRecurrenceId().isPresent() && applyToFuture ? masterId : event.getId();

        eventData.setInvData(eventData.getInvData().plusEmail(userManager.getEmailByUid(uid).get()));

        if (!findEventInvitation(eventIdToAttach, uid).isPresent()) {
            EventWithRelations eventToAttach = eventDbManager.getEventWithRelationsById(eventIdToAttach);
            Option<UserParticipantInfo> tokenInvitation = findEventInvitation(eventToAttach, privateToken);

            if (!tokenInvitation.isPresent()) {
                return EventsAttachedUser.empty();
            }
            WebReplyData reply = new WebReplyData(
                    Decision.MAYBE, eventData.getLayerId(),
                    NotificationsData.createFromWeb(eventData.getEventUserData().getNotifications()));

            return eventInvitationManager.handleEventInvitationDecision(
                    uid, tokenInvitation.get(), reply, applyToAll, actionInfo);
        }
        return EventsAttachedUser.empty();
    }

    public CreateInfo createUserEvent(
            PassportUid uid, EventData eventData,
            InvitationProcessingMode invitationMode, ActionInfo actionInfo)
    {
        NotificationsData.Create notifications = NotificationsData.createFromWeb(eventData.getEventUserData().getNotifications());
        return createUserEvent(uid, eventData, notifications, invitationMode, actionInfo);
    }

    public CreateInfo createUserEvent(
            final PassportUid uid, final EventData eventData, final NotificationsData.Create notifications,
            final InvitationProcessingMode invitationMode, final ActionInfo actionInfo)
    {
        authorizer.ensurePermittedOrganizerSetting(userManager.getUserInfo(uid), Optional.empty(), eventData.getInvData(), actionInfo.getActionSource());

        if (!eventData.getExternalId().isPresent()) {
            eventData.setExternalId(Option.of(CalendarUtils.generateExternalId()));
        }
        CreateInfo info = lockTransactionManager.lockAndDoInTransaction(
                () -> updateLock2.lockEvent(eventData.getExternalId()),
                () -> {
                    Option<String> extId = eventData.getExternalId();
                    UidOrResourceId userId = UidOrResourceId.user(uid);

                    if (extId.isPresent()
                        && eventRoutines.getMainEventBySubjectAndExternalId(userId, extId.get()).isPresent())
                    {
                        throw CommandRunException.createSituation(
                                "event with external id " + extId.get() + " already exists for user " + uid,
                                Situation.EVENT_ALREADY_EXISTS);
                    }
                    long mainEventId = eventRoutines.createMainEvent(uid, eventData, actionInfo);

                    return eventRoutines.createUserOrFeedEvent(
                            UidOrResourceId.user(uid),
                            eventData.getEvent().getFieldValueO(EventFields.TYPE).getOrElse(EventType.USER),
                            mainEventId, eventData,
                            notifications, invitationMode, actionInfo);
                });

        xivaNotificationManager.notifyLayersUsersAboutEventsChangeAsynchronously(info.getAllLayerIds(), actionInfo);
        xivaMobileNotifier.notify(info.getEventAndRepetition(), uid, Optional.empty(), actionInfo);
        return info;
    }

    public void attachEvent(
            final UserInfo user, final long eventId, Option<Long> layerId,
            final EventUser eventUserData, final NotificationsData.Create notificationsData, final ActionInfo actionInfo)
    {
        EventsAttachedLayerIds result = lockTransactionManager.lockAndDoInTransaction(
                () -> updateLock2.lockEvent(eventId),
                () -> {
                    EventWithRelations event = eventDbManager.getEventWithRelationsById(eventId);

                    val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(user, event);
                    authorizer.ensureCanViewEvent(user, eventAuthInfo, actionInfo.getActionSource());

                    EventsAttachedUser attach = eventRoutines.attachEventOrMeetingsToUserByMainEventId(
                            user, event.getMainEventId(), layerId,
                            eventUserData, notificationsData, actionInfo);

                    EventsAttachedLayerIds layerIds = attach.getLayers();

                    logAttach(user, event.getExternalId(), attach, actionInfo);

                    if (layerIds.getEvents().exists(t -> t.get2().getAttachedId().isPresent())) {
                        eventInvitationManager.createAndSendEventInvitationMailIfOutlooker(
                                user.getUid(), event.getEvent(), actionInfo);
                    }
                    sendEventsAttachNotifyMails(user.getUid(), layerIds, actionInfo);
                    return layerIds;
                });

        xivaNotificationManager.notifyLayersUsersAboutEventsChangeAsynchronously(result.getAllLayerIds(), actionInfo);
    }

    public void detachEvent(UserInfo user, long eventId, Option<Long> layerIdO, ActionInfo actionInfo) {
        MainEvent mainEvent = mainEventDao.findMainEventByEventId(eventId);

        EventsAttachedLayerIds result = lockTransactionManager.lockAndDoInTransaction(
                () -> updateLock2.lockEvent(Option.of(mainEvent.getExternalId())),
                () -> {
                    long layerId;

                    if (!layerIdO.isPresent()) {
                        val ownedLayerIds = Cf.toList(layerRoutines.findOwnLayerIds(user.getUid()));
                        ListF<EventLayerId> els = eventLayerDao.findEventLayerEventIdsAndLayerIds(Option.of(eventId), ownedLayerIds);

                        if (els.isEmpty()) {
                            throw CommandRunException.createSituation(
                                    "Event not found at user layer " + eventId, Situation.EVENT_NOT_FOUND);
                        }
                        layerId = els.first().getLayerId();
                    } else {
                        layerId = layerIdO.get();
                    }

                    EventsAttachedUser detach = eventRoutines.detachEventsFromLayerByMainEventId(user, mainEvent,
                        layerId, actionInfo);

                    EventsAttachedLayerIds layerIds = detach.getLayers();

                    logAttach(user, mainEvent.getExternalId(), detach, actionInfo);

                    sendEventsAttachNotifyMails(user.getUid(), layerIds, actionInfo);
                    return layerIds;
                });

        xivaNotificationManager.notifyLayersUsersAboutEventsChangeAsynchronously(result.getAllLayerIds(), actionInfo);
    }

    private void sendEventsAttachNotifyMails(PassportUid uid, EventsAttachedLayerIds layerIds, ActionInfo actionInfo) {
        eventInvitationManager.sendEventMails(
                eventsOnLayerChangeHandler.handleEventsAttach(uid, layerIds, actionInfo), actionInfo);
    }

    public Option<DateTimeZone> getEventTimezoneIfExists(Option<Long> eventId) {
        if (!eventId.isPresent()) return Option.empty();

        Option<Event> event = eventDbManager.getEventByIdSafe(eventId.get());
        return eventRoutines.getEventsTimeZones(event).get2().singleO();
    }

    public Option<DateTimeZone> getEventTimezoneIfExists(PassportUid uid, IdOrExternalId eventId) {
        if (!eventId.isExternal()) return getEventTimezoneIfExists(Option.of(eventId.getId()));

        return eventRoutines.getMainEventBySubjectAndExternalId(UidOrResourceId.user(uid), eventId.getExternalId())
                .map(MainEvent.getTimezoneIdF().andThen(AuxDateTime.getVerifyDateTimeZoneF()));
    }

    private ListF<Long> findMasterAndSingleEventsLayerIds(IdOrExternalId eventId) {
        return eventId.isExternal()
                ? eventRoutines.findLayerIdsByEventExternalId(eventId.getExternalId())
                : eventRoutines.findMasterAndSingleEventsLayerIds(eventId.getId());
    }

    private void logAttach(UserInfo user, String externalId, EventsAttachedUser attach, ActionInfo actionInfo) {
        Email userEmail = settingsRoutines.getSettingsByUid(attach.uid.getOrElse(user.getUid())).getEmail();

        attach.events.forEach(e -> eventsLogger.log(EventChangeLogEvents.updated(
                ActorId.user(user), new EventIdLogDataJson(externalId, e.event),
                EventChangesJson.empty()
                        .withEvent(e.eventChanges)
                        .withUsersAndLayers(UserRelatedChangesJson.find(userEmail, e))), actionInfo));
    }

    public static ListF<EventMessageParameters> mergeMasterAndRecurrencesMails(ListF<EventMessageParameters> mails) {
        ListF<EventMessageParameters> masterMails = mails.filter(m -> !m.getOccurrenceId().isPresent());
        ListF<EventMessageParameters> recurrencesMails = mails.filter(m -> m.getOccurrenceId().isPresent());

        Function<EventMessageParameters, Option<Email>> withIcsRecipient = p ->
                Option.when(p.getIcs().isPresent(), p.getRecipientEmail());

        Function<EventMessageParameters, Option<Email>> withNoIcsRecipient = p ->
                Option.when(!p.getIcs().isPresent(), p.getRecipientEmail());

        return masterMails.plus(Cf.list(withIcsRecipient, withNoIcsRecipient).flatMap(f -> {
            SetF<Email> recipients = masterMails.filterMap(f).unique();

            return recurrencesMails.filter(m -> f.apply(m).isMatch(recipients.containsF().notF()));
        }));
    }
}
