package ru.yandex.calendar.logic.event.web;

import java.util.Optional;

import lombok.val;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.ews.exp.EwsExportRoutines;
import ru.yandex.calendar.frontend.ews.exp.OccurrenceId;
import ru.yandex.calendar.logic.LastUpdateManager;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActorId;
import ru.yandex.calendar.logic.event.EventChangesInfo.EventChangesInfoFactory;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventDeletionSmsHandler;
import ru.yandex.calendar.logic.event.EventGetProps;
import ru.yandex.calendar.logic.event.EventInfo;
import ru.yandex.calendar.logic.event.EventInfoDbLoader;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.EventsOnLayerChangeHandler;
import ru.yandex.calendar.logic.event.ModificationInfo;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.meeting.CancelMeetingHandler;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.ics.exp.EventInstanceParameters;
import ru.yandex.calendar.logic.log.EventIdLogDataJson;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.log.change.EventChangeLogEvents;
import ru.yandex.calendar.logic.log.change.EventChangesJson;
import ru.yandex.calendar.logic.log.change.changes.EventFieldsChangesJson;
import ru.yandex.calendar.logic.log.change.changes.RepetitionChangesJson;
import ru.yandex.calendar.logic.notification.NotificationRoutines;
import ru.yandex.calendar.logic.sending.EventSendingInfo;
import ru.yandex.calendar.logic.sending.param.EventMessageParameters;
import ru.yandex.calendar.logic.sending.param.EventOnLayerChangeMessageParameters;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;

public class EventWebRemover {
    private static final Logger logger = LoggerFactory.getLogger(EventWebRemover.class);

    @Autowired
    private NotificationRoutines notificationRoutines;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private EventWebManager eventWebManager;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EwsExportRoutines ewsExportRoutines;
    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private EventDbManager eventDbManager;

    @Autowired
    private CancelMeetingHandler cancelMeetingHandler;
    @Autowired
    private EventInfoDbLoader eventInfoDbLoader;
    @Autowired
    private LastUpdateManager lastUpdateManager;
    @Autowired
    private EventsOnLayerChangeHandler eventsOnLayerChangeHandler;
    @Autowired
    private EventDeletionSmsHandler eventDeletionSmsHandler;
    @Autowired
    private EventsLogger eventsLogger;

    private ListF<Long> modifyMainInstanceOfRepeatingAndFuture(UserOrYaCalendar user, ModificationItem item, ActionInfo actionInfo) {
        ListF<Long> eventIds = eventDao.findEventIdsByMainEventId(item.event().getMainEventId());
        eventRoutines.deleteEventsFromDbAndExchange(user.toActorId(), eventIds, actionInfo, true);
        return eventIds;
    }

    private ListF<Long> modifyRecurrenceInstanceOfRepeatingEvent(UserOrYaCalendar user, ModificationItem item, ActionInfo actionInfo) {
        // separate instance of the event
        long eId = item.event().getId();
        eventRoutines.deleteEventFromDbAndExchange(user.toActorId(), eId, actionInfo);
        // add exdate instead of recurrence instance
        if (item.master.isPresent()) {
            Long mainEId = item.master.get().getId();

            Rdate rdate = new Rdate();
            rdate.setIsRdate(false);
            rdate.setStartTs(item.event().getRecurrenceId().getOrNull());
            rdate.setEventId(mainEId);

            repetitionRoutines.createRdates(Cf.list(rdate), actionInfo);

            modifyInstanceCommon(user, item, item.event().getRecurrenceId(), actionInfo);
        }
        return Cf.list(eId);
    }

    private ListF<Long> modifySingleEvent(UserOrYaCalendar user, ModificationItem item, ActionInfo actionInfo) {
        long eId = item.event().getId();
        eventRoutines.deleteEventFromDbAndExchange(user.toActorId(), eId, actionInfo);
        return Cf.list(eId);
    }

    private ListF<Long> modifySingleInstanceOfRepeatingEvent(UserOrYaCalendar user, ModificationItem item, ActionInfo actionInfo) {
        // repeated by RDATE or RRULE instance of the event
        // try to remove RDATE if exists
        long eventId = item.master.get().getId();
        Instant eventInstanceStartTs = item.instanceStart().get();
        eventDao.deleteRdateBy(true, eventId, eventInstanceStartTs);

        Option<Instant> exdate = Option.empty();

        // add EXDATE to main event instance, if it has repetition
        if (item.master.get().getRepetitionId().isPresent()) {
            Rdate rdate = new Rdate();
            rdate.setIsRdate(false);
            rdate.setStartTs(eventInstanceStartTs);
            rdate.setEventId(eventId);

            repetitionRoutines.createRdates(Cf.list(rdate), actionInfo);

            exdate = Option.of(eventInstanceStartTs);
        }
        notificationRoutines.recalcAllNextSendTs(eventId, actionInfo);
        eventRoutines.invalidateResourceScheduleCachesOnDelete(item.instance.getEventWithRelations());

        modifyInstanceCommon(user, item, exdate, actionInfo);

        return Cf.list(item.master.get().getId());
    }

    private void modifyInstanceCommon(
            UserOrYaCalendar user, ModificationItem item, Option<Instant> exdate, ActionInfo actionInfo)
    {
        Event updatedMaster = eventDbManager.incrementEventSequenceAndModificationInfo(item.master.get(), actionInfo);

        removeOccurrenceInExchange(item, actionInfo);

        eventsLogger.log(EventChangeLogEvents.updated(user.toActorId(),
                new EventIdLogDataJson(item.instance.getEventWithRelations().getExternalId(), item.master.get()),
                EventChangesJson.empty()
                        .withEvent(EventFieldsChangesJson.of(item.master.toOptional(), updatedMaster))
                        .withRepetition(exdate.map(RepetitionChangesJson::exdateAdded).getOrElse(RepetitionChangesJson::empty))),
                actionInfo);
    }

    private ListF<Long> applyChange(UserOrYaCalendar user, ModificationSituation modificationSituation, ModificationItem item, ActionInfo actionInfo) {
        switch (modificationSituation) {
        case SINGLE_EVENT:
            return modifySingleEvent(user, item, actionInfo);
        case MAIN_INST_AND_FUTURE:
            return modifyMainInstanceOfRepeatingAndFuture(user, item, actionInfo);
        case SINGLE_INST:
            return modifySingleInstanceOfRepeatingEvent(user, item, actionInfo);
        case RECURRENCE_INST:
            return modifyRecurrenceInstanceOfRepeatingEvent(user, item, actionInfo);
        default:
            throw new IllegalStateException();
        }
    }

    private EventInstanceParameters createEventInstanceParameters(ModificationItem item, ModificationSituation situation) {
        Event event = item.event();
        Instant instStartMs = item.instanceStart().getOrElse(event.getStartTs());
        Instant eInstEndMs = repetitionRoutines.calcEndTs(event, instStartMs);
        Option<Instant> occurrenceId = Option.of(instStartMs);
        return new EventInstanceParameters(instStartMs, eInstEndMs, occurrenceId);
    }

    private ListF<EventMessageParameters> preProcessInv(
            ModificationSituation modificationSituation, ActorId clientId, ModificationItem item, ActionInfo actionInfo)
    {
        Participants participants = item.instance.getEventWithRelations().getParticipants();
        if (participants.isExtMeeting() && clientId.isUser()) {
            try {
                return Cf.<EventMessageParameters>list(eventInvitationManager.createRejectMail(
                        clientId.getUid(), item.instance.toEventInfo(), actionInfo.getNow()));

            } catch (Exception e) {
                logger.error("Failed to create reply mail: " + e, e);
            }
        } else if (EventType.USER == item.event().getType()) {
            ListF<EventMessageParameters> recurrenceMails = Cf.list();

            if (modificationSituation == ModificationSituation.MAIN_INST_AND_FUTURE) {
                recurrenceMails = createCancelEmailsForFutureRecurrences(clientId, item, actionInfo);
            }

            EventWithRelations event = item.instance.getEventWithRelations();

            RepetitionInstanceInfo repetition = item.instance.getRepetitionInstanceInfo();

            EventInstanceParameters instance = ModificationSituation.SINGLE_INST == modificationSituation
                    ? createEventInstanceParameters(item, modificationSituation)
                    : EventInstanceParameters.fromEvent(event.getEvent());

            ListF<EventMessageParameters> masterMails =
                    createCancelEmails(clientId, instance, event, repetition, actionInfo);

            if (recurrenceMails.isNotEmpty()) {
                return EventWebManager.mergeMasterAndRecurrencesMails(masterMails.plus(recurrenceMails));
            } else {
                return masterMails;
            }
        }
        return Cf.list();
    }

    public ListF<EventMessageParameters> createCancelEmailsForFutureRecurrences(
            ActorId clientId, ModificationItem item, ActionInfo actionInfo)
    {
        ListF<Event> recurrenceEvents = eventDao.findFutureRecurrencesForUpdate(
                item.event().getMainEventId(), actionInfo);

        ListF<EventInfo> recurrenceInfos = eventInfoDbLoader.getEventInfosByEvents(
                Option.empty(), EventGetProps.any(), recurrenceEvents, actionInfo.getActionSource());

        return recurrenceInfos.flatMap(e -> createCancelEmails(
                clientId, EventInstanceParameters.fromEvent(e.getEvent()),
                e.getEventWithRelations(), e.getRepetitionInstanceInfo(), actionInfo));
    }

    private ListF<EventMessageParameters> createCancelEmails(
            ActorId clientId, EventInstanceParameters instance,
            EventWithRelations updatedEvent, RepetitionInstanceInfo updatedRepetition, ActionInfo actionInfo)
    {
        boolean isExportedWithEws = updatedEvent.isExportedWithEws();

        ListF<EventSendingInfo> sendingInfoList = cancelMeetingHandler.cancelMeeting(
                updatedEvent.getEvent(), clientId.getUidO(), instance,
                updatedEvent.isParkingOrApartmentOccupation(), isExportedWithEws, actionInfo);

        ListF<EventMessageParameters> update = eventInvitationManager.createEventInvitationOrCancelMails(
                clientId, updatedEvent, updatedRepetition, sendingInfoList, actionInfo);

        ListF<EventOnLayerChangeMessageParameters> notify = eventsOnLayerChangeHandler.handleEventDelete(
                clientId, updatedEvent, updatedRepetition, instance, actionInfo);

        return update.plus(notify);
    }

    private void preProcessSmsNotifications(
            ModificationSituation situation, PassportUid clientId, ModificationItem item, ActionInfo actionInfo)
    {
        if (ModificationSituation.SINGLE_INST == situation) {
            eventDeletionSmsHandler.handleEventDeletion(
                    clientId, item.instance.getEventWithRelations(),
                    createEventInstanceParameters(item, situation), actionInfo);

        } else if (ModificationScope.SINGLE == situation.toDirection()) {
            eventDeletionSmsHandler.handleEventDeletion(
                    clientId, item.instance.getEventWithRelations(),
                    EventInstanceParameters.fromEvent(item.instance.getEvent()), actionInfo);

        } else {
            eventDeletionSmsHandler.handleEventClosestOccurrenceDeletion(
                    clientId, item.instance.getEvent().getMainEventId(), actionInfo);
        }
    }

    private void limitRepetitionInExchange(
            EventWithRelations updatedEvent, RepetitionInstanceInfo updatedRepetition, ActionInfo actionInfo)
    {
        EventChangesInfoFactory changesFactory = new EventChangesInfoFactory();
        changesFactory.setRepetitionChanges(updatedRepetition.getRepetition().get().<Repetition>withoutId());

        ewsExportRoutines.exportToExchangeIfNeededOnUpdate(
                updatedEvent, updatedRepetition, Option.empty(), changesFactory.create(), Option.empty(), actionInfo);
    }

    private void removeOccurrenceInExchange(ModificationItem item, ActionInfo actionInfo) {
        long eventId = item.master.get().getId();
        InstantInterval instanceInterval = item.getInstance().getInterval()
                .getOrElse(EventRoutines.getInstantInterval(item.getInstance().getEvent()));

        OccurrenceId occurrenceId = new OccurrenceId(eventDao.findExternalIdByEventId(eventId),
                instanceInterval, item.event().getRecurrenceId().orElse(item.instanceStart()).get());
        ewsExportRoutines.cancelOccurrenceIfNeeded(eventId, occurrenceId, actionInfo);
    }

    public ModificationInfo remove(
            UserInfo user, long eventId, Option<Instant> start, boolean applyToFuture, ActionInfo actionInfo)
    {
        return remove(UserOrYaCalendar.user(user), eventId, start, applyToFuture, actionInfo);
    }

    public ModificationInfo remove(UserOrYaCalendar userOrYaCalendar, long eventId, Option<Instant> startMsO,
                                   boolean applyToFuture, ActionInfo actionInfo) {
        return remove(userOrYaCalendar, eventId, startMsO, applyToFuture, true, actionInfo);
    }

    public ModificationInfo remove(UserOrYaCalendar userOrYaCalendar, long eventId, Option<Instant> startMsO,
                                   boolean applyToFuture, boolean sendMails, ActionInfo actionInfo) {
        Option<PassportUid> uid = userOrYaCalendar.getUidO();

        ModificationItem item = eventWebManager.getModificationItem(
                eventId, startMsO, uid, Option.empty(), applyToFuture, actionInfo);
        eventId = item.event().getId();
        ModificationSituation modificationSituation = eventWebManager.chooseModificationSituation(item, applyToFuture, false, actionInfo);

        if (userOrYaCalendar.isUser()) {
            eventWebManager.ensureCommonPerms(modificationSituation, userOrYaCalendar.getUserInfo(),
                    item, actionInfo.getActionSource());
            val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(userOrYaCalendar.getUserInfoO().toOptional(),
                item.getInstance().getEventWithRelations(), Optional.empty(), Optional.empty());
            authorizer.ensureCanDeleteEvent(userOrYaCalendar.getUserInfoO().get(), eventAuthInfo, actionInfo.getActionSource());
        }

        if (modificationSituation == ModificationSituation.SINGLE_INST) {
            lastUpdateManager.updateTimestampsAsync(item.event().getMainEventId(), actionInfo);
        } else {
            lastUpdateManager.updateMainEventAndLayerTimestamps(Cf.list(eventId), actionInfo);
        }

        ListF<EventMessageParameters> preProcessInvMails = preProcessInv(
                modificationSituation, userOrYaCalendar.toActorId(), item, actionInfo);

        if (uid.isPresent()) {
            preProcessSmsNotifications(modificationSituation, uid.get(), item, actionInfo);
        }

        ListF<Long> modifiedEventIds = applyChange(userOrYaCalendar, modificationSituation, item, actionInfo);

        if (sendMails) {
            // separate dsi statistics
            eventInvitationManager.sendEventMails(preProcessInvMails, actionInfo);
        }

        return ModificationInfo.removed(
                modificationSituation.toDirection(), modifiedEventIds, Cf.list(item.getExternalId()), preProcessInvMails);
    }
}
