package ru.yandex.calendar.logic.event.web;

import java.util.Set;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.CalendarUtils;
import ru.yandex.calendar.frontend.ews.exp.EwsExportRoutines;
import ru.yandex.calendar.frontend.ews.exp.OccurrenceId;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.frontend.web.cmd.run.ui.event.SpecialNames;
import ru.yandex.calendar.logic.LastUpdateManager;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.RepetitionFields;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.ActorId;
import ru.yandex.calendar.logic.event.ChangedEventInfoForMails;
import ru.yandex.calendar.logic.event.EventAndRelations;
import ru.yandex.calendar.logic.event.EventAttachedLayerId;
import ru.yandex.calendar.logic.event.EventChangesFinder;
import ru.yandex.calendar.logic.event.EventChangesInfo;
import ru.yandex.calendar.logic.event.EventChangesInfo.EventChangesInfoFactory;
import ru.yandex.calendar.logic.event.EventChangesInfoForMails;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventGetProps;
import ru.yandex.calendar.logic.event.EventInfo;
import ru.yandex.calendar.logic.event.EventInfoDbLoader;
import ru.yandex.calendar.logic.event.EventInstanceForUpdate;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventMoveSmsHandler;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventUserRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.EventsAttachedLayerIds;
import ru.yandex.calendar.logic.event.EventsOnLayerChangeHandler;
import ru.yandex.calendar.logic.event.LayerIdChangesInfo;
import ru.yandex.calendar.logic.event.ModificationInfo;
import ru.yandex.calendar.logic.event.attachment.EventAttachmentChangesInfo;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.EventLayerDao;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.event.meeting.CancelMeetingHandler;
import ru.yandex.calendar.logic.event.meeting.ExchangeMails;
import ru.yandex.calendar.logic.event.meeting.MeetingMailRecipients;
import ru.yandex.calendar.logic.event.meeting.UpdateMeetingHandler;
import ru.yandex.calendar.logic.event.meeting.UpdateMode;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.EventInstanceInterval;
import ru.yandex.calendar.logic.event.repetition.RecurrenceTimeInfo;
import ru.yandex.calendar.logic.event.repetition.RegularRepetitionRule;
import ru.yandex.calendar.logic.event.repetition.RepetitionConfirmationManager;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.ics.EventInstanceStatusInfo;
import ru.yandex.calendar.logic.ics.exp.EventInstanceParameters;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.log.change.EventChangeLogEvents;
import ru.yandex.calendar.logic.notification.NotificationRoutines;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.resource.ResourceAccessRoutines;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sending.EventSendingInfo;
import ru.yandex.calendar.logic.sending.param.EventMessageParameters;
import ru.yandex.calendar.logic.sending.param.EventOnLayerChangeMessageParameters;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.EventParticipantsChangesInfo;
import ru.yandex.calendar.logic.sharing.InvitationProcessingMode;
import ru.yandex.calendar.logic.sharing.MailType;
import ru.yandex.calendar.logic.sharing.ParticipantChangesInfo;
import ru.yandex.calendar.logic.sharing.participant.ParticipantData;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.UserParticipantInfo;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.telemost.TelemostManager;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.micro.perm.EventAction;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

@Slf4j
public class EventWebUpdater {
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private EventWebManager eventWebManager;
    @Autowired
    private NotificationRoutines notificationRoutines;
    @Autowired
    private EventUserRoutines eventUserRoutines;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private EventChangesFinder eventChangesFinder;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private UpdateMeetingHandler updateMeetingHandler;
    @Autowired
    private CancelMeetingHandler cancelMeetingHandler;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private EwsExportRoutines ewsExportRoutines;
    @Autowired
    private LastUpdateManager lastUpdateManager;
    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private RepetitionConfirmationManager repetitionConfirmationManager;
    @Autowired
    private EventsOnLayerChangeHandler eventsOnLayerChangeHandler;
    @Autowired
    private EventLayerDao eventLayerDao;
    @Autowired
    private EventMoveSmsHandler eventMoveSmsHandler;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private EventInfoDbLoader eventInfoDbLoader;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private ResourceAccessRoutines resourceAccessRoutines;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private EventWebRemover eventWebRemover;
    @Autowired
    private EventsLogger eventsLogger;
    @Value("#{'${experiments.access.tvm}'.split(',')}")
    private Set<Long> experimentAwareTvmIds;
    @Autowired
    private TelemostManager telemostManager;

    private static class Result {
        private final Option<Event> newEvent;
        private final ListF<Long> eventIdsToDelete;
        private final ListF<Long> modifiedEventIds;
        private final ListF<EventMessageParameters> recurrencesMails;

        private Result(
                Option<Event> newEvent, ListF<Long> eventIdsToDelete,
                ListF<Long> modifiedEventIds, ListF<EventMessageParameters> recurrencesMails)
        {
            this.newEvent = newEvent;
            this.eventIdsToDelete = eventIdsToDelete;
            this.modifiedEventIds = modifiedEventIds;
            this.recurrencesMails = recurrencesMails;
        }

        public Option<Long> getNewEventId() {
            return newEvent.map(EventFields.ID.getF());
        }

        public Result plusRecurrencesMails(ListF<EventMessageParameters> moreMails) {
            return new Result(newEvent, eventIdsToDelete, modifiedEventIds, recurrencesMails.plus(moreMails));
        }

        public Result plusModified(ListF<Long> moreModifiedIds) {
            return new Result(newEvent, eventIdsToDelete, modifiedEventIds.plus(moreModifiedIds), recurrencesMails);
        }

        public Result plusToDelete(ListF<Long> eventIds) {
            return new Result(newEvent, eventIdsToDelete.plus(eventIds), modifiedEventIds, recurrencesMails);
        }
    }

    private static class Events {
        private final Option<EventWithRelations> updatedEvent;
        private final Option<RepetitionInstanceInfo> updatedEventRepetition;
        private final Option<EventWithRelations> newEvent;
        private final Option<RepetitionInstanceInfo> newEventRepetition;
        private final Option<EventWithRelations> masterEventUpdated;
        private final Option<RepetitionInstanceInfo> masterEventUpdatedRepetition;

        private Events(
                Option<EventWithRelations> updatedEvent, Option<RepetitionInstanceInfo> updatedEventRepetition,
                Option<EventWithRelations> newEvent, Option<RepetitionInstanceInfo> newEventRepetition,
                Option<EventWithRelations> masterEventUpdated,
                Option<RepetitionInstanceInfo> masterEventUpdatedRepetition)
        {
            this.updatedEvent = updatedEvent;
            this.updatedEventRepetition = updatedEventRepetition;
            this.newEvent = newEvent;
            this.newEventRepetition = newEventRepetition;
            this.masterEventUpdated = masterEventUpdated;
            this.masterEventUpdatedRepetition = masterEventUpdatedRepetition;
        }

        public EventWithRelations newOrUpdatedEvent() {
            return newEvent.orElse(updatedEvent).get();
        }

        public RepetitionInstanceInfo newOrUpdatedEventRepetition() {
            return newEventRepetition.orElse(updatedEventRepetition).get();
        }

        public Option<EventAndRepetition> newEventAndRepetition() {
            return newEvent.isPresent()
                    ? Option.of(new EventAndRepetition(newEvent.get().getEvent(), newEventRepetition.get()))
                    : Option.empty();
        }

        public Option<EventAndRelations> newEventAndRelations() {
            return newEvent.map(e -> new EventAndRelations(e, newEventRepetition.get()));
        }

        public Option<EventAndRepetition> updatedEventAndRepetition() {
            return updatedEvent.map(e -> new EventAndRepetition(e.getEvent(), updatedEventRepetition.get()));
        }

        public Option<EventAndRelations> updatedEventAndRelations() {
            return updatedEvent.map(e -> new EventAndRelations(e, updatedEventRepetition.get()));
        }

        public Events withUpdatedEvent(EventWithRelations updatedEvent) {
            return new Events(Option.of(updatedEvent), updatedEventRepetition,
                    newEvent, newEventRepetition, masterEventUpdated, masterEventUpdatedRepetition);
        }
    }

    private static ListF<Long> reverseAttachedLayerIds(
            ListF<Long> currentLayerIds, Option<EventAttachedLayerId> preattached)
    {
        if (preattached.isPresent()) {
            return currentLayerIds
                    .plus(preattached.get().getDetachedId())
                    .filter(preattached.get().getAttachedId().containsF().notF());
        } else {
            return currentLayerIds;
        }
    }

    private class Do {
        private final EventChangesInfo eventChangesInfo;

        private final UserInfo client;
        private final PassportUid clientUid;
        private final ModificationItem item;
        private final EventData eventData;
        private final ListF<Long> masterLayerIdsBeforeUpdate;
        private final ListF<Long> instanceLayerIdsBeforeUpdate;
        private final ActionInfo actionInfo;
        private final ModificationSituation modificationSituation;
        private final InvitationProcessingMode invitationProcessingMode;
        private final Option<Boolean> mailToAll;
        private final boolean disableAllMails;
        private final UpdateMode updateMode;
        private final boolean exportToExchangeImmediately;

        private Do(
                UserInfo client, ModificationSituation modificationSituation,
                ModificationItem modificationItem, EventData eventData,
                EventChangesInfo eventChangesInfo, EventsAttachedLayerIds preattachedLayerIds,
                InvitationProcessingMode invitationProcessingMode,
                Option<Boolean> mailToAll, boolean disableAllMails, UpdateMode updateMode, ActionInfo actionInfo,
                boolean exportToExchangeImmediately)
        {
            this.client = client;
            this.clientUid = client.getUid();

            this.eventData = eventData;
            this.eventChangesInfo = eventChangesInfo;
            this.item = modificationItem;

            this.actionInfo = actionInfo;
            this.modificationSituation = modificationSituation;
            this.invitationProcessingMode = invitationProcessingMode;

            ListF<Long> masterLayerIds = item.isMaster()
                    ? item.instance.getEventWithRelations().getLayerIds()
                    : eventLayerDao.findLayersIdsByEventIds(item.master.map(Event.getIdF()));

            this.masterLayerIdsBeforeUpdate = reverseAttachedLayerIds(
                    masterLayerIds, preattachedLayerIds.findMasterEvent());

            if (!item.isMaster()) {
                this.instanceLayerIdsBeforeUpdate = reverseAttachedLayerIds(
                        item.instance.getEventWithRelations().getLayerIds(),
                        preattachedLayerIds.findByEventId(item.instance.getEventId()));
            } else {
                this.instanceLayerIdsBeforeUpdate = this.masterLayerIdsBeforeUpdate;
            }
            this.mailToAll = mailToAll;
            this.disableAllMails = disableAllMails;
            this.updateMode = updateMode;
            this.exportToExchangeImmediately = exportToExchangeImmediately;
        }

        private Result cloneEventForEwsRecreationAndExdate(Instant instanceStartTs, ActionInfo actionInfo) {
            Rdate rdate = new Rdate();
            rdate.setIsRdate(false);
            rdate.setStartTs(instanceStartTs);
            rdate.setEventId(item.master.get().getId());

            repetitionRoutines.createRdates(Cf.list(rdate), actionInfo);

            Event overrides = new Event();
            overrides.setRecurrenceIdNull();
            overrides.setRepetitionIdNull();
            overrides.setSequence(0);

            Result result = cloneEventWithMain(overrides, EventAttachmentChangesInfo.NO_CHANGES, actionInfo);

            ListF<Long> eventIds = result.getNewEventId().plus(item.master.get().getId());

            notificationRoutines.recalcAllNextSendTs(eventIds, actionInfo);
            lastUpdateManager.updateMainEventAndLayerTimestamps(eventIds, actionInfo);

            return result;
        }

        private Result cloneEventForEwsRecreation(ActionInfo actionInfo) {
            Event overrides = new Event();
            overrides.setSequence(0);

            Result result = cloneEventWithMain(overrides, EventAttachmentChangesInfo.NO_CHANGES, actionInfo);

            notificationRoutines.recalcAllNextSendTs(result.getNewEventId(), actionInfo);
            lastUpdateManager.updateMainEventAndLayerTimestamps(result.getNewEventId(), actionInfo);

            return result;
        }

        private Result cloneEventWithMain(
                Event eventDataOverrides,
                EventAttachmentChangesInfo attachmentChangesInfo,
                ActionInfo actionInfo) {
            Option<Long> newRepetitionIdO = Option.empty();

            Repetition repetitionChanges = eventChangesInfo.getRepetitionChanges();

            if (eventData.getRepetition().getType() != RegularRepetitionRule.NONE) {
                Option<Repetition> oldRepetition = item.getInstance().getRepetitionInstanceInfo().getRepetition();
                if (oldRepetition.isPresent()) {
                    Repetition cloneData = new Repetition();
                    cloneData.setFields(oldRepetition.get());
                    cloneData.setFields(repetitionChanges);
                    cloneData.unsetField(RepetitionFields.ID);
                    newRepetitionIdO = Option.of(repetitionRoutines.createRepetition(cloneData));
                } else {
                    newRepetitionIdO = Option.of(repetitionRoutines.createRepetition(repetitionChanges));
                }
            }

            String externalId = Cf.x(item.instance.getEventWithRelations().getExternalId().split("_")).lastO()
                    .map(s -> StringUtils.drop(s, s.length() - 72)) // suffix match is required for Mailer
                    .plus(CalendarUtils.generateExternalId()).mkString("_");

            eventDataOverrides.setFieldDefaults(eventChangesInfo.getEventChanges().getFieldValues());
            eventDataOverrides.setRepetitionId(newRepetitionIdO);

            Option<Boolean> orgIsEwser = item.instance.getEventWithRelations().getParticipants().getOrganizerSafe()
                    .filterMap(p -> p.getSettingsIfYandexUser().map(s -> s.getYt().exists(SettingsYt::getIsEwser)));

            boolean isExpEws = newOrganizerIsEwser().orElse(orgIsEwser).getOrElse(item.instance::isExportedWithEws);

            eventDataOverrides.setMainEventId(eventRoutines.createMainEvent(
                    externalId, item.instance.getEventWithRelations().getTimezone(), isExpEws, actionInfo));

            if (isExpEws) {
                eventDataOverrides.setSequence(0);
            }
            setInstanceStartEndIfNotSet(eventDataOverrides);

            return cloneEvent(eventDataOverrides, attachmentChangesInfo, actionInfo);
        }

        private Result cloneEvent(
                Event eventDataOverrides,
                EventAttachmentChangesInfo attachmentChangesInfo,
                ActionInfo actionInfo) {
            TelemostManager.PatchResult telemost = telemostManager.patchUpdateData(
                    eventDataOverrides, item.instance,
                    eventChangesInfo.getEventParticipantsChangesInfo(), updateMode);

            long newEventId = eventDbManager.cloneEventWithDependents(
                    item.getInstance().getEventWithRelations(),
                    item.getInstance().getAttachments(),
                    eventDataOverrides,
                    attachmentChangesInfo,
                    actionInfo);

            telemostManager.onEventUpdated(telemost, newEventId, actionInfo);

            if (item.instance.getEventUserWithNotifications().isPresent()) {
                long eventUserId = eventUserDao.findEventUserByEventIdAndUid(newEventId, clientUid).get().getId();

                eventUserRoutines.updateEventUserAndNotification(
                        eventUserId, eventChangesInfo.getEventUserChanges(),
                        eventChangesInfo.getNotificationsUpdate(), actionInfo);
            }
            if (item.instance.getEventLayer().isPresent() && eventChangesInfo.eventLayerChanges()) {
                eventRoutines.updateEventLayerByMainEventId(
                        client, eventDataOverrides.getMainEventId(),
                        item.instance.getEventLayer().get().getLayerId(),
                        eventChangesInfo.getNewLayerId().get(), actionInfo);
            }

            ListF<Long> modifiedEventIds = Cf.arrayList();
            modifiedEventIds.add(newEventId);
            modifiedEventIds.add(item.instance.getEvent().getId());

            return new Result(Option.of(eventDbManager.getEventById(newEventId)), Cf.list(), modifiedEventIds, Cf.list());
        }

        private Result applyChange() {
            switch (modificationSituation) {
            case SINGLE_EVENT:
                return modifySingleEvent();
            case MAIN_INST_AND_FUTURE:
                return modifyMainInstanceOfRepeatingAndFuture();
            case THIS_AND_FUTURE:
                return modifyThisAndFuture();
            case SINGLE_INST:
                return modifySingleInstanceOfRepeatingEvent();
            case RECURRENCE_INST:
                return modifyRecurrenceInstanceOfRepeatingEvent();
            default:
                throw new IllegalStateException();
            }
        }

        private Result modifyRecurrenceInstanceOfRepeatingEvent() {
            if (needsEwsRecreation()) {
                return cloneEventForEwsRecreationAndExdate(item.event().getRecurrenceId().get(), actionInfo)
                        .plusToDelete(Cf.list(item.event().getId()));
            }
            return updateCommon();
        }

        private Result modifySingleEvent() {
            if (needsEwsRecreation()) {
                return cloneEventForEwsRecreation(actionInfo)
                        .plusToDelete(Cf.list(item.event().getId()));
            }
            return updateCommon();
        }

        private Result modifySingleInstanceOfRepeatingEvent() {
            if (needsEwsRecreation()) {
                return cloneEventForEwsRecreationAndExdate(item.instanceStart().get(), actionInfo);
            }

            // apply overrides to clone event
            Event eventDataOverrides = eventChangesInfo.getEventChanges().copy();
            eventDataOverrides.setRecurrenceId(item.instanceStart());
            eventDataOverrides.setMainEventId(item.master.get().getMainEventId());

            setInstanceStartEndIfNotSet(eventDataOverrides);
            eventDataOverrides.setRepetitionIdNull();

            Result r = cloneEvent(eventDataOverrides, eventChangesInfo.getAttachmentChangesInfo(), actionInfo);
            long newEventId = r.getNewEventId().get();
            notificationRoutines.recalcNextSendTsForNewRecurrenceOrTail(item.event().getId(), newEventId, actionInfo);
            return r;
        }

        private void setInstanceStartEndIfNotSet(Event eventData) {
            InstantInterval interval = item.instance.getIntervalOrEventInterval();

            eventData.setFieldValueDefault(EventFields.START_TS, interval.getStart());
            eventData.setFieldValueDefault(EventFields.END_TS, interval.getEnd());
        }

        private ModificationInfo update() {
            EventWithRelations event = item.getInstance().getEventWithRelations();

            if (updateMode != UpdateMode.RECURRENCE_FOR_MASTER) {
                eventWebManager.ensureCommonPerms(modificationSituation, client, item, actionInfo.getActionSource());
                val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(client, event);
                authorizer.ensureCanViewEvent(client, eventAuthInfo, actionInfo.getActionSource());
            }

            val isRoomAdmin = authorizer.canAdminSomeEventResources(client, event.getResourcesForPermsCheck());

            val isParticipant = event.getParticipants().isParticipantWithInconsistent(clientUid)
                    || eventChangesInfo.getEventParticipantsChangesInfo().wasNewParticipantWithUid(clientUid);

            EventUser eventUser = eventData.getEventUserData().getEventUser().copy();
            eventUser.setIsSubscriber(!isParticipant && passportAuthDomainsHolder.containsYandexTeamRu());
            eventData.setEventUserData(eventData.getEventUserData().withEventUser(eventUser));

            Result r = applyChange();
            Option<EventSendingInfo> outlookerSendingInfo = Option.empty();

            if ((!isRoomAdmin || isParticipant)
                    && !eventRoutines.findUserLayerWithEvent(clientUid, event).isPresent()
                    && !event.isParkingOrApartmentOccupation()
                    && updateMode != UpdateMode.RECURRENCE_FOR_MASTER)
            {
                if (!modificationSituation.isSingle()) {
                    eventRoutines.attachEventOrMeetingsToUserByMainEventId( // CAL-6183
                            client, r.newEvent.map(Event.getMainEventIdF()).getOrElse(event.getMainEventId()),
                            eventData.getLayerId(),
                            eventData.getEventUserData().getEventUser(),
                            NotificationsData.createFromWeb(eventData.getEventUserData().getNotifications()), actionInfo);
                } else {
                    eventRoutines.attachEventOrMeetingToUser( // CAL-6896
                            client, r.getNewEventId().getOrElse(event.getId()), eventData.getLayerId(), Decision.YES,
                            eventData.getEventUserData().getEventUser(),
                            NotificationsData.createFromWeb(eventData.getEventUserData().getNotifications()), actionInfo);
                }
                outlookerSendingInfo = eventInvitationManager.prepareSendingInfoForOutlookerSubscriber(
                        clientUid, r.newEvent.getOrElse(event.getEvent()));
            }

            ListF<EventMessageParameters> eventMails = Cf.list();

            Events events;

            // TODO check: must be able to delete 'eventMails' and act like in BringBackModifier.bringBack(),
            // i.e. first update last_update_ts, then send mails / export updates to exchange // ssytnik@
            if (eventChangesInfo.wasNonPerUserFieldsChange()) {
                ListF<EventSendingInfo> sendingInfos = updateParticipants(r);

                boolean indentsChanged = eventChangesInfo.timeOrRepetitionChanges()
                        || modificationSituation.toDirection() != ModificationScope.SINGLE;

                boolean checkExchangeCompatibleAndResourcesAvailable = indentsChanged
                        || eventChangesInfo.getEventParticipantsChangesInfo().wasResourcesChange();

                events = loadEvents(r, indentsChanged);

                if (checkExchangeCompatibleAndResourcesAvailable) {
                    resourceRoutines.lockResourcesByIds(indentsChanged
                            ? events.newOrUpdatedEvent().getResourceIds()
                            : eventChangesInfo.getEventParticipantsChangesInfo().getNewResources());

                    eventRoutines.ensureExchangeCompatibleIfNeeded(
                            ActorId.user(clientUid), events.newOrUpdatedEvent());

                    resourceAccessRoutines.checkForInaccessibleResources(
                            ActorId.user(clientUid),
                            events.newOrUpdatedEvent(), events.newOrUpdatedEventRepetition(),
                            ResourceAccessRoutines.InaccessibilityCheck.onUpdate(eventChangesInfo),
                            true, actionInfo);

                    if (updateMode == UpdateMode.RECURRENCE_FOR_MASTER
                            && !events.masterEventUpdatedRepetition.get()
                                    .containsInterval(events.updatedEventRepetition.get().getEventInterval()))
                    {
                        ListF<Long> deletedIds = resourceAccessRoutines.checkForBusyResources(
                                ActorId.user(clientUid), events.updatedEvent.get(),
                                events.updatedEventRepetition.get(), Cf.list(), false, actionInfo).getAllIds();

                        events = events.withUpdatedEvent(events.updatedEvent.get().withoutResources(deletedIds));

                    } else {
                        resourceAccessRoutines.checkForBusyResources(ActorId.user(clientUid),
                                events.newOrUpdatedEvent(), events.newOrUpdatedEventRepetition(),
                                r.eventIdsToDelete.plus(events.newOrUpdatedEvent().getId()), true, actionInfo);
                    }
                }
                if (indentsChanged) {
                    ListF<EventAndRepetition> affectedEvents = events.newEventAndRepetition();
                    if (modificationSituation != ModificationSituation.SINGLE_INST) {
                        affectedEvents = affectedEvents.plus(events.updatedEventAndRepetition());
                    }
                    affectedEvents.forEach(e -> eventDbManager.updateEventLayersAndResourcesIndents(e, actionInfo));
                }

                eventRoutines.invalidateResourceScheduleCachesOnUpdate(
                        item.instance.getEventWithRelations(), eventChangesInfo.getEventParticipantsChangesInfo());

                if (events.newEvent.isPresent()) {
                    repetitionConfirmationManager.rescheduleConfirmation(
                            events.newEvent.get(), events.newEventRepetition.get(), actionInfo);
                }
                if (events.updatedEvent.isPresent()) {
                    repetitionConfirmationManager.rescheduleConfirmation(
                            events.updatedEvent.get(), events.updatedEventRepetition.get(), actionInfo);
                }

                Tuple2<ListF<EventSendingInfo>, ListF<EventSendingInfo>> updatedAndNewMails = sendingInfos.partition(
                        EventSendingInfo.getEventIdF().andThen(r.getNewEventId().containsF().notF()));
                ActorId sender = ActorId.user(clientUid);

                if (events.newEvent.isPresent()) {
                    eventMails = eventInvitationManager.createEventInvitationOrCancelMails(
                            sender, events.newEvent.get(), events.newEventRepetition.get(),
                            updatedAndNewMails.get2(), actionInfo);
                }
                eventMails = eventMails.plus(eventInvitationManager.createEventInvitationOrCancelMails(
                        sender, updatedAndNewMails.get1(), actionInfo));

                eventMails = eventMails.plus(handleEventOnLayersChange(
                        events.newOrUpdatedEvent(), events.newOrUpdatedEventRepetition(),
                        events.masterEventUpdated, events.masterEventUpdatedRepetition));

                if (outlookerSendingInfo.isPresent()
                    && !sendingInfos.exists(s -> s.getUnivContact().getUid().isSome(clientUid)))
                {
                    eventMails = eventMails.plus(eventInvitationManager.createEventInvitationOrCancelMails(
                            ActorId.yaCalendar(), events.newOrUpdatedEvent(), events.newOrUpdatedEventRepetition(),
                            outlookerSendingInfo, actionInfo));
                }
                if (updateMode != UpdateMode.RECURRENCE_FOR_MASTER) {
                    exportUpdatesToExchange(
                            events.updatedEvent, events.updatedEventRepetition,
                            events.newEvent, events.newEventRepetition, exportToExchangeImmediately);
                }
                if (r.eventIdsToDelete.isNotEmpty()) {
                    r = r.plusModified(r.eventIdsToDelete);

                    eventRoutines.deleteEventsFromDbAndExchange(
                            ActorId.user(clientUid), r.eventIdsToDelete, actionInfo, true);
                }

            } else {
                events = loadEvents(r, false);

                eventMails = eventMails.plus(handleEventOnLayersChange(
                        events.newOrUpdatedEvent(), events.newOrUpdatedEventRepetition(),
                        events.masterEventUpdated, events.masterEventUpdatedRepetition));

                if (outlookerSendingInfo.isPresent()) {
                    eventMails = eventMails.plus(eventInvitationManager.createEventInvitationOrCancelMails(
                            ActorId.yaCalendar(), events.newOrUpdatedEvent(), events.newOrUpdatedEventRepetition(),
                            outlookerSendingInfo, actionInfo));
                }
            }

            if (r.recurrencesMails.isNotEmpty()) {
                eventMails = EventWebManager.mergeMasterAndRecurrencesMails(
                        eventMails.plus(r.recurrencesMails));
            }

            if (disableAllMails) {
                eventMails = Cf.list();
            }

            if (invitationProcessingMode == InvitationProcessingMode.SAVE_ATTACH_SEND) {
                eventInvitationManager.sendEventMails(eventMails, actionInfo);
            }

            if (events.updatedEvent.isPresent() && events.updatedEventAndRepetition().isPresent()
                    && eventChangesInfo.timeOrRepetitionRuleOrDueTsChanges())
            {
                eventInvitationManager.sendDecisionFixingMailsIfNeeded(events.updatedEvent.get(),
                        events.updatedEventAndRepetition().get().getRepetitionInfo(), actionInfo);
            }

            logEventChanges(events);
            updateSequenceAndDtstamp();

            return ModificationInfo.updated(
                    events.updatedEventAndRelations(), modificationSituation.toDirection(),
                    r.modifiedEventIds, Cf.list(item.getExternalId())
                            .plus(events.newEvent.map(EventWithRelations::getExternalId)).stableUnique(),
                    events.newEventAndRelations(), eventMails);
        }

        private void logEventChanges(Events events) {
            if (needsEwsRecreation()) {
                Option<RecurrenceTimeInfo> occurrence = item.instanceStart().orElse(item.event()::getRecurrenceId)
                        .map(s -> new RecurrenceTimeInfo(s, item.instance.getIntervalOrEventInterval()));

                eventsLogger.log(EventChangeLogEvents.recreated(ActorId.user(clientUid),
                        occurrence.map(RecurrenceTimeInfo::getRecurrenceId),
                        occurrence.map(item.instance.getEventWithRelations()::withRecurrenceTimeInfo)
                                .getOrElse(item.instance::getEventWithRelations),
                        occurrence.map(RecurrenceTimeInfo::getInterval)
                                .map(events.newOrUpdatedEventRepetition()::withInterval)
                                .getOrElse(item.instance::getRepetitionInstanceInfo),
                        events.newOrUpdatedEvent(), events.newOrUpdatedEventRepetition()), actionInfo);

            } else if (modificationSituation == ModificationSituation.SINGLE_INST) {
                RecurrenceTimeInfo timeInfo = new RecurrenceTimeInfo(
                        item.instanceStart().get(), item.instance.getInterval().get());

                eventsLogger.log(EventChangeLogEvents.updated(ActorId.user(clientUid),
                        item.instance.getEventWithRelations().withRecurrenceTimeInfo(timeInfo),
                        events.newOrUpdatedEventRepetition().withInterval(timeInfo.getInterval()),
                        events.newOrUpdatedEvent(), events.newOrUpdatedEventRepetition()), actionInfo);

            } else if (modificationSituation == ModificationSituation.THIS_AND_FUTURE) {
                eventsLogger.log(EventChangeLogEvents.updated(ActorId.user(clientUid),
                        item.instance.getEventWithRelations(), item.instance.getRepetitionInstanceInfo(),
                        events.updatedEvent.get(), events.updatedEventRepetition.get()), actionInfo);

                eventsLogger.log(EventChangeLogEvents.tailed(ActorId.user(clientUid),
                        item.instance.getEventWithRelations()
                                .withInterval(item.instance.getInterval().get()),
                        item.instance.getRepetitionInstanceInfo()
                                .withInterval(item.instance.getInterval().get())
                                .withoutExdatesAndRdatesAndRecurrences(),
                        events.newEvent.get(), events.newEventRepetition.get()), actionInfo);

            } else {
                eventsLogger.log(EventChangeLogEvents.updated(ActorId.user(clientUid),
                        item.instance.getEventWithRelations(), item.instance.getRepetitionInstanceInfo(),
                        events.newOrUpdatedEvent(), events.newOrUpdatedEventRepetition()), actionInfo);
            }
        }

        private ListF<EventOnLayerChangeMessageParameters> handleEventOnLayersChange(
                EventWithRelations newOrUpdated, RepetitionInstanceInfo newOrUpdatedRepetition,
                Option<EventWithRelations> masterUpdated, Option<RepetitionInstanceInfo> masterUpdatedRepetition)
        {
            if (modificationSituation == ModificationSituation.THIS_AND_FUTURE) {
                LayerIdChangesInfo layerIdChanges = LayerIdChangesInfo.changes(
                        masterLayerIdsBeforeUpdate, newOrUpdated.getLayerIds());

                ListF<EventOnLayerChangeMessageParameters> dead = eventsOnLayerChangeHandler.handleEventChange(
                        UidOrResourceId.user(clientUid), masterUpdated.get(), masterUpdatedRepetition.get(),
                        LayerIdChangesInfo.unchanged(layerIdChanges.getRemoved()),
                        EventChangesInfoForMails.timeOrRepetitionChanged(), actionInfo);

                ListF<EventOnLayerChangeMessageParameters> survived = eventsOnLayerChangeHandler.handleEventChange(
                        UidOrResourceId.user(clientUid), newOrUpdated, newOrUpdatedRepetition,
                        layerIdChanges.withoutRemoved(),
                        eventChangesInfo.toEventChangesInfoForMails(), actionInfo);

                return dead.plus(survived);

            } else if (masterUpdated.isPresent() && modificationSituation.isInstNotFuture()) {
                LayerIdChangesInfo masterLayerIdChanges = LayerIdChangesInfo.changes(
                        masterLayerIdsBeforeUpdate, masterUpdated.get().getLayerIds());

                LayerIdChangesInfo instanceLayerIdChanges = LayerIdChangesInfo.changes(
                        instanceLayerIdsBeforeUpdate, newOrUpdated.getLayerIds());

                ListF<EventOnLayerChangeMessageParameters> master = eventsOnLayerChangeHandler.handleEventChange(
                        UidOrResourceId.user(clientUid), masterUpdated.get(), masterUpdatedRepetition.get(),
                        masterLayerIdChanges, EventChangesInfoForMails.EMPTY, actionInfo);

                ListF<EventOnLayerChangeMessageParameters> instance = eventsOnLayerChangeHandler.handleEventChange(
                        UidOrResourceId.user(clientUid), newOrUpdated, newOrUpdatedRepetition,
                        instanceLayerIdChanges.filterOut(masterLayerIdChanges.getAddedAndRemoved()),
                        eventChangesInfo.toEventChangesInfoForMails(), actionInfo);

                return master.plus(instance);

            } else {
                return eventsOnLayerChangeHandler.handleEventChange(
                        UidOrResourceId.user(clientUid), newOrUpdated, newOrUpdatedRepetition,
                        LayerIdChangesInfo.changes(instanceLayerIdsBeforeUpdate, newOrUpdated.getLayerIds()),
                        eventChangesInfo.toEventChangesInfoForMails(), actionInfo);
            }
        }

        private void exportUpdatesToExchange(
                Option<EventWithRelations> updatedEvent, Option<RepetitionInstanceInfo> updatedRepetition,
                Option<EventWithRelations> newEvent, Option<RepetitionInstanceInfo> newEventRepetition, boolean exportToExchangeImmediately)
        {
            if (modificationSituation == ModificationSituation.THIS_AND_FUTURE) {
                EventChangesInfoFactory changesFactory = new EventChangesInfoFactory();
                changesFactory.setRepetitionChanges(updatedRepetition.get().getRepetition().get().withoutId());

                ewsExportRoutines.exportToExchangeIfNeededOnUpdate(
                        updatedEvent.get(), updatedRepetition.get(), Option.empty(), changesFactory.create(),
                        mailToAll.map(ExchangeMails::fromMailToAll), actionInfo);

                ewsExportRoutines.exportToExchangeIfNeededOnCreate(
                        newEvent.get(), newEventRepetition.get(), mailToAll.map(ExchangeMails::fromMailToAll), actionInfo);

                return;
            }
            Option<OccurrenceId> occurrenceIdO = Option.empty();
            String externalId = item.getInstance().getEventWithRelations().getExternalId();

            if (modificationSituation == ModificationSituation.SINGLE_INST) {
                occurrenceIdO = Option.of(new OccurrenceId(
                        externalId, item.getInstance().getInterval().get(), item.instanceStart().get()));

                if (oldOrganizerId().isPresent() && item.instance.isExportedWithEws()) {
                    ewsExportRoutines.cancelOccurrenceIfNeeded(
                            item.instance.getEventWithRelations(), occurrenceIdO.get(), actionInfo);
                }
            }
            if (modificationSituation == ModificationSituation.RECURRENCE_INST) {
                InstantInterval interval = new InstantInterval(item.event().getStartTs(), item.event().getEndTs());
                occurrenceIdO = Option.of(new OccurrenceId(externalId, interval, item.event().getRecurrenceId().get()));
            }

            if (newOrganizerIsEwser().isSome(true)) {
                ewsExportRoutines.exportToExchangeIfNeededOnCreate(
                        newEvent.get(), newEventRepetition.get(), actionInfo);

                return;
            }

            if (exportToExchangeImmediately) {
                ewsExportRoutines.exportToExchangeIfNeededOnUpdate(
                        newEvent.getOrElse(updatedEvent::get), newEventRepetition.getOrElse(updatedRepetition::get),
                        occurrenceIdO, eventChangesInfo, mailToAll.map(ExchangeMails::fromMailToAll), actionInfo);
            } else {
               val event = newEvent.getOrElse(updatedEvent::get);
               ewsExportRoutines.scheduleNewExportTask(eventChangesInfo, event, occurrenceIdO.toOptional(), actionInfo);
            }
        }

        private Events loadEvents(Result r, boolean lockForUpdate) {
            Option<Long> updatedEventId = Option.of(item.instance.getEventId()).filterNot(r.eventIdsToDelete::containsTs);
            Option<Long> masterEventId = item.master.map(Event.getIdF()).filterNot(r.eventIdsToDelete::containsTs);
            Option<Long> newEventId = r.newEvent.map(Event.getIdF()).filterNot(r.eventIdsToDelete::containsTs);

            ListF<Long> eventIds = newEventId.plus(masterEventId).plus(updatedEventId);

            ListF<EventAndRepetition> evAndReps =
                    eventDbManager.getEventsAndRepetitionsByEventIds(eventIds, lockForUpdate);

            MapF<Long, EventWithRelations> eventById = eventDbManager
                    .getEventsWithRelationsByEvents(evAndReps.map(EventAndRepetition::getEvent))
                    .toMapMappingToKey(EventWithRelations::getId);

            MapF<Long, RepetitionInstanceInfo> repetitionByEventId = evAndReps.toMap(
                    EventAndRepetition::getEventId, EventAndRepetition::getRepetitionInfo);

            return new Events(
                    eventById.getO(updatedEventId.getOrNull()), repetitionByEventId.getO(updatedEventId.getOrNull()),
                    eventById.getO(newEventId.getOrNull()), repetitionByEventId.getO(newEventId.getOrNull()),
                    eventById.getO(masterEventId.getOrNull()), repetitionByEventId.getO(masterEventId.getOrNull()));
        }

        private void updateSequenceAndDtstamp() {
            long eventId = item.event().getId();

            boolean ignoreRepetition = modificationSituation == ModificationSituation.SINGLE_INST;
            if (eventChangesInfo.wasEnoughToIncrementSequenceChange(ignoreRepetition)) {
                eventDao.updateEventIncrementSequenceById(eventId);
            }

            eventUserDao.updateDtstampAndCopySequenceForWeb(eventId, clientUid, actionInfo);
            lastUpdateManager.updateTimestampsAsync(item.event().getMainEventId(), actionInfo);
        }

        private ListF<EventSendingInfo> updateParticipants(Result result) {
            ListF<EventSendingInfo> mainEventSendingInfoList = Cf.list();

            if (modificationSituation == ModificationSituation.THIS_AND_FUTURE) {
                mainEventSendingInfoList = updateMeetingHandler.handleMeetingUpdate(
                        item.master.get(), ActorId.user(clientUid),
                        new ChangedEventInfoForMails(
                                item.getInstance().withEmptyInterval(),
                                EventChangesInfoForMails.timeOrRepetitionChanged()),
                        MeetingMailRecipients.NOT_REJECTED_PARTICIPANTS_AND_OUTLOOKERS,
                        item.instance.getEventWithRelations().isParkingOrApartmentOccupation(),
                        actionInfo, EventInstanceStatusInfo.needToUpdate(item.event().getId()),
                        updateMode, item.instance.isExportedWithEws()).getSendingInfos();

            } else if (needsEwsRecreation()) {
                SetF<Email> removed = eventChangesInfo.getEventParticipantsChangesInfo()
                        .getRemovedParticipants().map(ParticipantInfo::getEmail).unique();

                ChangedEventInfoForMails changes = new ChangedEventInfoForMails(
                        item.instance, eventChangesInfo.toEventChangesInfoForMails());

                mainEventSendingInfoList = cancelMeetingHandler.cancelMeeting(
                        item.instance.getEvent(), Option.of(clientUid),
                        new EventInstanceParameters(
                                item.instance.getIntervalOrEventInterval(),
                                item.instanceStart().orElse(item.event()::getRecurrenceId)),
                        item.instance.getEventWithRelations().isParkingOrApartmentOccupation(),
                        item.instance.isExportedWithEws(), actionInfo)
                        .map(es -> removed.containsTs(es.getUnivContact().getEmail())
                                ? es : es.reorganized(MailType.EVENT_REORGANIZATION, changes, result.getNewEventId()));
            }

            Event updatedInstance = item.event().copy();
            updatedInstance.setFields(eventChangesInfo.getEventChanges());

            Event e = result.newEvent.getOrElse(updatedInstance);
            EventChangesInfoForMails changesInfo;

            Option<Decision> decision = eventChangesInfo.getEventUserChanges().getFieldValueO(EventUserFields.DECISION);
            if (decision.isPresent()) {
                changesInfo = eventChangesInfo
                        .withParticipantDecision(UidOrResourceId.user(clientUid), decision.get())
                        .toEventChangesInfoForMails();
            } else {
                changesInfo = eventChangesInfo.toEventChangesInfoForMails();
            }

            MeetingMailRecipients recipients = eventChangesInfo.getMeetingMailRecipients();

            if (modificationSituation == ModificationSituation.THIS_AND_FUTURE || needsEwsRecreation()) {
                recipients = recipients != MeetingMailRecipients.ALL_PARTICIPANTS_AND_SUBSCRIBERS
                        ? MeetingMailRecipients.NOT_REJECTED_PARTICIPANTS_AND_SUBSCRIBERS
                        : MeetingMailRecipients.ALL_PARTICIPANTS_AND_SUBSCRIBERS;

            } else if (mailToAll.isSome(true) && recipients != MeetingMailRecipients.ALL_PARTICIPANTS_AND_SUBSCRIBERS) {
                recipients = MeetingMailRecipients.NOT_REJECTED_PARTICIPANTS_AND_SUBSCRIBERS;
            }

            EventInstanceForUpdate instanceForMails = item.getInstance();
            if (modificationSituation.isSingle()) {
                instanceForMails = instanceForMails.withNoRepetition();
            }

            ListF<EventSendingInfo> newEventSendingInfoList = updateMeetingHandler.handleMeetingUpdate(
                    e, ActorId.user(clientUid),
                    new ChangedEventInfoForMails(instanceForMails, changesInfo), recipients,
                    item.instance.getEventWithRelations().isParkingOrApartmentOccupation(),
                    actionInfo, EventInstanceStatusInfo.needToUpdate(result.getNewEventId().getOrElse(item.event()::getId)),
                    updateMode, newOrganizerIsEwser().getOrElse(item.instance::isExportedWithEws)).getSendingInfos();

            return mainEventSendingInfoList.plus(newEventSendingInfoList);
        }

        private Result modifyMainInstanceOfRepeatingAndFuture() {
            if (needsEwsRecreation()) {
                ListF<EventMessageParameters> recurrencesMails = eventWebRemover.createCancelEmailsForFutureRecurrences(
                        ActorId.user(clientUid), item, actionInfo);

                return cloneEventForEwsRecreation(actionInfo)
                        .plusToDelete(eventRoutines.findMasterAndSingleEventIds(item.instance.getEventId()))
                        .plusRecurrencesMails(recurrencesMails);
            }

            Tuple2<ListF<Long>, ListF<EventMessageParameters>> futureInstances =
                    eventRoutines.removeFutureInstancesIfTimeOrRepetitionChanges(
                            ActorId.user(clientUid), item.instance, eventChangesInfo, true, actionInfo);

            return updateCommon().plusModified(futureInstances.get1()).plusRecurrencesMails(futureInstances.get2());
        }

        private Result modifyThisAndFuture() {
            Tuple2<ListF<Long>, ListF<EventMessageParameters>> futureInstances =
                    eventRoutines.removeFutureInstances(
                            ActorId.user(clientUid), item.instance, item.master, item.instanceStart(), true, actionInfo);

            Option<InstantInterval> lastInterval = RepetitionUtils.getIntervals(
                    item.instance.getRepetitionInstanceInfo().withoutExdatesAndRdatesAndRecurrences(),
                    item.instance.getEvent().getStartTs(), item.instanceStart(), false, 0).lastO();

            DateTimeZone tz = item.instance.getEventWithRelations().getTimezone();

            Option<Instant> dueTs = lastInterval.map(i -> EventRoutines.calculateDueTsFromUntilDate(
                    i.getStart(), new LocalDate(i.getEnd(), tz), tz));

            eventDao.updateRepetitionDueTs(item.master.get().getRepetitionId().get(), dueTs.orElse(item.instanceStart()).get());

            Result r = cloneEventWithMain(new Event(), eventChangesInfo.getAttachmentChangesInfo(), actionInfo);

            long newEventId = r.getNewEventId().get();
            notificationRoutines.recalcNextSendTsForNewRecurrenceOrTail(item.event().getId(), newEventId, actionInfo);

            lastUpdateManager.updateMainEventAndLayerTimestamps(futureInstances.get1().plus(newEventId), actionInfo);

            return r.plusModified(futureInstances.get1()).plusRecurrencesMails(futureInstances.get2());
        }

        private Result updateCommon() {
            eventRoutines.updateEventCommon(
                    Option.of(clientUid), item.getInstance(), eventChangesInfo,
                    updateMode, actionInfo);

            return new Result(Option.empty(), Cf.list(), Cf.list(item.event().getId()), Cf.list());
        }

        private Option<ParticipantId> oldOrganizerId() {
            return eventChangesInfo.getEventParticipantsChangesInfo().getNewOrganizer().isPresent()
                    ? eventChangesInfo.getEventParticipantsChangesInfo().getOldOrganizers().singleO()
                    : Option.empty();
        }

        private Option<ParticipantId> newOrganizerId() {
            return eventChangesInfo.getEventParticipantsChangesInfo().getOldOrganizers().isNotEmpty()
                    ? eventChangesInfo.getEventParticipantsChangesInfo().getNewOrganizer()
                    : Option.empty();
        }

        private Option<Boolean> newOrganizerIsEwser() {
            return newOrganizerId().map(id -> id.getUidIfYandexUser().exists(settingsRoutines::getIsEwser));
        }

        private boolean needsEwsRecreation() {
            return updateMode != UpdateMode.RECURRENCE_FOR_MASTER
                    && modificationSituation != ModificationSituation.THIS_AND_FUTURE
                    && (newOrganizerIsEwser().isSome(true)
                            || oldOrganizerId().isPresent() && item.instance.isExportedWithEws());
        }
    }

    public ModificationInfo update(
            UserInfo user, EventData eventData, NotificationsData.Update notificationsUpdate,
            boolean applyToFuture, ActionInfo actionInfo)
    {
        return update(
                user, eventData, notificationsUpdate, EventsAttachedLayerIds.empty(),
                applyToFuture, InvitationProcessingMode.SAVE_ATTACH_SEND, Option.empty(), actionInfo, true);
    }

    public ModificationInfo update(UserInfo user, EventData eventData, NotificationsData.Update updateNotificationsData,
                                   EventsAttachedLayerIds preattachedLayerIds, boolean applyToFuture,
                                   InvitationProcessingMode invitationMode, Option<Boolean> mailToAll, ActionInfo actionInfo,
                                   boolean exportToExchangeImmediately) {
        return update(user, eventData, updateNotificationsData, preattachedLayerIds, applyToFuture, invitationMode,
            mailToAll, false, actionInfo, exportToExchangeImmediately);
    }

    public ModificationInfo update(
            UserInfo user, EventData eventData, NotificationsData.Update updateNotificationsData,
            EventsAttachedLayerIds preattachedLayerIds, boolean applyToFuture,
            InvitationProcessingMode invitationMode, Option<Boolean> mailToAll, boolean disableAllMails,
            ActionInfo actionInfo, boolean exportToExchangeImmediately)
    {
        Validate.isTrue(actionInfo.getActionSource().isWebOrApi()
                || actionInfo.getActionSource() == ActionSource.DISPLAY
                || actionInfo.getActionSource() == ActionSource.DB_REPAIRER
        );
        PassportUid uid = user.getUid();

        if (!passportAuthDomainsHolder.containsYandexTeamRu() && applyToFuture) {
            shiftToMainInstance(eventData);
        }

        ModificationItem modificationItem = eventWebManager.getModificationItem(
                eventData.getEvent().getId(), eventData.getInstanceStartTs(),
                Option.of(uid), eventData.getPrevLayerId(), applyToFuture, actionInfo);

        if (modificationItem.getInstance().getEvent().getName().startsWith(SpecialNames.FAIL_ON_UPDATE)) {
            throw new RuntimeException(SpecialNames.FAIL_ON_UPDATE);
        }

        val isEventDataCouldBeChanged = actionInfo.getTvmId()
            .stream()
            .anyMatch(experimentAwareTvmIds::contains);

        var eventChangesInfo = findEventChanges(user, modificationItem, eventData, updateNotificationsData,
            actionInfo.getActionSource(), isEventDataCouldBeChanged);

        ModificationSituation modificationSituation = eventWebManager.chooseModificationSituation(
                modificationItem, applyToFuture, eventChangesInfo.wasCuttingChange(), actionInfo);

        if (modificationSituation.isInstNotFuture()) {
            eventChangesInfo = eventChangesInfo.withoutRepetitionChanges();
        }
        if (eventChangesInfo.getEventParticipantsChangesInfo().wasNewParticipantWithUid(user.getUid())) {
            eventChangesInfo = eventChangesInfo.withParticipantDecision(UidOrResourceId.user(user.getUid()), Decision.YES);
        }

        EventChangesInfo masterChangesInfo = eventChangesInfo;

        if (!eventChangesInfo.wasChange() && preattachedLayerIds.getAllLayerIds().isEmpty() && !applyToFuture) {
            log.info("No change");

            return ModificationInfo.updated(
                    Option.of(modificationItem.instance.getEventAndRelations()), ModificationScope.NONE,
                    Cf.list(), Cf.list(modificationItem.getExternalId()), Option.empty(), Cf.list());
        }
        if (modificationSituation.isMainAndFuture() && !masterChangesInfo.timeOrRepetitionRuleChanges()) {
            return updateFutureInstances(
                    user, modificationItem, modificationItem,
                    eventData, updateNotificationsData,
                    masterChangesInfo, preattachedLayerIds,
                    invitationMode, mailToAll, actionInfo, exportToExchangeImmediately, isEventDataCouldBeChanged);

        } else {
            return updateWithSmsHandling(
                    user, modificationSituation, modificationItem, eventData, eventChangesInfo,
                    preattachedLayerIds, invitationMode,
                    mailToAll, disableAllMails, UpdateMode.NORMAL, actionInfo, exportToExchangeImmediately);
        }
    }

    private ModificationInfo updateWithSmsHandling(
        UserInfo user, ModificationSituation modificationSituation, ModificationItem modificationItem,
        EventData eventData, EventChangesInfo eventChangesInfo,
        EventsAttachedLayerIds preattachedLayerIds,
        InvitationProcessingMode invitationMode, Option<Boolean> sendToAll,
        UpdateMode updateMode, ActionInfo actionInfo, boolean exportToExchangeImmediately) {
        return updateWithSmsHandling(user, modificationSituation, modificationItem, eventData, eventChangesInfo,
            preattachedLayerIds, invitationMode, sendToAll, false, updateMode, actionInfo, exportToExchangeImmediately);
    }

    private ModificationInfo updateWithSmsHandling(
            UserInfo user, ModificationSituation modificationSituation, ModificationItem modificationItem,
            EventData eventData, EventChangesInfo eventChangesInfo,
            EventsAttachedLayerIds preattachedLayerIds,
            InvitationProcessingMode invitationMode, Option<Boolean> sendToAll, boolean disableAllMails,
            UpdateMode updateMode, ActionInfo actionInfo, boolean exportToExchangeImmediately)
    {
        InstantInterval suitableInterval = eventMoveSmsHandler.suitableInterval(actionInfo.getNow());

        ListF<EventInstanceInterval> moveWatchingInstances = eventDbManager
                .getEventsAndRepetitionsByMainEventId(modificationItem.instance.getMainEventId())
                .flatMap(EventAndRepetition.getInstancesStartingInIntervalF(suitableInterval));

        ModificationInfo result = new Do(
                user, modificationSituation, modificationItem,
                eventData, eventChangesInfo, preattachedLayerIds,
                invitationMode, sendToAll, disableAllMails, updateMode, actionInfo, exportToExchangeImmediately).update();

        ListF<MainEvent> mainEvents = Cf.list(modificationItem.instance.getEventWithRelations().getMainEvent());

        if (result.getNewEvent().exists(e -> e.getMainEventId() != modificationItem.instance.getMainEventId())) {
            mainEvents = mainEvents.plus(mainEventDao.findMainEventById(result.getNewEvent().get().getMainEventId()));
        }

        eventMoveSmsHandler.handleEventUpdate(user.getUid(), moveWatchingInstances, mainEvents, actionInfo);

        return result;
    }

    private ModificationInfo updateFutureInstances(
        UserInfo user, ModificationItem initialItem, ModificationItem masterItem,
        EventData eventData, NotificationsData.Update updateNotificationsData,
        EventChangesInfo masterChangesInfo, EventsAttachedLayerIds preattachedLayerIds,
        InvitationProcessingMode invitationMode, Option<Boolean> sendToAll, ActionInfo actionInfo, boolean exportToExchangeImmediately,
        boolean eventDataCouldBeChanged) {
        return updateFutureInstances(user, initialItem, masterItem, eventData, updateNotificationsData, masterChangesInfo,
            preattachedLayerIds, invitationMode, sendToAll, false, actionInfo, exportToExchangeImmediately,
            eventDataCouldBeChanged);
    }

    private ModificationInfo updateFutureInstances(
            UserInfo user, ModificationItem initialItem, ModificationItem masterItem,
            EventData eventData, NotificationsData.Update updateNotificationsData,
            EventChangesInfo masterChangesInfo, EventsAttachedLayerIds preattachedLayerIds,
            InvitationProcessingMode invitationMode, Option<Boolean> sendToAll, boolean disableAllMails,
            ActionInfo actionInfo, boolean exportToExchangeImmediately, boolean eventDataCouldBeChanged)
    {
        Validate.isFalse(masterChangesInfo.timeOrRepetitionRuleChanges());

        EventParticipantsChangesInfo participants = masterChangesInfo.getEventParticipantsChangesInfo();

        Tuple2List<ParticipantId, ParticipantData> newParticipants = participants.getNewParticipants();

        ListF<ParticipantId> removedParticipantIds = participants.getRemovedParticipants().map(ParticipantInfo::getId);

        ListF<Event> recurrenceEvents = eventDao.findFutureRecurrencesForUpdate(
                masterItem.event().getMainEventId(), actionInfo);

        Option<Instant> dueTs = masterChangesInfo.getRepetitionChanges()
                .getFieldValueO(RepetitionFields.DUE_TS).filterNotNull();

        if (dueTs.isPresent()) {
            recurrenceEvents = recurrenceEvents.filter(e -> e.getRecurrenceId().get().isBefore(dueTs.get()));
        }

        ListF<EventInfo> recurrenceInfos = eventInfoDbLoader.getEventInfosByEvents(
                Option.of(user), EventGetProps.any(), recurrenceEvents, actionInfo.getActionSource());

        ListF<EventMessageParameters> recurrenceMails = Cf.arrayList();

        for (EventInfo recurrence : recurrenceInfos) {
            EventInstanceForUpdate instance = new EventInstanceForUpdate(
                    Option.empty(), recurrence.getInfoForPermsCheck(),
                    recurrence.getEventWithRelations(), recurrence.getRepetitionInstanceInfo(),
                    recurrence.getEventUserWithNotifications(), Option.empty(), recurrence.getAttachments());

            ModificationItem item = new ModificationItem(instance, Option.of(masterItem.event()));

            Tuple2<ListF<ParticipantInfo>, ListF<ParticipantInfo>> removedAndSurvived = recurrence
                    .getEventParticipants().getParticipants().getParticipantsSafe()
                    .partition(removedParticipantIds.unique().containsF().compose(ParticipantInfo::getId));

            SetF<ParticipantId> survivedIds = removedAndSurvived.get2().map(ParticipantInfo::getId).unique();

            EventParticipantsChangesInfo participantsChanges = EventParticipantsChangesInfo.changes(
                    newParticipants.filterBy1Not(survivedIds::containsTs),

                    Cf2.flatBy2(removedAndSurvived.get2().toTuple2List(ParticipantInfo::getId, p -> {
                        Option<UserParticipantInfo> upO = Option.of(p).filterByType(UserParticipantInfo.class);

                        if (upO.exists(up -> participants.getNewOrganizer().isSome(up.getId()))) {
                            return Option.of(new ParticipantChangesInfo(upO.get(), true));

                        } else if (participants.wasOrganizerChange() && upO.exists(ParticipantInfo::isOrganizer)) {
                            return Option.of(new ParticipantChangesInfo(upO.get(), false));
                        }
                        return Option.empty();
                    })),
                    removedAndSurvived.get1());

            EventData data = eventData.copy();
            data.getEvent().unsetAllFields();

            data.getEvent().setFields(masterChangesInfo.getEventChanges()
                    .getFieldsIfSet(EventFields.PERM_ALL, EventFields.PERM_PARTICIPANTS));

            Cf.list(EventFields.LOCATION, EventFields.NAME, EventFields.DESCRIPTION).forEach(field -> {
                if (ObjectUtils.equals(masterItem.event().getFieldValue(field), item.event().getFieldValue(field))) {
                    data.getEvent().setFields(masterChangesInfo.getEventChanges().getFieldsIfSet(field));
                }
            });

            data.setRepetition(RepetitionRoutines.createNoneRepetition());
            data.setInstanceStartTs(Option.empty());

            telemostManager.patchJsonDataForRecurrenceUpdateWithMaster(
                    data.getEvent(), instance, masterChangesInfo, masterItem.instance
            );

            var changes = eventChangesFinder.getEventChangesInfo(item.instance, data, updateNotificationsData,
                Option.of(user.getUid()), true, eventDataCouldBeChanged);

            changes = changes
                    .withParticipantsChangesInfo(participantsChanges)
                    .withAttachmentChangesInfo(masterChangesInfo.getAttachmentChangesInfo());

            ModificationInfo info = new Do(
                    user, ModificationSituation.RECURRENCE_INST, item, data,
                    changes, preattachedLayerIds, InvitationProcessingMode.SAVE_ATTACH,
                    sendToAll, disableAllMails, UpdateMode.RECURRENCE_FOR_MASTER, actionInfo, exportToExchangeImmediately).update();

            recurrenceMails.addAll(info.getEventMails());
        }
        ModificationInfo masterInfo = updateWithSmsHandling(
                user, ModificationSituation.MAIN_INST_AND_FUTURE,
                masterItem.withEmptyInterval(), eventData, masterChangesInfo.withoutEventStartEndTsChanges(),
                preattachedLayerIds, invitationMode, sendToAll, UpdateMode.NORMAL, actionInfo, exportToExchangeImmediately);

        ListF<EventMessageParameters> mergedMails = EventWebManager.mergeMasterAndRecurrencesMails(
                masterInfo.getEventMails().plus(recurrenceMails));

        if (disableAllMails) {
            mergedMails = Cf.list();
        }

        if (invitationMode.isSend()) {
            eventInvitationManager.sendEventMails(recurrenceMails.filter(mergedMails.unique().containsF()), actionInfo);
        }

        return ModificationInfo.updated(
                Option.of(initialItem.instance.getEventAndRelations()), ModificationScope.ALL,
                masterInfo.getEventIds(), masterInfo.getExternalIds(),
                masterInfo.getNewEventAndRelations(), mergedMails);
    }

    private EventChangesInfo findEventChanges(UserInfo user, ModificationItem modificationItem, EventData eventData,
                                              NotificationsData.Update notificationsData, ActionSource actionSource,
                                              boolean eventDataCouldBeChanged) {
        var eventChangesInfo = eventChangesFinder.getEventChangesInfo(modificationItem.instance, eventData, notificationsData,
            Option.of(user.getUid()), true, eventDataCouldBeChanged);
        val event = modificationItem.getInstance().getEventWithRelations();

        val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(user, event);
        val eventPermissions = authorizer.getEventPermissions(user, eventAuthInfo, actionSource);
        if (!eventPermissions.contains(EventAction.EDIT)) {
            if (eventPermissions.contains(EventAction.INVITE)) {
                if (eventChangesInfo.wasNonPerUserAndNonPartcipantsFieldsChange()) {
                    log.warn("Ignoring non per user changes and non partcipant changes from {}", user.getUid());
                }
                eventChangesInfo = eventChangesInfo.onlyInvitationPermittedChanges();
            } else {
                if (eventChangesInfo.wasNonPerUserFieldsChange()) {
                    log.warn("Ignoring non per user changes from {}", user.getUid());
                }
                eventChangesInfo = eventChangesInfo.onlyPerUserChanges();
            }
        }
        return eventChangesInfo;
    }

    private void shiftToMainInstance(EventData eventData) {
        long eventId = eventData.getEvent().getId();

        Event master = eventDao.findMasterEventByEventId(eventId).getOrThrow(() ->
                CommandRunException.createSituation("no master for event id" + eventId, Situation.EVENT_NOT_FOUND));

        Instant instanceStart;

        if (master.getId() != eventId) {
            instanceStart = eventDao.findEventById(eventId).getRecurrenceId().get();

        } else if (eventData.getInstanceStartTs().isPresent()) {
            instanceStart = eventData.getInstanceStartTs().get();

        } else {
            instanceStart = eventData.getEvent().getStartTs();
        }

        DateTimeZone tz = eventData.getTimeZone();

        Instant newEventStartTs = AuxDateTime.addPeriod(master.getStartTs(),
                AuxDateTime.periodBetween(instanceStart, eventData.getEvent().getStartTs(), tz), tz);

        eventData.setInstanceStartTs(Option.empty());
        eventData.getEvent().setId(master.getId());

        eventData.getEvent().setEndTs(AuxDateTime.addPeriod(newEventStartTs,
                AuxDateTime.periodBetween(eventData.getEvent().getStartTs(), eventData.getEvent().getEndTs(), tz), tz));
        eventData.getEvent().setStartTs(newEventStartTs);
    }

} //~
