package ru.yandex.calendar.logic.ics;

import ru.yandex.misc.lang.DefaultObject;

/**
 * @author akirakozov
 * @author gutman
 */
public abstract class EventInstanceStatusInfo extends DefaultObject {

    public static EventInstanceStatusInfo notFound() {
        return new NotFound();
    }

    public static EventInstanceStatusInfo needToUpdate(long eventId) {
        return new NeedToUpdate(eventId);
    }

    public static EventInstanceStatusInfo needToUpdateWithNewExternalId(
            long eventId, String newExternalId, boolean isExportedWithEws) {
        return new NeedToUpdateWithNewExternalId(eventId, newExternalId, isExportedWithEws);
    }

    public static EventInstanceStatusInfo alreadyUpdated(long eventId) {
        return new AlreadyUpdated(eventId);
    }

    public static EventInstanceStatusInfo fromCaldavAlreadyUpdated(long eventId) {
        return new FromCaldavAlreadyUpdated(eventId);
    }

    public boolean isFound() {
        return !isNotFound();
    };

    public abstract long getEventId();

    public String getChangedExternalId() {
        if (!isNeedToUpdateWithNewExternalId()) {
            throw new IllegalStateException("not isNeedToUpdateWithNewExternalId " + this);
        }
        return ((NeedToUpdateWithNewExternalId) this).changedExternalId;
    }

    public boolean getIsExportedWithEws() {
        if (!isNeedToUpdateWithNewExternalId()) {
            throw new IllegalStateException("not isNeedToUpdateWithNewExternalId " + this);
        }
        return ((NeedToUpdateWithNewExternalId) this).isExportedWithEws;
    }


    private static class NotFound extends EventInstanceStatusInfo {
        @Override
        public String toString() {
            return "notFound";
        }

        @Override
        public long getEventId() {
            throw new IllegalStateException("not found");
        }
    }


    private static class NeedToUpdate extends EventInstanceStatusInfo {
        private final long eventId;

        public NeedToUpdate(long eventId) {
            this.eventId = eventId;
        }

        @Override
        public String toString() {
            return "needToUpdate(" + eventId + ")";
        }

        @Override
        public long getEventId() {
            return eventId;
        }
    }

    // CAL-2545: exchange sometimes don't set external_id for non-meeting events,
    // but if meeting is created from non-meeting event, it should contains real external_id
    // in new event data
    private static class NeedToUpdateWithNewExternalId extends EventInstanceStatusInfo {
        private final long eventId;
        private final String changedExternalId;
        private final boolean isExportedWithEws;

        public NeedToUpdateWithNewExternalId(long eventId, String changedExternalId, boolean isExportedWithEws) {
            this.eventId = eventId;
            this.changedExternalId = changedExternalId;
            this.isExportedWithEws = isExportedWithEws;
        }

        @Override
        public String toString() {
            return "needToUpdateWithNewExternalId(" + eventId + ", " + changedExternalId + ")";
        }

        @Override
        public long getEventId() {
            return eventId;
        }
    }

    private static class AlreadyUpdated extends EventInstanceStatusInfo {
        private final long eventId;

        public AlreadyUpdated(long eventId) {
            this.eventId = eventId;
        }

        @Override
        public String toString() {
            return "alreadyUpdated";
        }

        @Override
        public long getEventId() {
            return eventId;
        }
    }

    // we apply any decision from CalDAV, even outdated, otherwise user experience suffers :)
    private static class FromCaldavAlreadyUpdated extends EventInstanceStatusInfo {
        private final long eventId;

        public FromCaldavAlreadyUpdated(long eventId) {
            this.eventId = eventId;
        }

        @Override
        public String toString() {
            return "fromCaldavAlreadyUpdated";
        }

        @Override
        public long getEventId() {
            return eventId;
        }
    }

    public boolean isNotFound() {
        return this instanceof NotFound;
    }

    public boolean isNeedToUpdate() {
        return this instanceof NeedToUpdate;
    }

    public boolean isNeedToUpdateWithNewExternalId() {
        return this instanceof NeedToUpdateWithNewExternalId;
    }

    public boolean isAlreadyUpdated() {
        return this instanceof AlreadyUpdated;
    }

    public boolean isFromCaldavAlreadyUpdated() {
        return this instanceof FromCaldavAlreadyUpdated;
    }

}
