package ru.yandex.calendar.logic.ics;

import io.micrometer.core.instrument.MeterRegistry;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.calendar.logic.ics.exp.IcsEventExporter;
import ru.yandex.calendar.logic.ics.exp.IcsTodoExporter;
import ru.yandex.calendar.logic.ics.feed.IcsFeedDownloader;
import ru.yandex.calendar.logic.ics.feed.IcsFeedManager;
import ru.yandex.calendar.logic.ics.feed.IcsFeedUpdater;
import ru.yandex.calendar.logic.ics.feed.MixedIcsFeedDownloader;
import ru.yandex.calendar.logic.ics.feed.PlainIcsFeedDownloader;
import ru.yandex.calendar.logic.ics.feed.ZoraIcsFeedDownloader;
import ru.yandex.calendar.logic.ics.imp.IcsEventImporter;
import ru.yandex.calendar.logic.ics.imp.IcsEventReplyHandler;
import ru.yandex.calendar.logic.ics.imp.IcsImporter;
import ru.yandex.calendar.logic.ics.imp.IcsTodoImporter;
import ru.yandex.calendar.tvm.TvmClient;
import ru.yandex.calendar.util.HttpClientConfiguration;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@Configuration
public class IcsContextConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(IcsContextConfiguration.class);

    @Bean
    public IcsFeedManager icsFeedManager() {
        return new IcsFeedManager();
    }

    @Bean
    public IcsImporter icsImporter() {
        return new IcsImporter();
    }

    @Bean
    public IcsEventImporter icsEventImporter() {
        return new IcsEventImporter();
    }

    @Bean
    public IcsEventReplyHandler icsEventReplyHandler() {
        return new IcsEventReplyHandler();
    }

    @Bean
    public IcsTodoImporter icsTodoImporter() {
        return new IcsTodoImporter();
    }

    @Bean
    public IcsEventExporter icsEventExporter() {
        return new IcsEventExporter();
    }

    @Bean
    public IcsTodoExporter icsTodoExporter() {
        return new IcsTodoExporter();
    }

    @Bean
    public IcsFeedUpdater icsFeedUpdater() {
        return new IcsFeedUpdater();
    }

    @Bean
    public IcsFeedDownloader icsFeedDownloader(TvmClient client, MeterRegistry registry, @Value("${tvm.gozora.client-id}") int zoraClientId,
                                               @Value("${ics.plain.feed.pattern}") String icsPlainFeedPattern,
                                               @Value("${ics.feed.mode}") IcsFeedMode mode,
                                               @Value("${ics.zora.offline.hostname}") String zoraOfflineHostname,
                                               @Value("${ics.gozora.hostname}") String zoraOnlineHostname) {
        switch (mode) {
            case ZORA:
                logger.debug("Using zora feed downloader");
                return getZoraDownloader(client, zoraClientId, registry, zoraOfflineHostname, zoraOnlineHostname);
            case PLAIN:
                logger.debug("Using plain feed downloader");
                return getPlainDownloader();
            case MIXED:
                logger.debug("Using mixed feed downloader");
                return new MixedIcsFeedDownloader(
                        getPlainDownloader(),
                        getZoraDownloader(client, zoraClientId, registry, zoraOfflineHostname, zoraOnlineHostname),
                        icsPlainFeedPattern
                );
            default:
                throw new IllegalArgumentException("Parameter ics.feed.mode has incorrect value: " + mode);
        }
    }

    private static ZoraIcsFeedDownloader getZoraDownloader(TvmClient client, int zoraClientId, MeterRegistry registry,
                                                           String offlineHostname, String onlineHostname) {
        return new ZoraIcsFeedDownloader(
                HttpClientConfiguration.fromProperties("ics.zora"),
                client,
                zoraClientId,
                registry,
                offlineHostname,
                onlineHostname
        );
    }

    private static PlainIcsFeedDownloader getPlainDownloader() {
        return new PlainIcsFeedDownloader(HttpClientConfiguration.fromProperties("ics.plain"));
    }
}
