package ru.yandex.calendar.logic.ics;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.util.Arrays;
import java.util.Date;

import net.fortuna.ical4j.data.CalendarBuilder;
import net.fortuna.ical4j.model.Calendar;
import net.fortuna.ical4j.model.Component;
import net.fortuna.ical4j.model.DateTime;
import net.fortuna.ical4j.model.Period;
import net.fortuna.ical4j.model.Property;
import net.fortuna.ical4j.model.component.CalendarComponent;
import net.fortuna.ical4j.model.component.VEvent;
import net.fortuna.ical4j.model.property.Attendee;
import net.fortuna.ical4j.model.property.Categories;
import net.fortuna.ical4j.model.property.Method;
import net.fortuna.ical4j.model.property.Organizer;
import net.fortuna.ical4j.model.property.Uid;
import net.fortuna.ical4j.model.property.Url;
import net.fortuna.ical4j.util.CompatibilityHints;
import net.fortuna.ical4j.util.Uris;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.event.IcsEventSynchData;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVEvent;
import ru.yandex.calendar.logic.ics.iv5j.ical.parameter.IcsPartStat;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsAttendee;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsEmailPropertyBase;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsOrganizer;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.resources.UStringLiteral;
import ru.yandex.commune.mail.ContentType;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.InputStreamX;
import ru.yandex.misc.io.ReaderSource;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.parse.ByteReaderForParser;

/**
 * @author Stepan Koltsov
 */
public class IcsUtils {
    private static final Logger logger = LoggerFactory.getLogger(IcsUtils.class);

    // Ics feed properties
    static {
        // Should we use System.setProperty(...) also or instead?
        CompatibilityHints.setHintEnabled(CompatibilityHints.KEY_RELAXED_PARSING, true);
        CompatibilityHints.setHintEnabled(CompatibilityHints.KEY_RELAXED_UNFOLDING, true);
        //CompatibilityHints.setHintEnabled(CompatibilityHints.KEY_RELAXED_VALIDATION, true);
        //CompatibilityHints.setHintEnabled(CompatibilityHints.KEY_OUTLOOK_COMPATIBILITY, true);
    }

    private static final String NO_CAT_EN = "(none)"; // turns to our no-category

    public static final ContentType CALENDAR_MIME_TYPE = ContentType.valueOf("text/calendar");
    public static final ContentType CALENDAR_MIME_TYPE_FULL = ContentType.valueOf("text/calendar; charset=utf-8");

    public static Email getEmail(Property emailProperty) {
        return IcsEmailPropertyBase.fromIcal4j(emailProperty).getEmail();
    }

    public static Function<Attendee, Email> getAttendeeEmailF() {
        return new Function<Attendee, Email>() {
            public Email apply(Attendee attendee) {
                return IcsAttendee.fromIcal4j(attendee).getEmail();
            }
        };
    }

    public static Decision getDecision(IcsAttendee attendee) {
        final Option<IcsPartStat> partStat = attendee.getPartStat();
        if (partStat.isPresent()) {
            return Decision.findByIcsPartStat(partStat.get()).getOrElse(Decision.UNDECIDED);
        } else {
            return Decision.UNDECIDED;
        }
    }

    public static Function<Property, String> getPropertyValueF() {
        return new Function<Property, String>() {
            public String apply(Property a) {
                return a.getValue();
            }
        };
    }

    public static IcsCalendar parse(InputStream inputStream) {
        return parseBytesInGuessedEncoding(new InputStreamX(inputStream).readBytes());
    }

    public static IcsCalendar parse(Reader reader) {
        try {
            return IcsCalendar.fromIcal4j(new CalendarBuilder().build(reader));

        } catch (Exception t) {
            throw translate(t);
        }
    }

    public static IcsCalendar parse(InputStreamSource input) {
        return parse(new ByteArrayInputStream(input.readBytes()));
    }

    public static IcsCalendar parse(ReaderSource input) {
        return parse(new StringReader(input.readString()));
    }


    public static byte[] serializeToBytes(IcsCalendar calendar) {
        return CharsetUtils.encodeToArray(CharsetUtils.UTF8_CHARSET, calendar.serializeToString());
    }

    public static RuntimeException translate(Exception e) {
        if (e instanceof net.fortuna.ical4j.data.ParserException) {
            return CommandRunException.createSituation(e, Situation.ICS_PARSING_ERROR);
        } else {
            return ExceptionUtils.translate(e);
        }
    }

    public static Organizer organizer(Email email) {
        return new IcsOrganizer(email).toProperty();
    }

    public static Organizer organizer(String email) {
        return organizer(new Email(email));
    }

    public static Attendee attendee(Email email) {
        return new IcsAttendee(email).toProperty();
    }

    public static Attendee attendee(String email) {
        return attendee(new Email(email));
    }

    public static Attendee attendee(String email, Decision decision) {
        return new IcsAttendee(new Email(email), decision.getPartStat()).toProperty();
    }

    @SuppressWarnings("unchecked")
    public static <T extends Property> ListF<T> getProperties(Component component, String name) {
        return Cf.x(component.getProperties(name));
    }

    @SuppressWarnings("unchecked")
    public static ListF<Property> getProperties(Component component) {
        return Cf.x(component.getProperties());
    }

    public static ListF<Attendee> getAttendees(VEvent vevent) {
        return getProperties(vevent, Property.ATTENDEE);
    }

    public static Instant parseTimestamp(String string) {
        try {
            return new Instant(new DateTime(string).getTime());
        } catch (Exception e) {
            throw translate(e);
        }
    }

    public static Function<String, Instant> parseTimestampF() {
        return new Function<String, Instant>() {
            public Instant apply(String a) {
                return parseTimestamp(a);
            }
        };
    }

    public static DateTime toDateTime(Instant instant) {
        DateTime r = new DateTime(instant.getMillis());
        r.setUtc(true);
        return r;
    }

    public static Period period(Instant start, Instant end) {
        return new Period(toDateTime(start), toDateTime(end));
    }

    public static Instant getTs(DateTimeZone tz, Date date, boolean useLocalTz) {
        long res = date.getTime();
        // If for given instance we need to create a new one in user's timezone preserving time,
        // then we perform converting: ms -> local time (utc) -> same local time (given dtf tz).
        if (useLocalTz) {
            res = DateTimeFormatter.convertUTCToLocal(res, tz);
        }
        return new Instant(res);
    }

    public static Option<String> getPropIfSet(Property pValue) {
        return pValue != null ? Option.ofNullable(pValue.getValue()) : Option.<String>empty();
    }

    public static Option<String> getNotBlankValue(Property pValue) {
        return getPropIfSet(pValue).filterNot(StringUtils::isBlank);
    }

    public static String getFirstCategory(CalendarComponent cc) {
        // XXX should we iterate through getProperties() until we found a non-empty one?
        Categories catsProp = (Categories) cc.getProperty(Property.CATEGORIES);
        ListF<String> catsArr;
        if (
            catsProp == null || StringUtils.isEmpty(catsProp.getValue())
            || NO_CAT_EN.equals(catsProp.getValue()) // can be switched off
        ) {
            catsArr = Cf.list();
        } else {
            String str = catsProp.getValue();
            catsArr = Cf.list(str.split(","));
        }
        logger.debug("Categories array = " + catsArr);
        return catsArr.firstO().getOrElse(UStringLiteral.NO_CATEGORY);
    }

    public static Method getMethod(Calendar calendar) {
        return ObjectUtils.defaultIfNull(calendar.getMethod(), Method.REQUEST);
    }

    public static Url url(String url) {
        try {
            return new Url(Uris.create(url));
        } catch (Exception e2) {
            throw ExceptionUtils.translate(e2);
        }
    }

    public static Uid uid(String externalId) {
        return new Uid(externalId);
    }

    public static IcsCalendar parseBytesInGuessedEncoding(byte[] contents) {
        byte[] unfolded = unfold(contents);
        String encoding = ru.yandex.calendar.util.CharsetUtils.guessEncoding(unfolded);
        return parseBytesUnfolded(unfolded, encoding);
    }

    public static IcsCalendar parseBytes(byte[] contents, String encoding) {
        return parseBytesUnfolded(unfold(contents), encoding);
    }

    public static String unfoldUtf8(String contents) {
        return new String(unfold(contents.getBytes(CharsetUtils.UTF8_CHARSET)), CharsetUtils.UTF8_CHARSET);
    }

    public static byte[] unfold(byte[] contents) {
        ByteReaderForParser reader = new ByteReaderForParser(contents);
        ByteArrayOutputStream writer = new ByteArrayOutputStream();

        for (;;)  {
            while (consumeIcsSplitOptional(reader)) {}

            if (!reader.hasNext()) break;
            writer.write(reader.next());
        }

        return writer.toByteArray();
    }

    private static boolean consumeIcsSplitOptional(ByteReaderForParser reader) {
        int remaining = reader.remaining();

        if (remaining >= 2) {
            byte b1 = reader.lookahead(0);
            byte b2 = reader.lookahead(1);

            if ((b1 == '\r' || b1 == '\n') && (b2 == ' ' || b2 == '\t')) {
                reader.advance(2);
                return true;
            }

            if (remaining > 2) {
                byte b3 = reader.lookahead(2);

                if (b1 == '\r' && b2 == '\n' && (b3 == ' ' || b3 == '\t')) {
                    reader.advance(3);
                    return true;
                }
            }
        }

        return false;
    }

    private static IcsCalendar parseBytesUnfolded(byte[] contents, String encoding) {
        if (ru.yandex.calendar.util.CharsetUtils.hasUtf8Bom(contents)) {
            contents = Arrays.copyOfRange(contents, 3, contents.length);
        }

        try (InputStreamReader reader = new InputStreamReader(new ByteArrayInputStream(contents), encoding)) {
            return parse(reader);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }


    public static IcsEventSynchData createIcsSynchData(IcsVEvent ve, IcsVTimeZones tz) {
        Option<String> extId = ve.getUid();
        Option<Instant> recurId = ve.getRecurrenceIdInstant(tz);
        Option<Instant> lastModifiedO = ve.getLastModifiedInstant(tz);
        Option<Instant> dtstampO = ve.getDtStampInstant(tz);

        // CAL-2093, dtstamp is required field, but
        // sometimes we get ics without dtstamps
        Instant dtstamp = dtstampO.getOrElse(new Instant(0));

        // (RR) ssytnik@
        Instant lastModified = lastModifiedO.getOrElse(dtstamp);

        // TODO: to think, could be sequence null and what we should do in that case
        int seq = getPropIfSet(ve.toComponent().getSequence()).map(Cf.Integer::parse).getOrElse(0);
        return new IcsEventSynchData(extId, recurId, seq, lastModified, dtstamp);
    }

    public static IcsEventSynchData createIcsSynchData(EventData eventData) {
        Event event = eventData.getEvent();
        return new IcsEventSynchData(
                Option.of(eventData.getExternalId().get()),
                event.getRecurrenceId(),
                event.getSequence(), event.getLastUpdateTs(), event.getDtstamp().get());
    }

} //~
