package ru.yandex.calendar.logic.ics.exp;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVTimeZone;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsMethod;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsXWrCalname;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsXWrTimezone;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Stepan Koltsov
 */
public class IcsEventExportData {
    private static final Logger logger = LoggerFactory.getLogger(IcsEventExportData.class);

    private final IcsMethod method;
    private final String xwrTimezone;
    private final String xwrCalname;
    private final ListF<IcsSingleEventExportData> events;

    private IcsEventExportData(IcsMethod method, String xwrTimezone, String xwrCalname, ListF<IcsSingleEventExportData> events) {
        this.method = method;
        this.xwrCalname = xwrCalname;
        this.xwrTimezone = xwrTimezone;
        this.events = events;

    }

    public static class Heavy extends IcsEventExportData {
        private final Tuple2List<String, IcsVTimeZone> timezones;

        public Heavy(IcsMethod method, String xwrTimezone, String xwrCalname, ListF<IcsSingleEventExportData> events, Tuple2List<String, IcsVTimeZone> timezones) {
            super(method, xwrTimezone, xwrCalname, events);

            for (IcsSingleEventExportData event : events) {
                Validate.V.isTrue(event.isHeavy());
            }

            this.timezones = timezones;
        }


    }

    public static class Light extends IcsEventExportData {
        public Light(IcsMethod method, String xwrTimezone, String xwrCalname, ListF<IcsSingleEventExportData> events) {
            super(method, xwrTimezone, xwrCalname, events);
        }
    }

    public Heavy heavy() {
        return (Heavy) this;
    }

    /**
     * @see IcsEventGroupExportData#toCalendar()
     */
    public IcsCalendar toCalendar() {
        IcsCalendar calendar = IcsCalendar.fromIcal4j(IcsEventExporter.createCommonCalendarPart(method));
        calendar = calendar.addProperty(new IcsXWrTimezone(xwrTimezone));
        calendar = calendar.addProperty(new IcsXWrCalname(xwrCalname));
        calendar = calendar.addComponents(events.map(IcsSingleEventExportData.getVeventF()));

        // XXX use only for emails
        calendar = calendar.addComponents(heavy().timezones.get2());
        return calendar;
    }

    public ListF<IcsSingleEventExportData> getEvents() {
        return events;
    }

    public ListF<IcsEventGroupExportData> groups() {
        return events.groupBy(IcsSingleEventExportData.getExternalIdF()).entries().map(new Function2<String, ListF<IcsSingleEventExportData>, IcsEventGroupExportData>() {
            public IcsEventGroupExportData apply(String externalId, ListF<IcsSingleEventExportData> events) {
                Instant lastModified = events.map(IcsSingleEventExportData.getMainEventLastModifiedF()).max();
                if (IcsEventExportData.this instanceof Heavy) {
                    ListF<DateTimeZone> tzs = events.map(IcsSingleEventExportData.getTzF());
                    if (tzs.size() > 1) {
                        logger.warn("Events with id " + externalId + " has different timezones");
                    }
                    return new IcsEventGroupExportData.Heavy(externalId, lastModified, events.map(IcsSingleEventExportData.getVeventF()), tzs.first());
                } else {
                    return new IcsEventGroupExportData.Light(externalId, lastModified);
                }
            }
        });
    }

} //~
