package ru.yandex.calendar.logic.ics.exp;


import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import net.fortuna.ical4j.model.Calendar;
import net.fortuna.ical4j.model.ComponentList;
import net.fortuna.ical4j.model.Dur;
import net.fortuna.ical4j.model.ParameterList;
import net.fortuna.ical4j.model.Property;
import net.fortuna.ical4j.model.PropertyList;
import net.fortuna.ical4j.model.TemporalAmountAdapter;
import net.fortuna.ical4j.model.component.VAlarm;
import net.fortuna.ical4j.model.component.VEvent;
import net.fortuna.ical4j.model.parameter.Cn;
import net.fortuna.ical4j.model.parameter.CuType;
import net.fortuna.ical4j.model.parameter.Rsvp;
import net.fortuna.ical4j.model.parameter.XParameter;
import net.fortuna.ical4j.model.property.Action;
import net.fortuna.ical4j.model.property.Attendee;
import net.fortuna.ical4j.model.property.CalScale;
import net.fortuna.ical4j.model.property.Categories;
import net.fortuna.ical4j.model.property.Comment;
import net.fortuna.ical4j.model.property.Conference;
import net.fortuna.ical4j.model.property.Description;
import net.fortuna.ical4j.model.property.LastModified;
import net.fortuna.ical4j.model.property.Location;
import net.fortuna.ical4j.model.property.Organizer;
import net.fortuna.ical4j.model.property.ProdId;
import net.fortuna.ical4j.model.property.Sequence;
import net.fortuna.ical4j.model.property.Transp;
import net.fortuna.ical4j.model.property.Uid;
import net.fortuna.ical4j.model.property.Version;
import net.fortuna.ical4j.model.property.XProperty;
import one.util.streamex.StreamEx;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.frontend.caldav.proto.caldav.report.TimeRange;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.beans.generated.MainEventFields;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventGetProps;
import ru.yandex.calendar.logic.event.EventInfo;
import ru.yandex.calendar.logic.event.EventInfoDbLoader;
import ru.yandex.calendar.logic.event.EventLoadLimits;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.ExternalId;
import ru.yandex.calendar.logic.event.MainEventInfo;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.event.model.Priority;
import ru.yandex.calendar.logic.event.repetition.EventIndentAndRepetition;
import ru.yandex.calendar.logic.event.repetition.InfiniteInterval;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.PropertyNames;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVEvent;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVTimeZone;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsCreated;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsDtStamp;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsExDate;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsMethod;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsRDate;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsRRule;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsRecurrenceId;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsXWrCalname;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsXWrTimezone;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.dateTime.IcsDateTime;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.dateTime.IcsDateTimeFormats;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.IcsRecur;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.notification.Channel;
import ru.yandex.calendar.logic.notification.EventNotifications;
import ru.yandex.calendar.logic.notification.EventUserWithNotifications;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.participant.YandexUserParticipantInfo;
import ru.yandex.calendar.logic.svc.SvcRoutines;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.calendar.util.resources.UStringLiteral;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.regex.Pattern2;

import static java.util.function.Function.identity;

/**
 * @see IcsTodoExporter
 */
@Slf4j
public class IcsEventExporter {
    @Autowired
    private EventInfoDbLoader eventInfoDbLoader;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private SvcRoutines svcRoutines;
    @Autowired
    private UserManager userManager;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;

    private static final String OLD_ROOM_PHONES_PATTERN = "#\\d+|" + UStringLiteral.NO_PHONE; // CAL-6294
    private static final String ROOM_PHONES_PATTERN =
            "\\d+(?:, " + UStringLiteral.VIDEO_LC + " \\d+)?" +
                    "|" + UStringLiteral.VIDEO_LC + " \\d+" +
                    "|" + UStringLiteral.NO_PHONE_LC; // CAL-6491

    public static final Pattern2 ROOM_PHONES_IN_DESCRIPTION_PATTERN = Pattern2.compile(
            "(?:\\n\\n)?" + UStringLiteral.ROOM_PHONES + ":" +
                    "(?:\\n[^:\\n]+: (?:" + OLD_ROOM_PHONES_PATTERN + "|" + ROOM_PHONES_PATTERN + "))+");

    public static ListF<IcsVTimeZone> appendTimezoneIfNeeded(IcsVTimeZone vTimeZone) {
        boolean withTz = true;
        // Is a timezone information actually needed?
        // It seems to contain lots of needless data and be very big.
        if (withTz) {
            return Cf.list(vTimeZone);
        } else {
            return Cf.list();
        }
    }

    public static Calendar createCommonCalendarPart(IcsMethod method) {
        Calendar c = new Calendar();
        PropertyList pl = c.getProperties();
        pl.add(new ProdId("-//Yandex LLC//Yandex Calendar//EN"));
        pl.add(Version.VERSION_2_0);
        pl.add(CalScale.GREGORIAN);
        pl.add(method.toProperty());
        return c;
    }

    public IcsCalendar exportCalendar(PassportUid uid, long layerId, Instant now, ActionSource actionSource) {
        Validate.isTrue(actionSource.isWeb());

        Layer globalLayer = layerRoutines.getLayerById(layerId);
        String layerName = layerRoutines.evalLayerName(globalLayer, Option.empty());
        DateTimeZone layerTimezone = layerRoutines.getLayerTimezone(globalLayer);

        ListF<MainEvent> mainEvents;

        if (uid.isYandexTeamRu()) {
            Instant start = now.toDateTime(layerTimezone).weekOfWeekyear().roundFloorCopy().minusWeeks(2).toInstant();

            ListF<EventIndentAndRepetition> indents = eventInfoDbLoader.getEventIndentsOnLayers(Cf.list(layerId),
                    EventLoadLimits.intersectsInterval(new InfiniteInterval(start, Option.empty())));

            mainEvents = mainEventDao.findMainEventsByIds(indents.map(ei -> ei.getIndent().getMainEventId()));

        } else {
            mainEvents = mainEventDao.findMainEventsOnLayer(layerId);
        }

        ListF<EventInfo> eventInfos = getEventsOnLayerForExportAndMisses(uid, layerId, mainEvents, false, actionSource).get1();

        IcsExportParameters params = new IcsExportParameters(IcsExportMode.WEB, IcsMethod.PUBLISH, true, now);
        ListF<IcsSingleEventExportData> events = doExportEvents(uid, eventInfos, params);

        ListF<IcsVTimeZone> tzs = events.map(IcsSingleEventExportData.getTzF()).stableUnique()
                .map(AuxDateTime.getTzIdF().andThen(IcsTimeZones::icsVTimeZoneForIdFull));

        IcsCalendar calendar = IcsCalendar.fromIcal4j(createCommonCalendarPart(IcsMethod.PUBLISH));
        calendar = calendar.addComponents(tzs);
        calendar = calendar.addProperty(new IcsXWrTimezone(layerTimezone));
        calendar = calendar.addProperty(new IcsXWrCalname(layerName));
        calendar = calendar.addComponents(events.map(IcsSingleEventExportData.getVeventF()));

        return calendar;
    }

    private ListF<IcsEventGroupExportData> exportEventsGrouped(
            PassportUid uid, ListF<EventInfo> eventInfos, IcsExportParameters params)
    {
        return doExportEvents(uid, eventInfos, params)
            .groupBy(IcsSingleEventExportData::getExternalId)
            .mapEntries((externalId, events) -> {
                Instant lastModified = events.map(IcsSingleEventExportData::getMainEventLastModified).max();
                if (params.includeIcs()) {
                    ListF<DateTimeZone> tzs = events.map(IcsSingleEventExportData.getTzF()).stableUnique();
                    if (tzs.size() > 1) {
                        log.warn("Events with id {} have different timezones", externalId);
                    }
                    return new IcsEventGroupExportData.Heavy(externalId, lastModified,
                            events.map(IcsSingleEventExportData.getVeventF()), tzs.first());
                } else {
                    return new IcsEventGroupExportData.Light(externalId, lastModified);
                }
            });
    }

    private ListF<IcsSingleEventExportData> doExportEvents(
            PassportUid uid, ListF<EventInfo> eventInfos, IcsExportParameters params)
    {
        ListF<IcsSingleEventExportData> exports = Cf.arrayList();
        for (EventInfo eventInfo : eventInfos) {
            RepetitionInstanceInfo repetitionInfo = eventInfo.getRepetitionInstanceInfo();
            exports.add(doExportEvent(uid,
                    eventInfo.getEventWithRelations(),
                    repetitionInfo,
                    eventInfo.getEventUserWithNotifications().map(EventUserWithNotifications::getNotifications),
                    EventInstanceParameters.fromEvent(eventInfo.getEvent()),
                    params));
        }
        return exports;
    }

    @SuppressWarnings("unchecked")
    private IcsSingleEventExportData doExportEvent(
            PassportUid uid,
            EventWithRelations eventWithRelations,
            RepetitionInstanceInfo repetitionInstanceInfo,
            Option<EventNotifications> notifications,
            EventInstanceParameters eventParams, IcsExportParameters exportParams)
    {
        Event event = eventWithRelations.getEvent();
        Participants participants = eventWithRelations.getParticipants();

        IcsExportMode exportMode = exportParams.getMode();
        IcsMethod method = exportParams.getMethod();

        VEvent vevent = createExportEvent(eventWithRelations, eventParams).toComponentForSerialization();

        String externalId = eventWithRelations.getMainEvent().getExternalId();

        vevent.getProperties().add(new Uid(externalId));
        vevent.getProperties().add(new Sequence(event.getSequence()));

        // incomprehensible definition: http://tools.ietf.org/html/rfc5545#section-3.8.7.2
        // some clues to what it might mean: http://www.ietf.org/mail-archive/web/calsify/current/msg00433.html
        // :)
        vevent.getProperties().add(new IcsDtStamp(exportParams.getNow()).toPropertyForSerialization());

        vevent.getProperties().add(new IcsCreated(event.getCreationTs()).toPropertyForSerialization());

        if (!passportAuthDomainsHolder.containsYandexTeamRu()) {
            Option<String> location = StringUtils.notBlankO(event.getLocation());
            Option<String> resources = resourceRoutines.locationStringWithResourceNames(uid, participants);

            if (location.isPresent() || resources.isPresent() || exportMode == IcsExportMode.EMAIL) {
                vevent.getProperties().add(new Location(resources.plus(location).mkString(", ")));
            }
        } else if (StringUtils.isNotEmpty(event.getLocation())) {
            vevent.getProperties().add(new Location(event.getLocation()));
        } else {
            Option<String> resourcesNames = resourceRoutines.locationStringWithResourceNames(uid, participants);
            if (resourcesNames.isPresent()) {
                vevent.getProperties().add(new Location(resourcesNames.get()));
            } else if (exportMode == IcsExportMode.EMAIL) {
                vevent.getProperties().add(new Location(""));
            }
        }
        ListF<ResourceInfo> resources = eventWithRelations.getResources();
        Option<String> description = StringUtils.notEmptyO(event.getDescription());

        if (exportParams.getMode() == IcsExportMode.CALDAV && resources.isNotEmpty()) { // CAL-6294
            ListF<String> namesWithPhones = Cf.arrayListWithCapacity(resources.size());
            ListF<ResourceInfo> rooms = resources.filter(ResourceInfo.isMeetingRoomF());

            for (ResourceInfo r : resourceRoutines.sortResourcesFromUserOfficeAndCityFirst(uid, rooms)) {
                Option<String> phone = r.getPhone();
                Option<String> video = r.getVideo().map(s -> UStringLiteral.VIDEO_LC + " " + s);
                String phones = StringUtils.defaultIfEmpty(phone.plus(video).mkString(", "), UStringLiteral.NO_PHONE_LC);

                namesWithPhones.addAll(r.getNameWithAlterName().map(s -> s + (": " + phones)));
            }
            if (namesWithPhones.isNotEmpty()) {
                String phones = namesWithPhones.mkString(UStringLiteral.ROOM_PHONES + ":\n", "\n", "");
                // @see IcsEventImporter#restoreChangedByExportEventFields
                Check.some(ROOM_PHONES_IN_DESCRIPTION_PATTERN.findFirst(phones));

                description = Option.of(description.plus(phones).mkString("\n\n"));
            }
        }
        if (description.isPresent()) {
            vevent.getProperties().add(new Description(description.get()));
        } else if (exportMode == IcsExportMode.EMAIL) {
            vevent.getProperties().add(new Description(""));
        }

        if (StringUtils.isNotEmpty(event.getUrl().getOrNull())) {
            vevent.getProperties().add(IcsUtils.url(event.getUrl().getOrNull()));
        } else { // CAL-6294
            var calendarUrlPrefix = exportParams.getSettings()
                    .map(svcRoutines::getEventPageUrlPrefixBySettings)
                    .getOrElse(() -> svcRoutines.getEventPageUrlPrefixForUid(uid));
            vevent.getProperties().add(IcsUtils.url(calendarUrlPrefix + event.getId()));
        }
        // User settings (event-user)
        if (!eventParams.getOccurrenceId().isPresent()) {

            if (repetitionInstanceInfo.getRepetition().isPresent()) {
                exportRepetition(vevent.getProperties(), eventWithRelations, repetitionInstanceInfo.getRepetition().get());
            }

            exportRdates(vevent.getProperties(), eventWithRelations, true, repetitionInstanceInfo.getRdates());
            exportRdates(vevent.getProperties(), eventWithRelations, false, repetitionInstanceInfo.getExdates());
        }
        // Recurrence id
        exportRecurrenceId(vevent.getProperties(), eventWithRelations, eventParams);

        Option<EventUser> eventUser = eventWithRelations.findUserEventUser(uid);
        // FUTURE: support additional fields (availability - DONE, is_task/completion)
        if (eventUser.isPresent()) {
            exportEventUser(vevent.getProperties(), eventUser.get(), exportMode);
        }

        if (notifications.isPresent()) {
            vevent.getAlarms().addAll(createValarms(event, notifications.get().getNotifications()));
        }

        if (!method.sameMethodAs(IcsMethod.REPLY)) {
            eventWithRelations.findOwnUserLayer(uid).forEach(l ->
                    vevent.getProperties().add(new Categories(layerRoutines.evalLayerNameBySettings(l, exportParams.getSettings()))));
        }

        if (participants.isMeeting()) {
            vevent.getProperties().add(exportOrganizer(participants.getOrganizer()));
            if (method.sameMethodAs(IcsMethod.REPLY)) {
                ParticipantInfo attendee = participants.getParticipantByUid(uid)
                        .getOrThrow("Invitation not found by uid " + uid);

                if (StringUtils.isNotEmpty(attendee.getReason())) {
                    vevent.getProperties().add(new Comment(attendee.getReason()));
                }
                vevent.getProperties().add(exportAttendee(attendee));

            } else {
                var savedAttendeesId = exportParams.getExtraId(event.getId());
                if (savedAttendeesId.isPresent()) {
                    vevent.getProperties().add(new XProperty(PropertyNames.X_YANDEX_MESSAGE_EXTRA_ID, savedAttendeesId.get().toString()));
                } else {
                    vevent.getProperties().addAll(exportAttendees(participants.getAllAttendees()));
                }

                if (exportMode == IcsExportMode.CALDAV && participants.getOrganizer().getId().isYandexUserWithUid(uid)) {
                    vevent.getProperties().addAll(exportPrivateComments(participants.getAllAttendees()));
                }
            }

            if (method.sameMethodAs(IcsMethod.REQUEST) && !userManager.isYamoneyUser(uid)) {
                // http://ml.yandex-team.ru/thread/2080000000744974906/
                vevent.getProperties().add(new XProperty("X-MICROSOFT-CDO-BUSYSTATUS", "TENTATIVE"));
            }
        }

        if (StringUtils.isNotEmpty(event.getConferenceUrl().getOrNull())) {
            vevent.getProperties().add(new Conference(new ParameterList(), event.getConferenceUrl().getOrNull()));
        }


        vevent.getProperties().add(new LastModified(IcsUtils.toDateTime(event.getLastUpdateTs())));

        IcsVEvent icsVevent = IcsVEvent.fromIcal4j(vevent);
        Instant mainEventLastUpdateTs = eventWithRelations.getMainEvent().getLastUpdateTs();

        if (exportParams.includeIcs()) {
            return new IcsSingleEventExportData.Heavy(icsVevent, mainEventLastUpdateTs,
                    eventWithRelations.getTimezone(), externalId);
        } else {
            return new IcsSingleEventExportData.Light(mainEventLastUpdateTs, externalId);
        }
    }

    private Attendee exportAttendee(ParticipantInfo inv) {
        return exportAttendees(Cf.list(inv)).single();
    }

    public static ListF<Attendee> exportAttendees(ListF<ParticipantInfo> invList) {
        return invList.map(new Function<ParticipantInfo, Attendee>() {
            public Attendee apply(ParticipantInfo inv) {
                String attName = inv.getName();
                Decision attDecision = inv.getDecision();
                return exportAttendee(inv.getEmail(), attName, inv.getCuType(), attDecision, false);
            }
        });
    }

    private ListF<XProperty> exportPrivateComments(ListF<ParticipantInfo> attendees) {
        return attendees.filterMap(new Function<ParticipantInfo, Option<XProperty>>() {
            public Option<XProperty> apply(ParticipantInfo p) {
                if (!(p instanceof YandexUserParticipantInfo)) return Option.empty();

                EventUser eu = ((YandexUserParticipantInfo) p).getEventUser();
                Option<String> reason = eu.getReason().filter(StringUtils::isNotBlank);

                if (!reason.isPresent()) return Option.empty();

                ParameterList pl = new ParameterList();
                pl.add(new XParameter(PropertyNames.X_CALENDARSERVER_DTSTAMP,
                        IcsDateTimeFormats.formatDateTime(eu.getDtstamp().getOrElse(Instant.now()))));
                pl.add(new XParameter(PropertyNames.X_CALENDARSERVER_ATTENDEE_REF,
                        "\"" + Emails.getUnicodedMailto(p.getEmail()) + "\""));

                return Option.of(new XProperty(PropertyNames.X_CALENDARSERVER_ATTENDEE_COMMENT, pl, reason.get()));
            }
        });
    }

    private static String exportName(String userName, Email email) {
        if (StringUtils.isNotBlank(userName)) {
            return userName;
        } else {
            return email.getLocalPart();
        }
    }

    private Organizer exportOrganizer(ParticipantInfo organizer) {
        Organizer icsOrganizer = IcsUtils.organizer(organizer.getEmail());
        icsOrganizer.getParameters().add(new Cn(exportName(organizer.getName(), organizer.getEmail())));
        return icsOrganizer;
    }

    public CuType cuType(Email email) {
        return resourceRoutines.isResource(email) ? CuType.ROOM : CuType.INDIVIDUAL;
    }

    private static IcsVEvent createExportEvent(
            EventWithRelations e, EventInstanceParameters exportParameters)
    {
        Instant startTs = exportParameters.getStartTs();
        Instant endTs = exportParameters.getEndTs();

        IcsVEvent ve = new IcsVEvent();
        if (e.getEvent().getIsAllDay()) {
            ve = ve.withDtStart(startTs.toDateTime(e.getTimezone()).toLocalDate());
            ve = ve.withDtEnd(endTs.toDateTime(e.getTimezone()).toLocalDate());
        } else {
            ve = ve.withDtStart(startTs.toDateTime(e.getTimezone()));
            ve = ve.withDtEnd(endTs.toDateTime(e.getTimezone()));
        }
        ve = ve.withSummary(e.getEvent().getName());
        return ve;
    }

    private static void exportRecurrenceId(
            PropertyList pl, EventWithRelations e, EventInstanceParameters extportParameters)
    {
        Instant recurTS = extportParameters.getOccurrenceId().getOrNull();
        if (recurTS != null) {
            IcsRecurrenceId recurrenceId = e.getEvent().getIsAllDay() ?
                    new IcsRecurrenceId(recurTS.toDateTime(e.getTimezone()).toLocalDate()) :
                    new IcsRecurrenceId(recurTS.toDateTime(e.getTimezone()));
            // akirakozov (02.10.09) :
            // many popular calendar agents (google, outlook)
            // doesn't support THISANDFUTURE property
            // if (extData.getApplyToFuture()) { recurrenceId.getParameters().add(Range.THISANDFUTURE); }
            pl.add(recurrenceId.toPropertyForSerialization());
        }
    }

    public static Attendee exportAttendee(Email email, String name, CuType cuType, Decision decision, boolean needRsvp) {
        Attendee attendee = IcsUtils.attendee(email);
        ParameterList pl = attendee.getParameters();
        if (cuType != CuType.INDIVIDUAL)
            pl.add(cuType);
        pl.add(decision.getPartStat().toParameter());
        pl.add(new Cn(exportName(name, email)));
        // other: Member, RSVP, DelegatedTo, DelegatedFrom, SentBy, Dir, Language
        if (needRsvp) {
            pl.add(new Rsvp(true));
        }
        // TODO: do we need SENT-BY (if inv.creator_uid != event.creator_uid)
        return attendee;
    }

    private static void exportRepetition(PropertyList eventPl, EventWithRelations e, Repetition r) {
        IcsRecur recur = RepetitionUtils.getRecur(e, r);
        eventPl.add(new IcsRRule(recur).toPropertyForSerialization());
    }

    private void exportRdates(
            PropertyList eventPropertyList, EventWithRelations e, boolean isRdate, ListF<Rdate> rdates)
    {
        // Must output rdates and exdates without grouping (one per line).
        // Date-times must be in UTC (and with 'Z' suffix).
        // Seems like there is a bug in both iCal and Lightning,
        // that fail to parse records otherwize.

        for (Rdate rdate : rdates) {
            final Property property;

            if (rdate.getEndTs().isPresent()) {
                Validate.isTrue(isRdate, "exdate cannot contain period");
                DateTime start = rdate.getStartTs().toDateTime(e.getTimezone());
                DateTime end = rdate.getEndTs().get().toDateTime(e.getTimezone());
                property = new IcsRDate(IcsDateTime.dateTime(start), IcsDateTime.dateTime(end)).toPropertyForSerialization();
            } else {
                IcsDateTime dateTime = e.getEvent().getIsAllDay()
                        ? IcsDateTime.localDate(new LocalDate(rdate.getStartTs(), e.getTimezone()))
                        : IcsDateTime.dateTime(rdate.getStartTs().toDateTime(e.getTimezone()));

                property = (isRdate ? new IcsRDate(dateTime) : new IcsExDate(dateTime)).toPropertyForSerialization();
            }
            eventPropertyList.add(property);
        }
    }

    private static void exportEventUser(PropertyList pl, EventUser eu, IcsExportMode exportMode) {
        Option<Priority> priorityO = eu.getFieldValueO(EventUserFields.PRIORITY);
        if (priorityO.isPresent()) {
            Priority priority = priorityO.get();
            // XXX calendar for now does not distinguish between UNDEFINED
            // and MEDIUM (NORMAL) priority values. See CAL-2453 for details.
            if (Priority.NORMAL != priority) {
                pl.add(priority.toIcalPriority());
            }
        }

        Option<Availability> availabilityO = eu.getFieldValueO(EventUserFields.AVAILABILITY);
        if (availabilityO.isPresent()) {
            Option<Transp> transpO = availabilityO.get().toTransp();
            if (transpO.isPresent()) {
                pl.add(transpO.get());
            }
        }

        if (eu.getIcalXMozLastack().isPresent()) {
            pl.add(new XProperty(
                    PropertyNames.X_MOZ_LASTACK,
                    IcsDateTimeFormats.formatDateTime(eu.getIcalXMozLastack().get())));
            }
        if (eu.getIcalXMozSnoozeTime().isPresent()) {
            pl.add(new XProperty(
                    PropertyNames.X_MOZ_SNOOZE_TIME,
                    IcsDateTimeFormats.formatDateTime(eu.getIcalXMozSnoozeTime().get())));
        }
        if (exportMode == IcsExportMode.CALDAV && eu.getReason().exists(StringUtils::isNotBlank)) {
            pl.add(new XProperty(PropertyNames.X_CALENDARSERVER_PRIVATE_COMMENT, eu.getReason().get()));
        }
    }

    private static ComponentList createValarms(Event e, ListF<Notification> notifications) {
        ComponentList valarms = new ComponentList();

        Instant eStartTs = e.getStartTs();
        for (Notification notification : notifications) {
            Instant ntfTs = eStartTs.plus(notification.getOffset());
            Dur dur = new Dur(new java.util.Date(eStartTs.getMillis()), new java.util.Date(ntfTs.getMillis()));
            Channel channel = notification.getChannel();

            if (Channel.AUDIO == channel) {
                valarms.add(createValarm(Action.AUDIO, dur));
            } else if (Channel.DISPLAY == channel) {
                // XXX wrong:
                // the alarm MUST also include a "DESCRIPTION" property
                // http://tools.ietf.org/html/rfc5545#section-3.6.6
                valarms.add(createValarm(Action.DISPLAY, dur));
            }
        }
        return valarms;
    }

    private static VAlarm createValarm(Action action, Dur dur) {
        VAlarm valarm = new VAlarm(TemporalAmountAdapter.from(dur).getDuration());
        PropertyList propertyList = valarm.getProperties();
//      // This was a simple case, no RELATED parameter
//      // ('START' must be assumed). COULD BE:
//      Trigger trigger = new Trigger(dur);
//      trigger.getParameters().add(Related.START);
//      VAlarm va = new VAlarm();
//      PropertyList vaPl = va.getProperties();
//      vaPl.add(trigger);
        propertyList.add(action);
        // NOTE: to add a repeat, you should have done:
        //vaPl.add(new Repeat(1));
        //vaPl.add(new Duration(ntfTs, nextNtfTs));
        return valarm;
    }

    public IcsCalendar exportEvents(
            PassportUid uid, ListF<EventInfo> events, IcsExportParameters params)
    {
        Validate.hasSize(1, events.map(EventInfo::getMainEventId).unique());

        return exportEventsGrouped(uid, events, params)
                .single().toCalendarO(params.getMethod(), params.getMode() != IcsExportMode.EMAIL).get();
    }

    public IcsCalendar exportEvent(
            PassportUid uid, EventWithRelations event, RepetitionInstanceInfo repetitionInfo,
            Option<EventNotifications> notifications,
            EventInstanceParameters eventParms, IcsExportParameters exportParams)
    {
        return doExportEvent(uid, event, repetitionInfo, notifications, eventParms, exportParams)
                .toCalendar(exportParams.getMethod(), exportParams.getMode() != IcsExportMode.EMAIL);
    }

    public List<IcsEventGroupExportData> exportEventsByExternalIdsForCaldav(
            PassportUid uid, long layerId, ListF<ExternalId> externalIds, IcsExportOptions options, Instant now)
    {
        ListF<MainEvent> mainEvents = mainEventDao.findMainEventsOnLayer(layerId, externalIds);
        ListF<EventInfo> events = getEventsOnLayerForExportAndMisses(
                uid, layerId, mainEvents, options.includeDeclined(), ActionSource.CALDAV).get1();

        IcsExportParameters exportParams = new IcsExportParameters(
                IcsExportMode.CALDAV, IcsMethod.PUBLISH, options.includeIcs(), now);

        ListF<IcsEventGroupExportData> groups = exportEventsGrouped(uid, events, exportParams);

        MapF<String, IcsEventGroupExportData> byExternalIdNormalized = groups.toMapMappingToKey(
                IcsEventGroupExportData.getExternalIdF().andThen(ExternalId.getNormalizedF()));

        return StreamEx.of(externalIds)
                .map(ExternalId::getNormalized)
                .flatCollection(byExternalIdNormalized::getO)
                .toImmutableList();
    }

    public List<IcsEventGroupExportData> exportEventsOnLayerForCaldav(
            PassportUid uid, long layerId, IcsExportOptions options, TimeRange timeRange, Instant now)
    {
        val events = exportEventsOnLayerForCaldav(uid, layerId, options, timeRange, Option.empty(), now);
        return StreamEx.of(events)
                .map(IcsEventGroup::getExport)
                .flatMap(Optional::stream)
                .toImmutableList();
    }

    public ListF<IcsEventGroup> exportEventsOnLayerForCaldavCreatedOrModifiedSince(
            PassportUid uid, long layerId, Instant modifiedSince, IcsExportOptions options, Instant now)
    {
        return exportEventsOnLayerForCaldav(uid, layerId, options, TimeRange.unlimited(), Option.of(modifiedSince), now);
    }

    private ListF<IcsEventGroup> exportEventsOnLayerForCaldav(
            PassportUid uid, long layerId, IcsExportOptions options,
            TimeRange timeRange, Option<Instant> modifiedSince, Instant now)
    {
        if (!options.includeIcs()) {
            return getLightGroupsOnLayer(
                    uid, layerId, timeRange, modifiedSince, options.includeDeclined(), ActionSource.CALDAV);

        } else {
            ListF<MainEvent> mainEvents = getMainEventsOnLayer(layerId, timeRange, modifiedSince);
            Map<String, Instant> mapExternalIdsToLastUpdate = mainEvents.stream()
                    .collect(Collectors.toMap(MainEvent::getExternalId, MainEvent::getLastUpdateTs, (o, n) -> n.isAfter(o) ? n : o));

            Tuple2<ListF<EventInfo>, ListF<ExternalId>> eventsAndMisses = getEventsOnLayerForExportAndMisses(
                    uid, layerId, mainEvents, options.includeDeclined(), ActionSource.CALDAV);

            IcsExportParameters exportParams = new IcsExportParameters(
                    IcsExportMode.CALDAV, IcsMethod.PUBLISH, options.includeIcs(), now);

            val misses = StreamEx.of(eventsAndMisses.get2())
                    .mapToEntry(identity(), ExternalId::getRaw)
                    .mapValues(mapExternalIdsToLastUpdate::get)
                    .mapKeyValue(IcsEventGroup::miss)
                    .toImmutableList();

            return exportEventsGrouped(uid, eventsAndMisses.get1(), exportParams).map(IcsEventGroup::export)
                    .plus(misses);
        }
    }

    private ListF<MainEvent> getMainEventsOnLayer(long layerId, TimeRange timeRange, Option<Instant> modifiedSince) {
        EventLoadLimits limits = timeRange.toEventLoadLimits()
                .withModifiedSince(modifiedSince);

        return limits.hasTimeLimits()
                ? eventInfoDbLoader.getMainEventsOnLayer(layerId, limits)
                : modifiedSince.isPresent()
                ? mainEventDao.findMainEventsOnLayerModifiedSince(layerId, modifiedSince.get())
                : mainEventDao.findMainEventsOnLayer(layerId);
    }

    private ListF<IcsEventGroup> getLightGroupsOnLayer(
            PassportUid uid, long layerId, TimeRange timeRange, Option<Instant> modifiedSince,
            boolean includeDeclined, ActionSource actionSource)
    {
        EventGetProps egp = EventGetProps.none().loadMainEventField(
                Cf.list(MainEventFields.EXTERNAL_ID, MainEventFields.LAST_UPDATE_TS));

        EventLoadLimits limits = timeRange.toEventLoadLimits()
                .withModifiedSince(modifiedSince);

        ListF<EventInfo> events = eventInfoDbLoader.getEventInfosOnLayer(
                Option.of(uid), egp, layerId, limits, actionSource);

        PassportUid layerCreatorUid = layerRoutines.getLayerById(layerId).getCreatorUid();

        SetF<Long> declinedEventIds = eventUserDao.findEventUserEventIds(SqlCondition.trueCondition()
                .and(EventUserFields.EVENT_ID.column().inSet(events.map(EventInfo::getEventId)))
                .and(EventUserFields.UID.column().eq(layerCreatorUid))
                .and(layerCreatorUid.sameAs(uid) && includeDeclined
                        ? EventUserFields.DECISION.eq(Decision.NO).and(EventUserFields.IS_ATTENDEE.eq(false))
                        : EventUserFields.DECISION.eq(Decision.NO))).unique();

        Function1B<EventInfo> isVisible = ei -> ei.mayView() && !declinedEventIds.containsTs(ei.getEventId());

        return events.groupBy(e -> e.getMainEvent().getExternalId()).values().filterMap(group -> {
            MainEvent me = group.first().getMainEvent();

            val lastUpdateTs = me.getLastUpdateTs();

            final IcsEventGroup result;
            if (group.exists(isVisible)) {
                result = IcsEventGroup.export(new IcsEventGroupExportData.Light(me.getExternalId(), lastUpdateTs));
            } else {
                result = IcsEventGroup.miss(new ExternalId(me.getExternalId()), lastUpdateTs);
            }
            return Option.of(result);
        });
    }

    private Tuple2<ListF<EventInfo>, ListF<ExternalId>> getEventsOnLayerForExportAndMisses(
            PassportUid uid, long layerId, ListF<MainEvent> mainEvents,
            boolean includeDeclined, ActionSource actionSource)
    {
        EventGetProps egp = EventGetProps.any()
                .loadByLayerId(layerId)
                .excludeSubscribers();

        ListF<MainEventInfo> mainEventInfos =
                eventInfoDbLoader.getMainEventInfos(Option.of(uid), mainEvents, egp, actionSource);

        Function1B<EventInfo> isVisible = ei -> {
            EventWithRelations e = ei.getEventWithRelations();

            return ei.mayView() && e.existsOnLayerWithId(layerId)
                    && !e.findLayerById(layerId).exists(l -> e.findUserEventUser(l.getCreatorUid())
                            .exists(eu -> eu.getDecision() == Decision.NO
                                    && !(includeDeclined && l.getCreatorUid().sameAs(uid) && eu.getIsAttendee())));
        };

        MapF<Long, ListF<Instant>> exByMainId = Cf.hashMap();

        mainEventInfos.iterator().forEachRemaining(me -> me.getRecurrenceEventInfos().iterator()
                .filterNot(isVisible).forEachRemaining(ei ->
                        exByMainId.getOrElseUpdate(ei.getMainEventId(), Cf::arrayList).addAll(ei.getRecurrenceId())));

        ListF<EventInfo> events = mainEventInfos.iterator().flatMap(
                me -> me.getMasterEventInfos().iterator().filterMap(e -> Option.when(isVisible.apply(e),
                        e.plusExdates(exByMainId.getOrElse(e.getMainEventId(), Cf.list())))).iterator()

                        .plus(me.getRecurrenceEventInfos().iterator().filter(isVisible))).toList();

        ListF<ExternalId> misses = mainEventInfos.filterMap(me ->
                Option.when(!me.getEventInfos().exists(isVisible), new ExternalId(me.getMainEvent().getExternalId())));

        return Tuple2.tuple(events, misses);
    }
}
