package ru.yandex.calendar.logic.ics.exp;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.frontend.caldav.proto.facade.ExportOptions;
import ru.yandex.calendar.logic.event.ExternalId;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVEvent;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVTimeZone;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsMethod;

/**
 * @see IcsEventExportData
 * @see IcsSingleEventExportData
 */
public class IcsEventGroupExportData {

    private final ExternalId externalId;
    private final Instant lastModified;

    private IcsEventGroupExportData(String externalId, Instant lastModified) {
        this.externalId = new ExternalId(externalId);
        this.lastModified = lastModified;
    }

    public static class Heavy extends IcsEventGroupExportData {
        private final ListF<IcsVEvent> vevents;
        private final DateTimeZone tz;

        public Heavy(String externalId, Instant lastModified, ListF<IcsVEvent> vevents, DateTimeZone tz) {
            super(externalId, lastModified);
            this.vevents = vevents;
            this.tz = tz;
        }

        public IcsCalendar toCalendar(IcsMethod method, ExportOptions options) {
            IcsCalendar calendar = IcsCalendar.fromIcal4j(IcsEventExporter.createCommonCalendarPart(method));

            if (options.getEtcGmtForTimezones().contains(tz.getID())) {
                IcsVTimeZone fixedTimezone = IcsTimeZones.icsVTimeZoneForUtcOffset(tz.getOffset(Instant.now()));
                return calendar
                        .addComponent(fixedTimezone)
                        .addComponents(getVevents().map(e -> e.withTzIdParametersValue(fixedTimezone.getTzId())));
            }
            calendar = calendar.addComponent(options.isWithFullTimezones()
                    ? IcsTimeZones.icsVTimeZoneForIdFull(getTz().getID())
                    : IcsTimeZones.icsVTimeZoneForIdForOutlook(getTz().getID()));

            calendar = calendar.addComponents(getVevents());

            return calendar;
        }

        public ListF<IcsVEvent> getVevents() {
            return vevents;
        }

        public DateTimeZone getTz() {
            return tz;
        }
    }

    public static class Light extends IcsEventGroupExportData {
        public Light(String externalId, Instant lastModified) {
            super(externalId, lastModified);
        }
    }

    public static Function2<String, Instant, IcsEventGroupExportData> lightF() {
        return new Function2<String, Instant, IcsEventGroupExportData>() {
            public IcsEventGroupExportData apply(String s, Instant i) {
                return new Light(s, i);
            }
        };
    }

    public ExternalId getExternalId() {
        return externalId;
    }

    public Instant getLastModified() {
        return lastModified;
    }

    public static Function<IcsEventGroupExportData, ExternalId> getExternalIdF() {
        return new Function<IcsEventGroupExportData, ExternalId>() {
            public ExternalId apply(IcsEventGroupExportData d) {
                return d.getExternalId();
            }
        };
    }

    public Heavy heavy() {
        return (Heavy) this;
    }

    public Option<IcsCalendar> toCalendarO(ExportOptions options) {
        return toCalendarO(IcsMethod.PUBLISH, options);
    }

    public Option<IcsCalendar> toCalendarO(IcsMethod method, boolean withFullTimezones) {
        return toCalendarO(method, new ExportOptions(false, false, withFullTimezones, Cf.set()));
    }

    /**
     * @see IcsEventExportData#toCalendar()
     */
    public Option<IcsCalendar> toCalendarO(IcsMethod method, ExportOptions options) {
        if (this instanceof Heavy) {
            return Option.of(heavy().toCalendar(method, options));
        } else {
            return Option.empty();
        }
    }

} //~
