package ru.yandex.calendar.logic.ics.exp;

import java.net.URI;

import net.fortuna.ical4j.model.WeekDay;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVEvent;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsMethod;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsRRule;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsXWrCalname;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.dateTime.IcsDateTimeFormats;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.Freq;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.IcsRecur;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartByDay;
import ru.yandex.calendar.util.dates.WeekdayConv;
import ru.yandex.commune.holidays.DayType;
import ru.yandex.commune.holidays.Holiday;
import ru.yandex.commune.holidays.HolidaysData;
import ru.yandex.inside.geobase.GeobaseIds;
import ru.yandex.misc.enums.EnumResolver;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.test.Assert;

/**
 * @author dbrylev
 */
public class IcsHolidaysExporter {

    public static final String URL_PREFIX = "holidays://";

    public enum Country {
        RUSSIA(GeobaseIds.RUSSIA, "Праздники России"),
        CRIMEA(GeobaseIds.CRIMEA, "Празники Республики Крым"),
        TATARSTAN(GeobaseIds.TATARSTAN, "Праздники Республики Татарстан"),
        UKRAINE(GeobaseIds.UKRAINE, "Праздники Украины"),
        BELARUS(GeobaseIds.BELARUS, "Праздники Беларусии"),
        KAZAKHSTAN(GeobaseIds.KAZAKHSTAN, "Праздники Казахстана"),
        TURKEY(GeobaseIds.TURKEY, "Turkey holidays"),
        USA(GeobaseIds.USA, "USA holidays"),
        GERMANY(GeobaseIds.GERMANY, "Germany holidays"),
        NETHERLANDS(GeobaseIds.NETHERLANDS, "Netherlands holidays"),
        FINLAND(GeobaseIds.FINLAND, "Finland holidays"),
        ISRAEL(GeobaseIds.ISRAEL, "Israel holidays"),
        ;

        public final int geoId;
        public final String layerName;

        public static final EnumResolver<Country> R = EnumResolver.er(Country.class);

        Country(int geoId, String layerName) {
            this.geoId = geoId;
            this.layerName = layerName;
        }
    }

    public static IcsCalendar export(String url) {
        URI uri = UrlUtils.uri(url);

        Assert.equals("holidays", uri.getScheme());
        boolean outLabels = UrlUtils.getQueryParameterFromUrl(url, "labels").isPresent();

        Country country = Country.R.valueOf(uri.getHost());

        ListF<Holiday> holidays = HolidaysData.getHolidays()
                .filter(h -> h.getCountryId() == country.geoId
                        && (h.isOverride() && h.getDayType() == DayType.HOLIDAY || outLabels && h.isLabel()));

        IcsCalendar calendar = IcsCalendar.fromIcal4j(IcsEventExporter.createCommonCalendarPart(IcsMethod.PUBLISH));

        calendar = calendar.addProperty(new IcsXWrCalname(country.layerName));
        calendar = calendar.addComponents(holidays.map(day -> {
            IcsVEvent event = new IcsVEvent();

            LocalDate start = day.getLocalDate(day.getStartYear().getOrElse(2000));

            event = event.withUid(country.geoId
                    + (day.isLabel() ? "l" : "h")
                    + IcsDateTimeFormats.DATE_FORMATTER.print(start) + "yandex.ru");

            event = event.withDtStart(start);
            event = event.withDtEnd(start);
            event = event.withSummary(day.getName());

            event = event.withSequenece(0);
            event = event.withDtStamp(Instant.now());

            if (!day.getStartYear().exists(day.getEndYear()::isSome)) {
                IcsRecur recur;

                if (day.getDay().get2().isPresent()) {
                    recur = new IcsRecur(Freq.MONTHLY).withInterval(12);

                    recur = recur.withPart(new IcsRecurRulePartByDay(Cf.list(
                            new WeekDay(WeekdayConv.jodaToIcal(day.getDay().get2().get()), day.getDay().get1()))));
                } else {
                    recur = new IcsRecur(Freq.YEARLY);
                }
                if (day.getEndYear().isPresent()) {
                    recur = recur.withUntilDate(start.withYear(day.getEndYear().get()));
                }
                event = event.addProperty(new IcsRRule(recur));
            }
            return event;
        }));

        return calendar;
    }
}
