package ru.yandex.calendar.logic.ics.exp;

import net.fortuna.ical4j.model.Property;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.Bean;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.beans.generated.TodoItemFields;
import ru.yandex.calendar.logic.beans.generated.TodoList;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.PropertyNames;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsComponent;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVAlarm;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVTimeZone;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVToDo;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsAction;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsCreated;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsMethod;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsProperty;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsTrigger;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsXProperty;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.dateTime.IcsDateTime;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.dateTime.IcsDateTimeFormats;
import ru.yandex.calendar.logic.ics.iv5j.support.IvParser;
import ru.yandex.calendar.logic.todo.TodoDao;
import ru.yandex.calendar.logic.todo.TodoRoutines;
import ru.yandex.calendar.util.color.ColorUtils;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.inside.passport.PassportUid;

/**
 * @author Stepan Koltsov
 * @see IcsEventExporter
 */
public class IcsTodoExporter {

    public static final String X_YANDEX_TODO_LIST_TITLE = "X-YANDEX-TODO-LIST-TITLE";
    public static final String X_YANDEX_TODO_LIST_UID = "X-YANDEX-TODO-LIST-UID";

    @Autowired
    private TodoRoutines todoRoutines;
    @Autowired
    private TodoDao todoDao;
    @Autowired
    private DateTimeManager dateTimeManager;

    public IcsCalendar createCalendarWithTodoItemsByIds(PassportUid uid, ListF<Long> todoListIds) {
        ListF<TodoList> todoLists = todoRoutines.getTodoLists(uid, Cf.x(todoListIds));
        ListF<TodoItem> todoItems = todoRoutines.getTodoItemsPlainSorted(uid, todoLists.map(Bean::getId));

        return createCalendarWithTodoItems(uid, todoItems);
    }

    public IcsCalendar createCalendarWithTodoItems(PassportUid uid, ListF<TodoItem> todoItems) {
        return createCalendarWithVTodos(uid, exportTodoItems(todoItems, uid));
    }

    public IcsCalendar createCalendarWithVTodos(PassportUid uid, ListF<IcsVToDo> vTodos) {
        IcsCalendar calendar = IcsCalendar.fromIcal4j(IcsEventExporter.createCommonCalendarPart(IcsMethod.PUBLISH));

        calendar = calendar.addComponents(vTodos);
        IcsVTimeZone vTimeZone = IcsTimeZones.icsVTimeZoneForIdFull(dateTimeManager.getTimeZoneForUid(uid).getID());
        return calendar.addComponents(IcsEventExporter.appendTimezoneIfNeeded(vTimeZone));
    }

    public ListF<IcsVToDo> exportTodoItems(ListF<TodoItem> todoItems, PassportUid uid) {
        DateTimeZone dateTimeZone = dateTimeManager.getTimeZoneForUid(uid);
        ListF<IcsVToDo> icsTodos = Cf.arrayList();
        for (TodoItem todoItem : todoItems) {
            IcsVToDo icsTodo = new IcsVToDo();
            icsTodo = icsTodo.addProperty(new IcsCreated(todoItem.getCreationTs()));

            icsTodo = icsTodo.addProperty(new IcsXProperty(PropertyNames.X_POS, String.valueOf(todoItem.getPos())));
            if (todoItem.getFieldValueO(TodoItemFields.COLOR).isPresent() && todoItem.getColor().isPresent()) {
                icsTodo = icsTodo.addProperty(new IcsXProperty(
                        PropertyNames.X_COLOR, ColorUtils.formatColor(todoItem.getColor().get())));
            }

            icsTodo = icsTodo.withSummary(todoItem.getTitle());

            icsTodo = icsTodo.withPropertyIfNotEmpty(Property.DESCRIPTION, todoItem.getDescription());
            icsTodo = icsTodo.withPropertyIfNotEmpty(Property.LOCATION, todoItem.getLocation());

            icsTodo = icsTodo.withUid(todoItem.getExternalId());

            if (todoItem.getNotificationTs().isPresent()) {
                IcsTrigger trigger = IcsTrigger.createDateTime(todoItem.getNotificationTs().get());
                IcsVAlarm alarm = new IcsVAlarm(Cf.<IcsProperty>list(IcsAction.DISPLAY, trigger), Cf.<IcsComponent>list());
                icsTodo = icsTodo.withVAlarm(alarm);
            }
            if (todoItem.getIcalGeoNotification().isPresent()) {
                icsTodo = icsTodo.withVAlarm(new IcsVAlarm(
                        IvParser.parseIcsProperties(todoItem.getIcalGeoNotification().get()),
                        Cf.<IcsComponent>list()));
            }
            if (todoItem.getCompletionTs().isPresent()) {
                icsTodo = icsTodo.withCompleted(todoItem.getCompletionTs().get());
            }
            icsTodo = icsTodo.withProperty(Property.SEQUENCE, "" + todoItem.getSequence());
            icsTodo = icsTodo.withPropertyIfDefined(Property.PERCENT_COMPLETE, todoItem.getCompletePercent());
            icsTodo = icsTodo.withPropertyIfDefined(Property.PRIORITY, todoItem.getPriority());
            icsTodo = icsTodo.withPropertyIfDefined(PropertyNames.X_APPLE_SORT_ORDER, todoItem.getIcalXAppleSortOrder());

            if (todoItem.getStatus().isPresent()) {
                icsTodo = icsTodo.withProperty(Property.STATUS, todoItem.getStatus().get().toIcsValue());
            }
            if (todoItem.getStartTs().isPresent()) {
                icsTodo = icsTodo.withDtStart(todoItem.getIsAllDay()
                        ? IcsDateTime.localDate(new LocalDate(todoItem.getStartTs().get(), dateTimeZone))
                        : IcsDateTime.dateTime(todoItem.getStartTs().get().toDateTime(dateTimeZone)));
            }
            if (todoItem.getDueTs().isPresent()) {
                icsTodo = icsTodo.withDue(todoItem.getIsAllDay()
                        ? IcsDateTime.localDate(new LocalDate(todoItem.getDueTs().get(), dateTimeZone))
                        : IcsDateTime.dateTime(todoItem.getDueTs().get().toDateTime(dateTimeZone)));
            }
            if (todoItem.getIcalXMozLastack().isPresent()) {
                Instant x = todoItem.getIcalXMozLastack().get();
                icsTodo = icsTodo.addProperty(new IcsXProperty(
                        PropertyNames.X_MOZ_LASTACK, IcsDateTimeFormats.formatDateTime(x)));
            }
            if (todoItem.getIcalXMozSnoozeTime().isPresent()) {
                Instant x = todoItem.getIcalXMozSnoozeTime().get();
                icsTodo = icsTodo.addProperty(new IcsXProperty(
                        PropertyNames.X_MOZ_SNOOZE_TIME, IcsDateTimeFormats.formatDateTime(x)));
            }

            icsTodos.add(icsTodo);
        }

        return icsTodos;
    }

    public Option<IcsSingleTodoItemExportData> exportTodoItemByExternalIdForCaldav(PassportUid uid, String todoItemExternalId) {
        return exportTodoItemsByExternalIdForCaldav(uid, Cf.list(todoItemExternalId)).firstO();
    }

    public ListF<IcsSingleTodoItemExportData> exportTodoItemsByExternalIdForCaldav(
            PassportUid uid, ListF<String> todoItemExternalIds)
    {
        ListF<TodoItem> todoItems =
                todoDao.findNotDeletedNotArchivedTodoItemsByExternalIdsAndCreatorUid(uid, todoItemExternalIds);
        return toExportDataForCaldav(todoItems, uid);
    }

    public ListF<IcsSingleTodoItemExportData> exportTodoItemsByExternalIdAndTodoListIdForCaldav(
            PassportUid uid, ListF<String> todoItemExternalIds, long todoListId)
    {
        ListF<TodoItem> todoItems =
                todoDao.findNotDeletedNotArchivedTodoItemsByExternalIdsAndTodoListId(todoItemExternalIds, todoListId);
        return toExportDataForCaldav(todoItems, uid);
    }

    public ListF<IcsSingleTodoItemExportData> exportTodoItemsByListIdForCaldav(long todoListId, final PassportUid uid) {
        ListF<TodoItem> todoItems = todoDao.findNotDeletedNotArchivedTodoItemsByTodoListIds(Cf.list(todoListId));
        return toExportDataForCaldav(todoItems, uid);
    }

    public ListF<IcsSingleTodoItemExportData> exportTodoItemsByListIdForCaldavCreatedOrModifiedSince(
            long todoListId, PassportUid uid, Instant since)
    {
        ListF<TodoItem> todoItems =
                todoDao.findNotDeletedNotArchivedTodoItemsByTodoListIdAndMinLastUpdateTs(todoListId, since);
        return toExportDataForCaldav(todoItems, uid);
    }

    public ListF<IcsSingleTodoItemExportData> exportTodoItemsByUidForCaldav(PassportUid uid) {
        ListF<TodoItem> todoItems = todoDao.findNotDeletedNotArchivedTodoItemsByUid(uid);
        return toExportDataForCaldav(todoItems, uid);
    }

    public ListF<IcsSingleTodoItemExportData> exportTodoItemsByUidForCaldavCreatedOrModifiedSince(
            PassportUid uid, Instant since)
    {
        ListF<TodoItem> todoItems = todoDao.findNotDeletedNotArchivedTodoItemsByUidAndMinLastUpdateTs(uid, since);
        return toExportDataForCaldav(todoItems, uid);
    }

    private ListF<IcsSingleTodoItemExportData> toExportDataForCaldav(ListF<TodoItem> items, PassportUid uid) {
        DateTimeZone tz = dateTimeManager.getTimeZoneForUid(uid);

        ListF<IcsSingleTodoItemExportData> itemsData = Cf.arrayList();
        for (TodoItem item : items) {
            itemsData.add(new IcsSingleTodoItemExportData(
                    exportTodoItems(Cf.list(item), uid).single(), tz, item.getLastUpdateTs(), item.getExternalId()));
        }
        return itemsData;
    }

} //~
