package ru.yandex.calendar.logic.ics.feed;

import java.util.regex.Pattern;

import javax.annotation.PreDestroy;

import org.apache.http.Header;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.DateUtils;
import org.apache.http.util.EntityUtils;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.generated.IcsFeed;
import ru.yandex.calendar.logic.ics.exp.IcsHolidaysExporter;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.http.apache.v4.ReadBytesResponseHandler;

/**
 * @author Sergey Shinderuk
 */
abstract class HttpIcsFeedDownloader implements IcsFeedDownloader {
    private final HttpClient offlineHttpClient;
    private final HttpClient onlineHttpClient;

    protected HttpIcsFeedDownloader(HttpClient httpClient) {
        this(httpClient, httpClient);
    }
    protected HttpIcsFeedDownloader(HttpClient offlineHttpClient, HttpClient onlineHttpClient) {
        this.offlineHttpClient = offlineHttpClient;
        this.onlineHttpClient = onlineHttpClient;
    }

    @PreDestroy
    public void destroy() {
        ApacheHttpClientUtils.stopQuietly(offlineHttpClient);
        ApacheHttpClientUtils.stopQuietly(onlineHttpClient);
    }

    protected abstract String nameForLog();
    protected abstract HttpGet makeRequest(String url);
    protected abstract HttpGet makeRequestNow(String url);

    public byte[] downloadNow(String url) {
        if (url.startsWith(IcsHolidaysExporter.URL_PREFIX)) {
            return IcsHolidaysExporter.export(url).serializeToBytes();
        }

        url = fixProtocol(url);
        HttpGet request = makeRequestNow(url);
        return ApacheHttpClientUtils.execute(request, onlineHttpClient, new ReadBytesResponseHandler(), urlForLog(url));
    }

    public Option<DownloadedFeed> downloadIfModified(final IcsFeed feed) {
        if (feed.getUrl().startsWith(IcsHolidaysExporter.URL_PREFIX)) {
            return Option.of(new DownloadedFeed(
                    IcsHolidaysExporter.export(feed.getUrl()).serializeToBytes(), Option.empty(), Option.empty()));
        }

        String url = fixProtocol(feed.getUrl());
        HttpGet request = makeRequest(url);
        if (feed.getHttpLastModTs().isPresent()) {
            request.setHeader(HttpHeaders.IF_MODIFIED_SINCE, formatHttpDate(feed.getHttpLastModTs().get()));
        }
        if (feed.getHttpEtag().isPresent()) {
            request.setHeader(HttpHeaders.IF_NONE_MATCH, feed.getHttpEtag().get());
        }

        return ApacheHttpClientUtils.execute(request, offlineHttpClient, response -> {
            {
                int statusCode = response.getStatusLine().getStatusCode();
                if (statusCode == HttpStatus.SC_NOT_MODIFIED) {
                    return Option.empty();
                }
                if (statusCode == HttpStatus.SC_OK) {
                    byte[] bytes = EntityUtils.toByteArray(response.getEntity());
                    return Option.of(new DownloadedFeed(bytes, getLastModified(response), getEtag(response)));
                }
                throw new HttpException(statusCode, "bad status code: " + response.getStatusLine());
            }
        }, urlForLog(url));
    }

    private String urlForLog(String url) {
        return url + " (" + nameForLog() + ")";
    }

    private static final Pattern WEBCAL_PATTERN = Pattern.compile("^webcal://");
    private static final Pattern WEBCALS_PATTERN = Pattern.compile("^webcals://");

    private static String fixProtocol(String url) {
        url = WEBCAL_PATTERN.matcher(url).replaceFirst("http://");
        return WEBCALS_PATTERN.matcher(url).replaceFirst("https://");
    }

    private static Option<Instant> getLastModified(HttpResponse response) {
        Header h = response.getLastHeader(HttpHeaders.LAST_MODIFIED);
        return h != null ? parseHttpDate(h.getValue()) : Option.<Instant>empty();
    }

    private static Option<String> getEtag(HttpResponse response) {
        Header h = response.getLastHeader(HttpHeaders.ETAG);
        return h != null ? Option.of(h.getValue()) : Option.<String>empty();
    }

    private static String formatHttpDate(Instant instant) {
        return DateUtils.formatDate(instant.toDate());
    }

    private static Option<Instant> parseHttpDate(String s) {
        return Option.ofNullable(DateUtils.parseDate(s)).map(Instant::new);
    }
}
