package ru.yandex.calendar.logic.ics.feed;

import java.net.URI;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.frontend.worker.task.UpdateIcsFeedTask;
import ru.yandex.calendar.logic.beans.generated.IcsFeed;
import ru.yandex.calendar.logic.beans.generated.IcsFeedFields;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.calendar.logic.ics.imp.IcsImportMode;
import ru.yandex.calendar.logic.ics.imp.IcsImportStats;
import ru.yandex.calendar.logic.ics.imp.IcsImporter;
import ru.yandex.calendar.logic.ics.imp.LayerReference;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.util.db.DbUtils;
import ru.yandex.commune.bazinga.pg.PgBazingaTaskManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.Validate;

/**
 * @author ssytnik
 */
public class IcsFeedManager {
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private IcsFeedDao icsFeedDao;
    @Autowired
    private IcsImporter icsImporter;
    @Autowired
    private IcsFeedDownloader icsFeedDownloader;
    @Autowired
    private IcsFeedUpdater icsFeedUpdater;
    @Autowired
    private PgBazingaTaskManager bazingaTaskManager;
    @Autowired
    private TransactionTemplate transactionTemplate;


    public IcsCalendar downloadAndParseNow(String url) {
        validateUrl(url);
        byte[] bytes = icsFeedDownloader.downloadNow(url);
        try {
            return IcsUtils.parseBytesInGuessedEncoding(bytes);
        } catch (Exception e) {
            throw CommandRunException.createSituation(e, Situation.ICS_PARSING_ERROR);
        }
    }

    private Option<String> loadCalendarName(String url) {
        // XXX don't load whole calendar
        IcsCalendar calendar = IcsUtils.parseBytesInGuessedEncoding(icsFeedDownloader.downloadNow(url));
        return calendar.getXWrCalname();
    }

    private void checkUrlLocatesIcsContent(String url) {
        downloadAndParseNow(url);
    }

    private void checkUrlIsNotSubscribed(PassportUid uid, String url) {
        ListF<IcsFeed> existing = icsFeedDao.findIcsFeeds(IcsFeedFields.UID.eq(uid).and(IcsFeedFields.URL.eq(url)));
        if (existing.isNotEmpty()) {
            throw CommandRunException.createSituation(
                    "Already subscribed to this url", Situation.ICS_FEED_ALREADY_SUBSCRIBED_URL);
        }
    }

    private static void validateUrl(String url) {
        try {
            URI uri = URI.create(url);
            Validate.isTrue(uri.isAbsolute());
            Validate.in(uri.getScheme().toLowerCase(), Cf.set("http", "https", "webcal", "webcals", "holidays"));
        } catch (Exception e) {
            throw CommandRunException.createSituation(e, Situation.ICS_FEED_INVALID_URL);
        }
    }

    public IcsImportStats importContent(
            PassportUid uid, byte[] content, String layerName, LayerReference layerReference)
    {
        IcsImportMode icsImportMode = IcsImportMode.importFile(layerReference, layerName);

        IcsCalendar calendar = IcsUtils.parseBytesInGuessedEncoding(content);
        return icsImporter.importIcsStuff(uid, calendar, icsImportMode);
    }

    public IcsImportStats importContent(PassportUid uid, String url, String layerName, LayerReference layerReference) {
        validateUrl(url);
        checkUrlIsNotSubscribed(uid, url);
        return importContent(uid, icsFeedDownloader.downloadNow(url), layerName, layerReference);
    }

    public IcsFeed subscribe(PassportUid uid, String url, String layerName) {
        validateUrl(url);
        checkUrlIsNotSubscribed(uid, url);
        return createIcsFeed(uid, url, layerRoutines.createFeedLayer(uid, loadCalendarName(url).getOrElse(layerName)));
    }

    public IcsFeed subscribe(PassportUid uid, String url, long layerId) {
        validateUrl(url);
        checkUrlIsNotSubscribed(uid, url);
        return createIcsFeed(uid, url, layerId);
    }

    public void updateFeedUrl(PassportUid uid, long layerId, String newUrl) {
        validateUrl(newUrl);
        IcsFeed feed = icsFeedDao.findIcsFeedByLayerId(layerId);

        if (!feed.getUrl().equals(newUrl)) {
            checkUrlIsNotSubscribed(uid, newUrl);
            checkUrlLocatesIcsContent(newUrl);

            IcsFeed icsFeed = new IcsFeed();
            icsFeed.setId(layerId);
            icsFeed.setUrl(newUrl);

            icsFeed.setHash(Option.<String>empty());
            icsFeed.setHttpEtag(Option.<String>empty());
            icsFeed.setHttpLastModTs(Option.<Instant>empty());

            icsFeed.setIntervalMin(icsFeedUpdater.minUpdateIntervalMinutes.get());
            icsFeed.setNextQueryTs(Instant.now());

            doInTransaction(() -> {
                icsFeedDao.updateIcsFeed(icsFeed);
                scheduleUpdateTask(icsFeed, true);
            });
        }
    }

    public void forceIcsFeedUpdate(PassportUid uid, long layerId) {
        ListF<IcsFeed> existing = icsFeedDao.findIcsFeeds(
                IcsFeedFields.LAYER_ID.eq(layerId).and(IcsFeedFields.UID.eq(uid)));
        Validate.sizeIs(1, existing);

        IcsFeed icsFeed = new IcsFeed();
        icsFeed.setId(layerId);
        icsFeed.setIntervalMin(icsFeedUpdater.minUpdateIntervalMinutes.get());
        icsFeed.setNextQueryTs(Instant.now());

        doInTransaction(() -> {
            icsFeedDao.updateIcsFeed(icsFeed);
            scheduleUpdateTask(icsFeed, true);
        });
    }

    private IcsFeed createIcsFeed(PassportUid uid, String url, long layerId) {
        IcsFeed icsFeed = new IcsFeed();
        icsFeed.setUid(uid);
        icsFeed.setUrl(url);
        icsFeed.setLayerId(layerId);
        icsFeed.setCreationTs(Instant.now());
        icsFeed.setIntervalMin(icsFeedUpdater.minUpdateIntervalMinutes.get());
        icsFeed.setNextQueryTs(Instant.now());

        doInTransaction(() -> {
            icsFeedDao.saveIcsFeed(icsFeed);
            scheduleUpdateTask(icsFeed, true);
        });

        return icsFeedDao.findIcsFeedByLayerId(layerId);
    }

    private void doInTransaction(Function0V function) {
        if (!DbUtils.isInTransaction()) {
            transactionTemplate.execute(function.asFunctionReturnParam()::apply);
        } else {
            function.apply();
        }
    }

    private void scheduleUpdateTask(IcsFeed feed, boolean highPriority) {
        UpdateIcsFeedTask task = new UpdateIcsFeedTask(feed.getLayerId());
        bazingaTaskManager.schedule(task, feed.getNextQueryTs(), highPriority ? task.priority() + 1 : task.priority());
    }

    public Tuple2List<Long, Option<IcsFeed>> findIcsFeedsByLayerIds(ListF<Long> layerIds) {
        ListF<IcsFeed> x = icsFeedDao.findIcsFeeds(IcsFeedFields.LAYER_ID.column().inSet(layerIds));
        return layerIds.zipWith(x.toMapMappingToKey(IcsFeed.getLayerIdF())::getO);
    }

}
