package ru.yandex.calendar.logic.ics.feed;

import java.nio.charset.Charset;
import java.util.Optional;
import java.util.regex.Pattern;

import io.micrometer.core.instrument.MeterRegistry;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.calendar.logic.beans.generated.IcsFeed;
import ru.yandex.calendar.logic.beans.generated.IcsFeedFields;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.EventLayerDao;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.calendar.logic.ics.imp.IcsImportMode;
import ru.yandex.calendar.logic.ics.imp.IcsImportStats;
import ru.yandex.calendar.logic.ics.imp.IcsImporter;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.util.exception.ExceptionUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.DataSourceUtils;
import ru.yandex.misc.digest.Md5;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.Instrument;
import ru.yandex.misc.monica.core.blocks.Profiler;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.random.Random2;

/**
 * @author Sergey Shinderuk
 */
public class IcsFeedUpdater implements MonicaContainer {

    private static final Logger logger = LoggerFactory.getLogger(IcsFeedUpdater.class);

    public final DynamicProperty<Integer> minUpdateIntervalMinutes =
            new DynamicProperty<>("icsFeedMinUpdateIntervalMinutes", 60);
    public final DynamicProperty<Integer> maxUpdateIntervalMinutes =
            new DynamicProperty<>("icsFeedMaxUpdateIntervalMinutes", (int) Duration.standardDays(14).getStandardMinutes());
    public final DynamicProperty<Double> updateIntervalMultiplier =
            new DynamicProperty<>("icsFeedUpdateIntervalMultiplier", 1.3);
    private final DynamicProperty<Integer> updateThreads = new DynamicProperty<>("icsFeedUpdateThreads", 10);

    private final DynamicProperty<Integer> updateSpreadIntervalMinutes =
            new DynamicProperty<>("icsFeedSpreadIntervalMinutes", 15);

    private static final String STARTED_COUNTER = "application.icsfeedupdater.started";
    private static final String FAILED_COUNTER = "application.icsfeedupdater.failed";

    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private EventLayerDao eventLayerDao;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private IcsFeedDao icsFeedDao;
    @Autowired
    private IcsFeedDownloader icsFeedDownloader;
    @Autowired
    private IcsImporter icsImporter;

    @MonicaMetric
    @GroupByDefault
    private final Profiler profiler = new Profiler();
    @MonicaMetric
    @GroupByDefault
    private final Instrument invocations = new Instrument();

    @Autowired
    private MeterRegistry registry;

    public Option<IcsFeed> updateIcsFeed(long layerId) {
        Option<IcsFeed> feed = icsFeedDao.findIcsFeeds(IcsFeedFields.LAYER_ID.eq(layerId)).singleO();

        return feed.map(f -> invocations.measure(() -> doUpdateIcsFeed(f)));
    }

    private IcsFeed doUpdateIcsFeed(IcsFeed feed) {
        Profiler.Session session = profiler.createAndStart();
        try {
            registry.counter(STARTED_COUNTER).increment();
            logger.info("Updating feed: uid={} layer_id={} url={}", feed.getUid(), feed.getLayerId(), feed.getUrl());

            session.step("download");
            Option<DownloadedFeed> downloaded = icsFeedDownloader.downloadIfModified(feed);

            if (downloaded.isPresent()) {
                String hash = md5Hash(downloaded.get().bytes);
                if (feed.getHash().isSome(hash)) {
                    logger.info("Feed md5 hash did not change");
                    return recordUselessAttempt(feed);
                } else {
                    logger.info("Feed was modified");
                    ModificationInfo modificationInfo = new ModificationInfo(
                            hash, downloaded.get().lastModified, downloaded.get().etag);

                    session.step("update");
                    return doUpdateIcsFeed(feed, downloaded.get().bytes, modificationInfo);
                }
            } else {
                logger.info("Feed was not modified on server");
                return recordUselessAttempt(feed);
            }
        } catch (Exception e) {
            registry.counter(FAILED_COUNTER).increment();
            ExceptionUtils.rethrowIfTlt(e);

            logger.info("Failed to update feed: {}", e);
            return recordError(feed);

        } finally {
            DataSourceUtils.setQuietMode(false);
            session.end();
        }
    }

    private static final Pattern DTSTAMP_PATTERN = Pattern.compile("DTSTAMP:[0-9]+T[0-9]+Z");

    public static String md5Hash(byte[] data) {
        Charset charset = Charset.forName(ru.yandex.calendar.util.CharsetUtils.guessEncoding(data));
        String dataString = CharsetUtils.decode(charset, data);
        String hashCheckString = DTSTAMP_PATTERN.matcher(dataString).replaceAll("");
        return Md5.A.digest(CharsetUtils.encodeToArray(charset, hashCheckString)).hex();
    }

    private IcsFeed doUpdateIcsFeed(IcsFeed feed, byte[] downloadedBytes, ModificationInfo modificationInfo) {
        IcsCalendar calendar = IcsUtils.parseBytesInGuessedEncoding(downloadedBytes);

        IcsImportStats importStats;

        DataSourceUtils.setQuietMode(true);
        try {
            importStats = icsImporter.importIcsStuff(
                    feed.getUid(), calendar, IcsImportMode.updateFeed(feed.getLayerId()), Optional.of(feed.getUrl()));
        } finally {
            DataSourceUtils.setQuietMode(false);
        }

        removeEventsDeletedFromFeed(feed.getUid(), feed.getLayerId(), importStats);

        return recordSuccessfulUpdate(feed, modificationInfo);
    }

    private void removeEventsDeletedFromFeed(PassportUid feedUid, long feedLayerId, IcsImportStats stats) {
        SetF<Long> eIds = Cf.toHashSet(eventLayerDao.findEventLayerEventIdsByLayerId(feedLayerId));
        // Remove ids of all processed events
        eIds.removeAllTs(stats.getProcessedEventIds());
        // Remove remaining events from the calendar database
        String requestId = RequestIdStack.current().getOrElse("?");
        ActionInfo actionInfo = new ActionInfo(ActionSource.WEB_ICS, requestId, Instant.now());

        ListF<Long> eventIdsToDelete = eventDao.findUserCreatedEventIdsWithoutOrganizer(feedUid, eIds.toList());

        eventIdsToDelete = eventIdsToDelete.plus(eventDao.findUserCreatedEventIdsWithoutOtherUserAndResource(
                feedUid, eIds.minus(eventIdsToDelete).toList()));

        ListF<Long> eventIdsToDetach = eIds.minus(eventIdsToDelete).toList();

        // Without archive because feed may contain events with random generated UIDS (e.g. from kinopoisk.ru)
        eventDbManager.deleteEventsByIds(eventIdsToDelete, true, actionInfo);
        eventLayerDao.deleteEventLayersByLayerIdAndEventIds(feedLayerId, eventIdsToDetach);
    }

    private IcsFeed recordSuccessfulUpdate(IcsFeed feed, ModificationInfo modificationInfo) {
        IcsFeed updated = feed.copy();
        Instant now = Instant.now();
        updated.setLastQueryTs(now);
        updated.setLastSuccessTs(now);
        updated.setUselessAttempts(0);
        updated.setIntervalMin(minUpdateIntervalMinutes.get());
        updated.setHash(modificationInfo.hash);
        updated.setHttpEtag(modificationInfo.etag);
        updated.setHttpLastModTs(modificationInfo.lastModified);
        updated.setNextQueryTs(nextSpreadQueryTs(now, updated.getIntervalMin()));

        icsFeedDao.updateIcsFeed(updated);

        return updated;
    }

    private IcsFeed recordUselessAttempt(IcsFeed feed) {
        IcsFeed updated = feed.copy();
        Instant now = Instant.now();
        updated.setLastQueryTs(now);
        updated.setLastSuccessTs(now);
        updated.setUselessAttempts(feed.getUselessAttempts() + 1);
        updated.setIntervalMin(nextBiggerInterval(feed.getIntervalMin()));
        updated.setNextQueryTs(nextSpreadQueryTs(now, updated.getIntervalMin()));

        icsFeedDao.updateIcsFeed(updated);

        return updated;
    }

    private IcsFeed recordError(IcsFeed feed) {
        IcsFeed updated = feed.copy();
        Instant now = Instant.now();
        updated.setLastQueryTs(now);
        updated.setUselessAttempts(feed.getUselessAttempts() + 1);
        updated.setIntervalMin(nextBiggerInterval(feed.getIntervalMin()));
        updated.setNextQueryTs(nextSpreadQueryTs(now, updated.getIntervalMin()));

        icsFeedDao.updateIcsFeed(updated);

        return updated;
    }

    private Instant nextSpreadQueryTs(Instant now, int desiredMinutes) {
        Duration desiredDelay = Duration.standardMinutes(desiredMinutes);

        Duration spread = Duration.standardMinutes(updateSpreadIntervalMinutes.get());

        return now.plus(desiredDelay.plus(Random2.R.nextLong(spread.getMillis())));
    }

    private int nextBiggerInterval(int currentInterval) {
        return Math.min((int) (updateIntervalMultiplier.get() * currentInterval), maxUpdateIntervalMinutes.get());
    }

    private static class ModificationInfo {
        final String hash;
        final Option<Instant> lastModified;
        final Option<String> etag;

        ModificationInfo(String hash, Option<Instant> lastModified, Option<String> etag) {
            this.hash = hash;
            this.lastModified = lastModified;
            this.etag = etag;
        }
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName(
                "calendar",
                new MetricName("calendar", "ics-feed-updater"),
                "Ics feed updating statistic"
        );
    }
}
