package ru.yandex.calendar.logic.ics.feed;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.val;
import org.apache.http.HttpHost;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.conn.routing.HttpRoute;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;

import ru.yandex.calendar.tvm.TvmClient;
import ru.yandex.calendar.tvm.TvmHeaders;
import ru.yandex.calendar.util.HttpClientConfiguration;

/**
 * CAL-5896
 *
 * @url http://wiki.yandex-team.ru/Robot/Manual/Zora/UserGuide
 */
public class ZoraIcsFeedDownloader extends HttpIcsFeedDownloader {
    private static final String REQUEST_METRIC = "application.zora.request";
    private final TvmClient tvmClient;
    private final int zoraClientId;
    private final int zoraResponseTimeout;

    public ZoraIcsFeedDownloader(HttpClientConfiguration conf, TvmClient client, int zoraClientId, MeterRegistry registry,
                                 String offlineHostname, String onlineHostname) {
        super(
                makeHttpClient(conf, offlineHostname, registry),
                makeHttpClient(conf, onlineHostname, registry)
        );
        this.tvmClient = client;
        this.zoraClientId = zoraClientId;
        // Zora demands response timeout in seconds. Socket timeout, on the other hand, is in milliseconds
        zoraResponseTimeout = conf.socketTimeout/1000;
    }

    private static HttpClient makeHttpClient(HttpClientConfiguration conf, String hostname, MeterRegistry registry) {
        val connectionManager = new PoolingHttpClientConnectionManager();
        val responseInterceptor = new ZoraResponseInterceptor(REQUEST_METRIC, registry);

        connectionManager.setDefaultMaxPerRoute(conf.maxConnections);
        connectionManager.setMaxTotal(conf.maxConnections);

        val params = RequestConfig.custom()
                .setConnectTimeout(conf.connectionTimeout)
                .setSocketTimeout(conf.socketTimeout)
                .setStaleConnectionCheckEnabled(true)
                .setMaxRedirects(conf.maxRedirects)
                .build();

        return HttpClientBuilder.create()
                .disableCookieManagement()
                .setDefaultRequestConfig(params)
                .setConnectionManager(connectionManager)
                .setRoutePlanner((target, request, context) ->
                        new HttpRoute(target, null, new HttpHost(hostname, 1080), false))
                .addInterceptorLast(responseInterceptor)
                .build();
    }

    private void setCommonHeaders(HttpGet request) {
        request.setHeader("Accept-Encoding", "gzip");
        request.setHeader(TvmHeaders.SERVICE_TICKET, tvmClient.getServiceTicketFor(zoraClientId));
        request.setHeader("X-Yandex-Response-Timeout", String.valueOf(zoraResponseTimeout));
        request.setHeader("X-Yandex-Redirs", "1");
    }

    @Override
    protected HttpGet makeRequest(String url) {
        val request = new HttpGet(url);

        request.setHeader("X-Yandex-Sourcename", "calendar");
        request.setHeader("X-Ya-Follow-Redirects", "true");
        request.setHeader("Cache-Control", "max-age=10");

        setCommonHeaders(request);
        return request;
    }

    @Override
    protected HttpGet makeRequestNow(String url) {
        val request = new HttpGet(url);

        request.setHeader("X-Yandex-Sourcename", "calendar_userproxy");
        request.setHeader("X-Yandex-Requesttype", "userproxy");
        request.setHeader("X-Ya-Follow-Redirects", "true");
        request.setHeader("Cache-Control", "max-age=0");

        setCommonHeaders(request);
        return request;
    }

    @Override
    protected String nameForLog() {
        return "zora";
    }
}
