package ru.yandex.calendar.logic.ics.imp;

import net.fortuna.ical4j.model.property.Conference;
import net.fortuna.ical4j.model.property.Transp;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalTime;
import org.joda.time.Period;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.event.AbstractEventDataConverter;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.model.EventUserData;
import ru.yandex.calendar.logic.event.model.Priority;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.PropertyNames;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVAlarm;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVEvent;
import ru.yandex.calendar.logic.ics.iv5j.ical.parameter.IcsRelated;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsAction;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsAttendee;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsExDate;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsOrganizer;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsProperty;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsRDate;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsRRule;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsRecurrenceId;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.IcsRDateInterval;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.dateTime.IcsDateTimeFormats;
import ru.yandex.calendar.logic.notification.Channel;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.participant.ParticipantData;
import ru.yandex.calendar.logic.sharing.participant.ParticipantsData;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;

public class IcsEventDataConverter extends AbstractEventDataConverter {
    private static final Logger logger = LoggerFactory.getLogger(IcsEventDataConverter.class);
    private final IcsVEvent vevent;
    private final boolean ignoreEventUser;
    private final IcsVTimeZones tzs;

    private IcsEventDataConverter(IcsVTimeZones tzs, IcsVEvent vevent, boolean ignoreEventUser) {
        super(tzs.getFallbackTz());
        this.tzs = tzs;
        this.vevent = vevent;
        this.ignoreEventUser = ignoreEventUser;
    }

    protected EventData doConvert() {
        EventData eventData = convertEventData();
        Instant startMs = eventData.getEvent().getStartTs();
        Instant endMs = eventData.getEvent().getEndTs();
        eventData.setRepetition(convertRepetition(startMs));
        eventData.setInvData(convertParticipantsData());
        eventData.getRdates().addAll(convertExdates());
        eventData.getRdates().addAll(convertRdates());
        if (!ignoreEventUser) {
            eventData.setEventUserData(convertEventUserData(startMs, endMs));
        }
        moveDueTsAfterEventStart(eventData);

        return eventData;
    }

    private EventData convertEventData() {
        EventData eventData = new EventData();
        InstantInterval startEndMs = getStartEndMs(vevent);
        eventData.getEvent().setStartTs(startEndMs.getStart());
        eventData.getEvent().setEndTs(startEndMs.getEnd());
        eventData.getEvent().setIsAllDay(vevent.isAllDay());
        eventData.setTimeZone(eventTz);

        Option<Instant> lastModifiedInstantO = vevent.getLastModifiedInstant(tzs);
        if (lastModifiedInstantO.isPresent()) {
            eventData.getEvent().setLastUpdateTs(lastModifiedInstantO.get());
        }
        eventData.getEvent().setRecurrenceId(vevent.getRecurrenceId().map(IcsRecurrenceId.getInstantF(tzs)));
        eventData.getEvent().setDtstamp(vevent.getDtStampInstant(tzs));
        eventData.getEvent().setName(IcsUtils.getPropIfSet(vevent.toComponent().getSummary()).getOrElse(EventRoutines.DEFAULT_NAME));
        eventData.getEvent().setLocation(StringUtils.notBlankO(
                IcsUtils.getPropIfSet(vevent.toComponent().getLocation()).getOrNull()).getOrElse(""));
        eventData.getEvent().setDescription(StringUtils.notBlankO(
                IcsUtils.getPropIfSet(vevent.toComponent().getDescription()).getOrNull()).getOrElse(""));
        eventData.getEvent().setUrl(StringUtils.defaultIfEmpty(
                IcsUtils.getPropIfSet(vevent.toComponent().getUrl()).getOrNull(), null));
        Option<String> sequence = IcsUtils.getPropIfSet(vevent.toComponent().getSequence());
        if (sequence.isPresent()) {
            eventData.getEvent().setSequence(Cf.Integer.parseSafe(sequence.get()).getOrElse(0));
        } else {
            eventData.getEvent().setSequence(0);
        }
        eventData.getEvent().setConferenceUrl(StringUtils.defaultIfEmpty(
                vevent.getPropertyValue(Conference.PROPERTY_NAME).getOrNull(), null));
        Option<String> extId = vevent.getUid();
        eventData.setExternalId(extId);
        return eventData;
    }

    static Repetition convertRrule(Instant startTs, IcsRRule rrule, DateTimeZone tz) {
        return RepetitionUtils.getRepetition(rrule.getRecur(), startTs, tz);
    }

    private Repetition convertRepetition(Instant startMs) {
        ListF<IcsRRule> rRules = vevent.getRRules();
        for (IcsRRule rrule : rRules) {
            try {
                return convertRrule(startMs, rrule, eventTz);
            } catch (Exception e) {
                logger.error("Failed to convert " + rrule + ": " + e, e);
            }
        }
        return RepetitionRoutines.createNoneRepetition();
    }

    private ParticipantsData convertParticipantsData() {

        ListF<ParticipantData> attendees =
                vevent.getAttendees().filterMap(new Function<IcsAttendee, Option<ParticipantData>>() {
                                                    public Option<ParticipantData> apply(IcsAttendee attendee) {
                                                        return attendeeToParticipantData(attendee);
                                                    }
                                                }
                );

        ListF<ParticipantData> uniqueAttendees = attendees.groupBy(ParticipantData.getEmailF()).values().map(Cf.List.<ParticipantData>getF(0));

        if (vevent.getOrganizer().isPresent()) {
            IcsOrganizer icsOrganizer = vevent.getOrganizer().get();
            if (icsOrganizer.isEmail()) {
                ParticipantData organizer = new ParticipantData(icsOrganizer.getEmail(), icsOrganizer.getCn(), Decision.YES, false, true, false);
                return ParticipantsData.merge(organizer, uniqueAttendees);
            } else {
                logger.warn("Organizer is incorrect: " + icsOrganizer);
                return ParticipantsData.notMeeting();
            }
        } else {
            return ParticipantsData.notMeeting();
        }
    }

    public static Option<ParticipantData> attendeeToParticipantData(IcsAttendee attendee) {
        if (attendee.isEmail()) {
            Decision decision = IcsUtils.getDecision(attendee);
            Email email = attendee.getEmail();
            String name = attendee.getCn();
            return Option.of(new ParticipantData(email, name, decision, true, false, false));
        } else {
            logger.warn("Incorrect attendee e-mail: " + attendee);
            return Option.empty();
        }
    }

    public static Function<IcsAttendee, Option<ParticipantData>> attendeeToParticipantDataF() {
        return new Function<IcsAttendee, Option<ParticipantData>>() {
            public Option<ParticipantData> apply(IcsAttendee a) {
                return attendeeToParticipantData(a);
            }
        };
    }

    private ListF<Rdate> convertExdates() {
        ListF<Rdate> exdateList = Cf.arrayList();
        for (IcsExDate icsExDate : vevent.getExDates()) {
            for (Instant date : icsExDate.getInstants(tzs)) {
                if (!icsExDate.getIcsTzId().isPresent()) {
                    // Some clients (e.g. Lightning 1.0b7) send exdate in UTC, but also use incorrect timezone
                    // Temporary fix to store correct exdate
                    Instant start = vevent.getStart(tzs);
                    LocalTime startTime = start.toDateTime(DateTimeZone.UTC).toLocalTime();
                    LocalTime exTime = date.toDateTime(DateTimeZone.UTC).toLocalTime();
                    if (startTime.plus(Period.hours(1)).equals(exTime)) {
                        date = date.minus(Duration.standardHours(1));
                    }
                    if (startTime.minus(Period.hours(1)).equals(exTime)) {
                        date = date.plus(Duration.standardHours(1));
                    }
                }
                exdateList.add(RepetitionUtils.consExdate(date));
            }
        }
        return exdateList;
    }

    private ListF<Rdate> convertRdates() {
        ListF<Rdate> rdateList = Cf.arrayList();
        for (IcsRDate icsRDate : vevent.getRDates()) {
            for (IcsRDateInterval interval : icsRDate.getIntervals()) {
                Rdate rdate = new Rdate();
                rdate.setIsRdate(true);
                rdate.setStartTs(interval.getStart(tzs));
                rdate.setEndTs(interval.getEnd(tzs));
                rdateList.add(rdate);
            }
        }
        return rdateList;
    }

    // 2nd fix needed for ical4j to be RFC-compliant, is handling the case
    // when dt end is absent at all, as well as duration absence.
    // According to RFC, in this case end_ts == start_ts (if ts)
    // and end_date == start_date (in fact, end_date = begin of start_date + 1 day).
    private InstantInterval getStartEndMs(IcsVEvent ve) {
        return new InstantInterval(ve.getStart(tzs), ve.getEnd(tzs));
    }

    private EventUserData convertEventUserData(Instant eventStartMs, Instant eventEndMs) {
        return new EventUserData(convertAvailabilityAndAlarms(), convertNotification(eventStartMs, eventEndMs));
    }

    private EventUser convertAvailabilityAndAlarms() {
        EventUser eventUser = new EventUser();

        Transp transparency = vevent.toComponent().getTransparency();
        if (transparency != null) {
            eventUser.setAvailability(Availability.byIcalTransparency(transparency));
        }

        Option<Priority> priorityO = Priority.findByIcalPriority(vevent.toComponent().getPriority());
        if (priorityO.isPresent()) {
            eventUser.setPriority(priorityO.get());
        }
        // lightning use these to track alarm dismissal and snooze
        // http://hg.mozilla.org/releases/comm-1.9.2/file/2d9fb6c132aa/calendar/base/src/calAlarm.js
        // http://hg.mozilla.org/releases/comm-1.9.2/file/2d9fb6c132aa/calendar/base/src/calAlarmService.js
        Option<Instant> lastack = vevent.getProperty(PropertyNames.X_MOZ_LASTACK)
                .map(IcsProperty.valueF().andThen(IcsUtils.parseTimestampF()));
        Option<Instant> snoozeTime = vevent.getProperty(PropertyNames.X_MOZ_SNOOZE_TIME)
                .map(IcsProperty.valueF().andThen(IcsDateTimeFormats.parseDateTimeF()));
        eventUser.setIcalXMozLastack(lastack);
        eventUser.setIcalXMozSnoozeTime(snoozeTime);

        return eventUser;
    }

    private ListF<Notification> convertNotification(Instant eventStartMs, Instant eventEndMs) {
        ListF<Notification> notifications = Cf.arrayList();
        ListF<IcsVAlarm> alarmComponents = vevent.getAlarms();

        // http://tools.ietf.org/html/draft-daboo-valarm-extensions-04#section-9
        // hack to prevent duplicating on snooze, replace snoozed alarm with the corresponding one
        SetF<String> snoozedUids = alarmComponents.filterMap(IcsVAlarm.getRelatedToF()).unique();
        alarmComponents = alarmComponents.filter(IcsVAlarm.getUidF().andThen(Cf2.isSomeOfF(snoozedUids).notF()));

        // http://tools.ietf.org/html/draft-daboo-valarm-extensions-04#section-11.2
        // hack to avoid extra notifications, default alarms are included each time they are not found
        alarmComponents = alarmComponents.filter(IcsVAlarm.isDefaultF().notF());

        for (IcsVAlarm alarm : alarmComponents) {
            Channel channel;
            IcsAction action = alarm.getAction();

            if (IcsAction.AUDIO.sameNameAndValue(action)) {
                channel = Channel.AUDIO;
            } else if (IcsAction.DISPLAY.sameNameAndValue(action)) {
                channel = Channel.DISPLAY;
            } else { // PROCEDURE
                logger.debug("[Alarm] Action ignored: " + action);
                continue;
            }
            Duration offsetFromEventStart;

            if (!alarm.hasRelativeTrigger()) {
                offsetFromEventStart = new Duration(eventStartMs, alarm.getMainNotificationTimestamp());
            } else if (alarm.getTriggerRelation().sameAs(IcsRelated.END)) {
                offsetFromEventStart = new Duration(eventStartMs, eventEndMs).plus(alarm.getMainNotificationOffset());
            } else {
                offsetFromEventStart = alarm.getMainNotificationOffset();
            }

            notifications.add(new Notification(channel, offsetFromEventStart));

        }

        return notifications;
    }

    public static EventData convert(IcsVTimeZones tzs, IcsVEvent vevent, boolean ignoreEventUser) {
        return new IcsEventDataConverter(tzs, vevent, ignoreEventUser).convert();
    }
}
