package ru.yandex.calendar.logic.ics.imp;

import java.util.List;
import java.util.NoSuchElementException;
import java.util.Optional;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;
import org.apache.commons.lang.NotImplementedException;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.CalendarUtils;
import ru.yandex.calendar.frontend.ews.exp.EwsExportRoutines;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.LastUpdateManager;
import ru.yandex.calendar.logic.XivaNotificationManager;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventLayer;
import ru.yandex.calendar.logic.beans.generated.EventTimezoneInfo;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.contact.ml.Ml;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.ActorId;
import ru.yandex.calendar.logic.event.CreateInfo;
import ru.yandex.calendar.logic.event.EventAttachedLayerId;
import ru.yandex.calendar.logic.event.EventChangesFinder;
import ru.yandex.calendar.logic.event.EventChangesInfo;
import ru.yandex.calendar.logic.event.EventChangesInfoForMails;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventDeletionSmsHandler;
import ru.yandex.calendar.logic.event.EventInfo;
import ru.yandex.calendar.logic.event.EventInfoDbLoader;
import ru.yandex.calendar.logic.event.EventInstanceForUpdate;
import ru.yandex.calendar.logic.event.EventInstanceStatusChecker;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventMoveSmsHandler;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventSynchData;
import ru.yandex.calendar.logic.event.EventUserRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.EventsOnLayerChangeHandler;
import ru.yandex.calendar.logic.event.ExternalId;
import ru.yandex.calendar.logic.event.IcsEventSynchData;
import ru.yandex.calendar.logic.event.LayerIdChangesInfo;
import ru.yandex.calendar.logic.event.MainEventWithRelations;
import ru.yandex.calendar.logic.event.SequenceAndDtStamp;
import ru.yandex.calendar.logic.event.UpdateInfo;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.dao.EventLayerDao;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.event.meeting.CancelMeetingHandler;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.EventInstanceInterval;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.ics.EventInstanceStatusInfo;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.calendar.logic.ics.exp.EventInstanceParameters;
import ru.yandex.calendar.logic.ics.exp.IcsEventExporter;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVEventGroup;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.PropertyNames;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVEvent;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVTimeZone;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsAttendee;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsExDate;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsMethod;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.layer.LayerType;
import ru.yandex.calendar.logic.log.EventIdLogDataJson;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.log.change.EventChangeLogEvents;
import ru.yandex.calendar.logic.notification.NotificationRoutines;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sending.EventSendingInfo;
import ru.yandex.calendar.logic.sending.param.EventMessageParameters;
import ru.yandex.calendar.logic.sending.param.EventOnLayerChangeMessageParameters;
import ru.yandex.calendar.logic.sending.param.ReplyMessageParameters;
import ru.yandex.calendar.logic.sending.so.SoChecker;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.InvAcceptingType;
import ru.yandex.calendar.logic.sharing.InvitationProcessingMode;
import ru.yandex.calendar.logic.sharing.participant.ParticipantData;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantsData;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.svc.SvcRoutines;
import ru.yandex.calendar.logic.update.DistributedSemaphore;
import ru.yandex.calendar.logic.update.LockResource;
import ru.yandex.calendar.logic.update.LockTransactionManager;
import ru.yandex.calendar.logic.update.UpdateLock2;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.calendar.util.exception.ExceptionUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

@Slf4j
public class IcsEventImporter {
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EventInstanceStatusChecker eventInstanceStatusChecker;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private UpdateLock2 updateLock2;
    @Autowired
    private LockTransactionManager lockTransactionManager;
    @Autowired
    private IcsEventReplyHandler icsEventReplyHandler;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private UserManager userManager;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private EventChangesFinder eventChangesFinder;
    @Autowired
    private LastUpdateManager lastUpdateManager;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private EventUserRoutines eventUserRoutines;
    @Autowired
    private EventInfoDbLoader eventInfoDbLoader;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private SvcRoutines svcRoutines;
    @Autowired
    private EventsOnLayerChangeHandler eventsOnLayerChangeHandler;
    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private XivaNotificationManager xivaNotificationManager;
    @Autowired
    private CancelMeetingHandler cancelMeetingHandler;
    @Autowired
    private EventDeletionSmsHandler eventDeletionSmsHandler;
    @Autowired
    private EventMoveSmsHandler eventMoveSmsHandler;
    @Autowired
    private EwsExportRoutines ewsExportRoutines;
    @Autowired
    private NotificationRoutines notificationRoutines;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private Ml ml;
    @Autowired
    private SoChecker soChecker;
    @Autowired
    private EventsLogger eventsLogger;
    @Autowired
    private EventLayerDao eventLayerDao;
    @Autowired
    private DistributedSemaphore distributedSemaphore;

    private final DynamicProperty<Boolean> caldavCheckETag = new DynamicProperty<>("caldavCheckETag", false);

    private static boolean isIgnoreEventUser(IcsImportMode icsImportMode) {
        return
            icsImportMode.getActionSource() == ActionSource.MAILHOOK ||
            icsImportMode.getActionSource() == ActionSource.MAIL ||
            // https://jira.yandex-team.ru/browse/CAL-3071
            icsImportMode.getActionSource() == ActionSource.WEB_ICS ||
            icsImportMode.getLayerImportInfo().getLayerType() == LayerType.FEED;
    }

    private NotificationsData.Create getNotificationsCreateData(EventData eventData, IcsImportMode importMode) {
        if (importMode.getNotifications().isPresent()) {
            return NotificationsData.create(importMode.getNotifications().get());
        }
        if (isIgnoreEventUser(importMode)) {
            return NotificationsData.useLayerDefaultIfCreate();
        }
        return NotificationsData.createFromIcs(eventData.getEventUserData().getNotifications());
    }

    private NotificationsData.Update getNotificationsUpdateData(EventData eventData, IcsImportMode importMode) {
        if (importMode.getNotifications().isPresent()) {
            return importMode.getActionSource() == ActionSource.MAIL
                    ? NotificationsData.updateFromMail(importMode.getNotifications().get())
                    : NotificationsData.updateFromIcs(importMode.getNotifications().get());
        }
        if (isIgnoreEventUser(importMode)) {
            return NotificationsData.notChanged();
        }
        return NotificationsData.updateFromIcs(eventData.getEventUserData().getNotifications());
    }

    private IcsImportMails createEventGroup(
            PassportUid uid, IcsMethod method, IcsVEventGroup veventGroup, ListF<IcsVTimeZone> timezones,
            Optional<Long> layerId, IcsImportStats stats, IcsImportMode icsImportMode, Optional<String> url) {
        val organizersIds = getOrganizerIds(veventGroup);
        val isOrganizedByResource = organizersIds.stream().anyMatch(ParticipantId::isResource);

        if (isOrganizedByResource || organizerMismatches(uid, organizersIds, icsImportMode.getActionSource())) {
            log.warn("Ignoring organizer mismatched group creation");

            stats.addIgnoredEvents(veventGroup.getEvents().size());
            return IcsImportMails.empty();
        }

        val tzs = getTimezonesForNewEvent(uid, veventGroup, timezones);

        val mainEventData = convertEventData(uid, veventGroup.getMasterOrFirstEvent(), tzs, icsImportMode, layerId);
        mainEventData.setExternalId(Option.of(veventGroup.getUid().getOrElse(CalendarUtils.generateExternalId())));

        val mainEventId = eventRoutines.createMainEvent(uid, mainEventData, icsImportMode.getActionInfo());

        Option<String> startTzId = veventGroup.getMasterOrFirstEvent().getStart().getTzId();

        if (startTzId.isPresent() && !IcsTimeZones.forId(startTzId.get()).isPresent()) {
            EventTimezoneInfo info = new EventTimezoneInfo();
            info.setMainEventId(mainEventId);
            info.setOriginalIcsId(startTzId);
            mainEventDao.insertEventTimezoneInfo(info);
        }

        Optional<Long> lastMasterEventId = Optional.empty();
        Option<EventInfo> masterEventInfo = Option.empty();

        IcsImportMails mails = IcsImportMails.empty();

        for (IcsVEvent vevent : veventGroup.getEvents().sortedBy(IcsVEvent.isRecurrenceF().asFunction())) {
            if (!masterEventInfo.isPresent() && lastMasterEventId.isPresent()) {
                masterEventInfo = Option.of(eventInfoDbLoader.getEventInfoByEvent(
                        Option.of(uid), eventDbManager.getEventByIdForUpdate(lastMasterEventId.get()),
                        icsImportMode.getActionSource()));
            }

            log.info("Creating {}", LogMarker.RECURRENCE_ID.format(vevent.getRecurrenceIdInstantUtc().getOrNull()));

            val eventData = convertEventData(uid, vevent, tzs, icsImportMode, layerId);
            setAvailabilityAndDecisionForActor(uid, vevent, eventData, Option.empty(), icsImportMode);

            final IcsImportCreateInfo create;

            val actorAttendeeO = getAttendeeFromVeventIfPresent(vevent, uid);
            if (!actorAttendeeO.isPresent() && eventRoutines.eventHasConflicts(uid, eventData, icsImportMode.getActionInfo())) {
                log.info("Creating local copy without attendees");
                // workaround against lightning uid rewrite
                // event is not found by uid but is likely to be a copy of existing meeting
                create = createLocalEventCopyWithoutAttendees(vevent, uid, mainEventId, eventData, method, icsImportMode, stats, url);
            } else if (eventData.getEvent().getRecurrenceId().isPresent() && masterEventInfo.isPresent()) {
                create = createRecurrence(vevent, uid, masterEventInfo.get(), eventData, tzs, stats, icsImportMode, url);
            } else {
                create = createEvent(vevent, uid, mainEventId, eventData, stats, icsImportMode, url);
            }

            if (!eventData.getEvent().getRecurrenceId().isPresent() && create.getEventId().isPresent()) {
                lastMasterEventId = create.getEventId();
            }
            mails = invitationProcessingMode(vevent, uid, icsImportMode).isSend()
                    ? mails.plus(create.getMails())
                    : mails.plusReplyAndNotifyMails(
                            create.getMails().getReplyMails(), create.getMails().getNotifyMails());
        }
        return mails;
    }

    // @see ru.yandex.calendar.logic.ics.exp.IcsEventExporter#exportEvent
    private void restoreChangedByExportEventFields(PassportUid uid, EventData eventData) { // CAL-6294
        if (resourceRoutines.selectResourceEmails(eventData.getParticipantEmails()).isNotEmpty()) {
            eventData.getEvent().unsetField(EventFields.LOCATION);
        }
        if (eventData.getEvent().getUrl().exists(s -> s.startsWith(svcRoutines.getEventPageUrlPrefixForUid(uid)))) {
            eventData.getEvent().setUrlNull();
        }
        eventData.getEvent().setDescription(IcsEventExporter.ROOM_PHONES_IN_DESCRIPTION_PATTERN
                .matcher(eventData.getEvent().getDescription()).replaceAll(""));
    }

    // CAL-5456
    private void setAvailabilityAndDecisionForActor(
            PassportUid actorUid, IcsVEvent vevent, EventData eventData,
            Option<EventUser> oldEventUser, IcsImportMode importMode)
    {
        Option<ParticipantData> participantData = findUserParticipantData(actorUid, vevent, eventData);
        SettingsInfo settingsInfo = settingsRoutines.getSettingsByUid(actorUid);

        boolean isIgnoreEventUser = isIgnoreEventUser(importMode);
        boolean isAutoAcceptInv = settingsInfo.getCommon().getInvAcceptType() != InvAcceptingType.MANUAL;

        Option<Availability> dataAvailability = eventData.getEventUserData().getEventUser()
                .getFieldValueO(EventUserFields.AVAILABILITY);

        Option<Availability> defaultIcsAvailability = settingsInfo.getYt().map(SettingsYt.getDefaultIcsAvailabilityF());

        Option<Decision> autoAcceptDecision = Option.when(isAutoAcceptInv,
                () -> participantData.exists(p -> p.getDecision() == Decision.NO) ? Decision.NO : Decision.YES);

        Option<Decision> oldDecision = oldEventUser.map(EventUser::getDecision);

        ActionSource source = importMode.getActionSource();

        final Decision decision;
        final Availability availability;

        if (importMode.getDecision().isPresent()) {
            decision = importMode.getDecision().get();
            availability = dataAvailability.getOrElse(Availability.byDecision(decision));

        } else if (!isIgnoreEventUser && participantData.isPresent()) {
            decision = autoAcceptDecision.getOrElse(participantData.get().getDecision());
            availability = dataAvailability.getOrElse(Availability.byDecision(decision));

        } else if (!isIgnoreEventUser) {
            decision = oldEventUser.map(EventUser.getDecisionF()).getOrElse(Decision.YES);
            availability = dataAvailability.getOrElse(Availability.byDecision(decision));

        } else if (oldEventUser.isPresent() && (source != ActionSource.WEB_ICS || !oldDecision.isSome(Decision.NO))) {
            decision = oldEventUser.get().getDecision();
            availability = oldEventUser.get().getAvailability();

        } else if (!actorUid.isYandexTeamRu() && (participantData.isPresent() || source.isFromMailOrHook())) { // CAL-7856, CAL-10413
            decision = autoAcceptDecision.getOrElse(Decision.UNDECIDED);
            availability = participantData.map(p -> Availability.byDecision(decision))
                    .orElse(defaultIcsAvailability).getOrElse(Availability.AVAILABLE);

        } else if (participantData.isPresent()) {
            decision = autoAcceptDecision.getOrElse(participantData.get().getDecision());
            availability = Availability.byDecision(decision);

        } else if (actorUid.isYandexTeamRu() && importMode.getActionSource().isFromMailOrHook()) {
            decision = Decision.YES;

            ListF<ParticipantId> mailRecipientIds = importMode.getMailRecipientIds().getOrElse(Cf.list());

            if (mailRecipientIds.containsTs(ParticipantId.yandexUid(actorUid))
                || userIsSubscribedToInboxViaMailList(actorUid, eventData.getParticipantEmails(), mailRecipientIds))
            {
                availability = Availability.BUSY; // CAL-6940
            } else {
                availability = defaultIcsAvailability.getOrElse(Availability.AVAILABLE);
            }
        } else {
            decision = Decision.YES;
            availability = defaultIcsAvailability.getOrElse(Availability.AVAILABLE);
        }

        EventUser eventUser = eventData.getEventUserData().getEventUser().copy();
        eventUser.setDecision(decision);
        eventUser.setAvailability(availability);

        if (importMode.getActionSource() == ActionSource.MAIL) {
            boolean isParticipant = participantData.isPresent() || oldEventUser.exists(EventUser::getIsAttendee);
            eventUser.setIsSubscriber(passportAuthDomainsHolder.containsYandexTeamRu() && !isParticipant);
        }

        eventData.setEventUserData(eventData.getEventUserData().withEventUser(eventUser));
    }

    private Option<ParticipantData> findUserParticipantData(PassportUid uid, IcsVEvent vevent, EventData eventData) {
        ListF<ParticipantData> participants = eventData.getInvData().getParticipantsData().getParticipantsSafe();

        Option<ParticipantData> participant = participants
                .find(ParticipantData.getEmailF().andThenEquals(userManager.getEmailByUid(uid).get()));

        if (participant.isPresent()) return participant;

        return getAttendeeFromVeventIfPresent(vevent, uid).filterMap(IcsEventDataConverter.attendeeToParticipantDataF());
    }

    private boolean userIsSubscribedToInboxViaMailList(
            PassportUid uid, ListF<Email> attendees, ListF<ParticipantId> mailRecipients)
    {
        if (!uid.isYandexTeamRu()) return false;

        ListF<Email> recipientEmails = mailRecipients.flatMap(ParticipantId.getEmailIfExternalUserF());
        ListF<Email> attendeeEmails = eventInvitationManager.getParticipantIdsByEmails(attendees)
                .filterMap(t -> t.get2().getEmailIfExternalUser());

        ListF<Email> emails = recipientEmails.plus(attendeeEmails).stableUnique();

        if (emails.isEmpty()) return false;

        Option<String> login = userManager.getYtUserLoginByUid(uid);

        if (!login.isPresent()) return false;

        try {
            return ml.userIsSubscribedToInboxViaMailList(login.get(), emails);
        } catch (Exception e) {
            log.error("Failed to get user maillist subscription", e);
            return false;
        }
    }

    private EventData convertEventData(
            PassportUid uid, IcsVEvent vevent, IcsVTimeZones tzs, IcsImportMode importMode, Optional<Long> layerId)
    {
        EventData eventData = IcsEventDataConverter.convert(tzs, vevent, isIgnoreEventUser(importMode));
        eventData.setLayerId(getCategoryLayerId(uid, vevent, importMode).orElse(Option.x(layerId)));

        restoreChangedByExportEventFields(uid, eventData);

        if (importMode.getActionSource().isFromMailOrHook()) {
            ParticipantsData data = eventData.getInvData().getParticipantsData();
            ListF<Email> resourceEmails = resourceRoutines.selectResourceEmails(data.getParticipantEmailsSafe());

            if (resourceEmails.isNotEmpty()) { // CAL-5630
                log.info("Processing meeting from mail without resources");
                eventData.setInvData(data.excludeParticipants(resourceEmails));
            }
        }
        if (isPublicExternalOrganizedEvent(IcsVEventGroup.single(vevent), importMode.getActionSource())) { // CAL-7933
            eventData.setParticipantsData(eventData.getInvData().withMakePseudoLocalCopy());
        }
        return eventData;
    }

    private IcsImportMails updateEventGroup(
            PassportUid uid, IcsMethod method, IcsVEventGroup veventGroup, ListF<IcsVTimeZone> timezones,
            Optional<Long> layerIdO, IcsImportStats stats, IcsImportMode icsImportMode, MainEvent mainEvent, Optional<String> url)
    {
        log.info("Updating {}", LogMarker.MAIN_EVENT_ID.format(mainEvent.getId()));

        MainEventWithRelations mainEventWithRelations =
                eventDbManager.getMainEventWithRelationsByMainEventForUpdate(mainEvent);
        DateTimeZone eventTz = mainEventWithRelations.getEventsWithRelations().first().getTimezone();
        IcsVTimeZones tzs = IcsVTimeZones.cons(
                timezones, eventTz, veventGroup.getEvents().first().isRepeatingOrRecurrence());

        veventGroup = excludeExdatesThatAreUserMissedRecurrences(uid, layerIdO, mainEventWithRelations, veventGroup, tzs);

        final IcsEventChangesInfo changes = IcsEventChangesFinder.changes(mainEventWithRelations, veventGroup, tzs);

        Function<Instant, LocalDate> dateF = AuxDateTime.instantDateF(eventTz);

        MapF<Long, RepetitionInstanceInfo> repetitionInfoByEventId =
                repetitionRoutines.getRepetitionInstanceInfosAmongEvents(mainEventWithRelations.getEvents());

        SetF<Instant> addedExdates = Cf.set();

        IcsImportMails mails = IcsImportMails.empty();
        UserInfo user = userManager.getUserInfo(uid);

        PassportUid remoteUid = icsImportMode.getMailFromUid().getOrElse(uid);

        ActionInfo actionInfo = icsImportMode.getActionInfo();
        ActionSource actionSource = icsImportMode.getActionSource();

        boolean gotStaleMaster = false;

        val updateEventAuthInfoById = authorizer.loadEventsInfoForPermsCheck(user, changes.getUpdateEvents().get1());
        for (Tuple2<EventWithRelations, IcsVEvent> t : changes.getUpdateEvents()) {
            EventWithRelations eventWithRelations = t._1;
            RepetitionInstanceInfo repetitionInfo = repetitionInfoByEventId.getOrThrow(t._1.getId());

            IcsVEvent vevent = t._2;
            long eventId = eventWithRelations.getId();

            log.info("Updating {} with {}", LogMarker.EVENT_ID.format(eventId),
                    LogMarker.RECURRENCE_ID.format(vevent.getRecurrenceIdInstantUtc().getOrNull()));

            EventSynchData eventSynchData = IcsUtils.createIcsSynchData(vevent, tzs);
            EventInstanceStatusInfo status = eventInstanceStatusChecker.getStatusBySubjectAndSyncData(
                    UidOrResourceId.user(uid), eventSynchData, eventWithRelations, icsImportMode.getActionSource());

            final boolean acceptOutdated = actionSource.isFromMailOrHook()
                    || actionSource == ActionSource.WEB_ICS
                    && eventWithRelations.findUserEventUser(uid).exists(eu -> eu.getDecision() == Decision.NO);

            if (status.isAlreadyUpdated() && !vevent.getRecurrenceId().isPresent()) {
                gotStaleMaster = true;
            }
            if (status.isNotFound() || status.isAlreadyUpdated() && !acceptOutdated) {
                stats.addIgnoredEventId(eventId);
                continue;
            }
            if (icsImportMode.getLayerImportInfo().getLayerType() == LayerType.FEED
                && eventWithRelations.findOwnUserLayer(uid).exists(Layer.getTypeF().andThenEquals(LayerType.USER)))
            {
                // XXX: shouldn't feed event be event updated?
                log.debug("Event ignored, because it already lies in layer with type 'user'");
                continue;
            }

            final IcsImportMails updateMails;
            EventData eventData = convertEventData(uid, vevent, tzs, icsImportMode, layerIdO);
            setAvailabilityAndDecisionForActor(
                    uid, vevent, eventData, eventWithRelations.findUserEventUser(uid), icsImportMode);

            val canEdit = authorizer.canEditEvent(user, updateEventAuthInfoById.get(eventWithRelations.getId()), actionSource);
            val isAttendee = eventWithRelations.getParticipants().isAttendee(uid);

            boolean organizerMismatches = organizerMismatches(
                    uid, Option.of(eventWithRelations), getOrganizerId(vevent), actionSource);

            boolean acceptChanges;

            if (passportAuthDomainsHolder.containsYandexTeamRu() || actionSource.isCaldav()) {
                acceptChanges = actionSource.isFromMailHook() || canEdit && !isAttendee && !organizerMismatches;
            } else {
                acceptChanges = !organizerMismatches;
            }

            if (!status.isAlreadyUpdated() && acceptChanges) {
                if (!eventWithRelations.getRecurrenceId().isPresent()) {
                    Function<Rdate, LocalDate> rdateF = Rdate.getStartTsF().andThen(dateF);
                    addedExdates = eventData.findExdates()
                            .filter(rdateF.andThen(repetitionInfo.getExdates().map(rdateF).containsF().notF()))
                            .map(Rdate.getStartTsF()).unique();

                    eventData.excludeExdates(addedExdates);
                }
                if (soChecker.isEventSpam(uid, vevent, icsImportMode.getActionInfo(), url)) {
                    continue;
                }
                updateMails = updateEvent(
                        uid, eventId, eventData, eventSynchData.getSequenceAndDtstamp(), status, icsImportMode, stats);

            } else {
                updateMails = processOnlyUserRelatedStuff(
                        uid, vevent, tzs, eventId,
                        eventWithRelations, repetitionInfo,
                        EventInstanceParameters.fromEvent(eventWithRelations.getEvent()),
                        eventData, icsImportMode.getActionInfo(), stats, status, icsImportMode);
            }
            mails = invitationProcessingMode(vevent, uid, icsImportMode).isSend()
                    ? mails.plus(updateMails)
                    : mails.plusReplyAndNotifyMails(updateMails.getReplyMails(), updateMails.getNotifyMails());
        }

        for (IcsVEvent vevent : changes.getCreateEvents().filter(IcsVEvent.isRecurrenceF().notF())) {
            log.info("Creating {}", LogMarker.RECURRENCE_ID.format(null));

            EventData eventData = convertEventData(uid, vevent, tzs, icsImportMode, layerIdO);
            setAvailabilityAndDecisionForActor(uid, vevent, eventData, Option.<EventUser>empty(), icsImportMode);

            addedExdates = eventData.findExdates().map(Rdate.getStartTsF()).unique();
            eventData.excludeExdates(addedExdates);

            IcsImportMails ms;

            if (!organizerMismatches(uid, getOrganizerId(vevent), actionSource)) {
                ms = createEvent(vevent, uid, mainEvent.getId(), eventData, stats, icsImportMode, url).getMails();
            } else {
                log.warn("Ignoring organizer mismatched event creation");
                ms = IcsImportMails.empty();
            }

            mails = invitationProcessingMode(vevent, uid, icsImportMode).isSend()
                    ? mails.plus(ms)
                    : mails.plusReplyAndNotifyMails(ms.getReplyMails(), ms.getNotifyMails());
        }

        Option<Long> masterEventId = eventRoutines.findMasterEventByMainId(mainEvent.getId()).map(Event.getIdF());
        Option<Event> masterEvent = masterEventId.map(eventDbManager::getEventByIdForUpdate);
        Option<EventInfo> masterEventInfo = eventInfoDbLoader.getEventInfosByEvents(
                Option.of(uid), masterEvent, icsImportMode.getActionSource()).singleO();

        Option<EventWithRelations> masterEventWithRelations = masterEventInfo.map(EventInfo::getEventWithRelations);
        final var masterEventAuthInfo = masterEventWithRelations.toOptional()
            .map(eventInfo -> authorizer.loadEventInfoForPermsCheck(user, eventInfo));
        Option<EventUser> masterEventUser = masterEventWithRelations
                .flatMapO(e -> e.getEventUsers().find(eu -> eu.getUid().sameAs(uid)));

        ListF<IcsVEvent> recurrencesToCreate = changes.getCreateEvents().filter(IcsVEvent.isRecurrenceF());

        if (gotStaleMaster && recurrencesToCreate.isNotEmpty()) {
            log.info("Ignoring recurrences creation because of stale master");
            recurrencesToCreate = Cf.list();
        }

        for (IcsVEvent vevent : recurrencesToCreate) {
            log.info("Creating {}", LogMarker.RECURRENCE_ID.format(vevent.getRecurrenceIdInstantUtc().getOrNull()));

            final IcsImportMails createMails;
            EventData eventData = convertEventData(uid, vevent, tzs, icsImportMode, layerIdO);
            setAvailabilityAndDecisionForActor(uid, vevent, eventData, masterEventUser, icsImportMode);

            Instant recurrenceId = eventData.getEvent().getRecurrenceId().get();

            if (!masterEventInfo.isPresent()) {
                log.warn("Master event is missing, importing recurrence");
                createMails = createEvent(vevent, uid, mainEvent.getId(), eventData, stats, icsImportMode, url).getMails();

            } else if (addedExdates.containsTs(recurrenceId) ||
                    !RepetitionUtils.isValidStart(masterEventInfo.get().getRepetitionInstanceInfo(), recurrenceId))
            {
                log.warn("Ignoring recurrence with invalid recurrence-id"); // CAL-6470
                stats.addIgnoredEvent();
                createMails = IcsImportMails.empty();

            } else if (isNewRecurrenceWithChangedParticipation(masterEventInfo.get(), eventData, uid)) {
                createMails = createRecurrenceWithChangedParticipation(
                        vevent, uid, masterEventInfo.get(), eventData, tzs, stats, icsImportMode, url).getMails();

            } else if (organizerMismatches(uid, masterEventWithRelations, getOrganizerId(vevent), actionSource)) {
                log.warn("Ignoring organizer mismatched event creation");
                stats.addIgnoredEvent();
                createMails = IcsImportMails.empty();

            } else if (icsImportMode.getActionSource().isFromMailHook()
                    || authorizer.canEditEvent(user, masterEventAuthInfo.get(), actionSource)) {
                createMails = createRecurrence(vevent, uid, masterEventInfo.get(), eventData, tzs, stats, icsImportMode, url).getMails();
            } else {
                log.warn("Ignoring new event from attendee");
                stats.addIgnoredEvent();
                createMails = IcsImportMails.empty();
            }
            mails = invitationProcessingMode(vevent, uid, icsImportMode).isSend()
                    ? mails.plus(createMails)
                    : mails.plusReplyAndNotifyMails(createMails.getReplyMails(), createMails.getNotifyMails());
        }

        MapF<Instant, EventWithRelations> exdateRecurrencesByInstant = changes.getDeleteEvents()
                .filter(e -> e.getRecurrenceId().isPresent())
                .zipWithFlatMapO(Cf2.<EventWithRelations, Instant>f(e -> e.getRecurrenceId().get())
                        .andThen(dateF.andThen(addedExdates.toMapMappingToKey(dateF)::getO))).invert().toMap();

        ListF<EventWithRelations> goneEvents = changes.getDeleteEvents()
                .filterNot(e -> exdateRecurrencesByInstant.values().exists(r -> r.getId() == e.getId()));

        SetF<Long> possiblyOrphanedMainEventIds = Cf.hashSet();

        if (method.sameAs(IcsMethod.PUBLISH) && !icsImportMode.getActionSource().isFromMailOrHook()) {
            for (EventWithRelations eventWithRelations : goneEvents) {
                long eventId = eventWithRelations.getId();
                log.info("Deleting {}", LogMarker.EVENT_ID.format(eventId));

                if (eventWithRelations.ownerIs(uid)) {
                    eventRoutines.deleteEventsFromDbAndExchange(ActorId.user(uid), Cf.list(eventId), icsImportMode.getActionInfo(), false);
                    possiblyOrphanedMainEventIds.add(eventWithRelations.getMainEventId());
                    stats.addDeletedEventId(eventId);

                } else {
                    stats.addIgnoredEventId(eventId);
                }
            }
        }
        if (addedExdates.isNotEmpty()) {
            repetitionRoutines.createExdates(masterEventId.get(), addedExdates.toList(), icsImportMode.getActionInfo());

            EventWithRelations me = masterEventInfo.get().getEventWithRelations();
            RepetitionInstanceInfo mr = masterEventInfo.get().getRepetitionInstanceInfo();

            eventsLogger.log(EventChangeLogEvents.updated(ActorId.user(uid),
                    me, mr, me, mr.withExdates(addedExdates.toList().map(RepetitionUtils::consExdate))), actionInfo);
        }
        ListF<EventWithRelations> eventsToDelete = Cf.arrayList();
        Tuple2List<EventWithRelations, EventInstanceParameters> exdateInstances = Tuple2List.arrayList();

        boolean isExportedWithEws = mainEvent.getIsExportedWithEws().getOrElse(false);

        for (Instant exdate : addedExdates) {
            Option<EventWithRelations> recurrence = exdateRecurrencesByInstant.getO(exdate);
            EventWithRelations event = recurrence.orElse(masterEventWithRelations).get();

            RepetitionInstanceInfo repetitionInfo = recurrence.map(r -> repetitionInfoByEventId.getOrThrow(r.getId()))
                    .orElse(masterEventInfo.map(EventInfo::getRepetitionInstanceInfo))
                    .orElseThrow(() -> new NoSuchElementException("No repetition found for event " + event.getId()));

            EventInstanceParameters instance;
            if (recurrence.isPresent()) {
                instance = new EventInstanceParameters(
                        event.getEvent().getStartTs(), event.getEvent().getEndTs(), Option.of(exdate));
            } else {
                instance = new EventInstanceParameters(
                        exdate, repetitionRoutines.calcEndTs(event.getEvent(), exdate), Option.of(exdate));
            }
            ListF<EventSendingInfo> sendingInfos = cancelMeetingHandler.cancelMeeting(
                    recurrence.orElse(masterEventWithRelations).get().getEvent(), Option.of(uid),
                    instance, event.isParkingOrApartmentOccupation(), isExportedWithEws, icsImportMode.getActionInfo());

            ListF<EventOnLayerChangeMessageParameters> notifyMails = eventsOnLayerChangeHandler.handleEventDelete(
                    UidOrResourceId.user(remoteUid), event, repetitionInfo, instance, icsImportMode.getActionInfo());

            boolean isSend = invitationProcessingMode(Either.right(event), uid, icsImportMode).isSend();

            if (recurrence.isPresent()) {
                if (isSend) {
                    mails = mails.plus(IcsImportMails.eventMails(
                            eventInvitationManager.createEventInvitationOrCancelMails(
                                    ActorId.user(uid), event, repetitionInfo,
                                    sendingInfos, icsImportMode.getActionInfo())));
                }
                mails = mails.plus(IcsImportMails.notifyMails(notifyMails));
                eventsToDelete.addAll(event);

            } else {
                mails = isSend
                        ? mails.plus(IcsImportMails.updateAndNotifyMails(sendingInfos, notifyMails))
                        : mails.plusReplyAndNotifyMails(Cf.list(), notifyMails);
            }
            exdateInstances.add(event, instance);
        }
        eventDeletionSmsHandler.handleEventClosestOccurrenceDeletion(
                remoteUid, exdateInstances, icsImportMode.getActionInfo());

        for (EventWithRelations event : eventsToDelete) {
            eventRoutines.deleteEventsFromDbAndExchange(ActorId.user(uid), Cf.list(event.getId()), icsImportMode.getActionInfo(), false);
            possiblyOrphanedMainEventIds.add(event.getMainEventId());
            stats.addDeletedEventId(event.getId());
        }
        if (masterEventWithRelations.isPresent() && addedExdates.isNotEmpty()) {
            lastUpdateManager.updateTimestampsAsync(masterEvent.map(Event::getMainEventId), icsImportMode.getActionInfo());
            notificationRoutines.recalcAllNextSendTs(masterEventId, icsImportMode.getActionInfo());

            ewsExportRoutines.cancelOccurrencesIfNeeded(
                    masterEventWithRelations.get(), addedExdates.toList(), icsImportMode.getActionInfo());
        }

        InstantInterval suitableInterval = eventMoveSmsHandler.suitableInterval(actionInfo.getNow());
        ListF<Long> exdateEventIds = exdateInstances.get1().map(EventWithRelations::getId);

        ListF<EventInstanceInterval> moveWatchingInstances = mainEventWithRelations.getEvents()
                .filter(Event.getIdF().andThen(exdateEventIds.containsF().notF()))
                .zipWith(e -> repetitionInfoByEventId.getOrThrow(e.getId()))
                .map(EventAndRepetition.consF())
                .flatMap(EventAndRepetition.getInstancesStartingInIntervalF(suitableInterval));

        eventMoveSmsHandler.handleEventUpdate(remoteUid, moveWatchingInstances, Cf.list(mainEvent), actionInfo);

        mainEventDao.deleteTimezoneInfosByMainEventIdsIfOrphaned(possiblyOrphanedMainEventIds);
        mainEventDao.deleteMainEventsIfOrphaned(possiblyOrphanedMainEventIds.toList());

        return mails;
    }

    private IcsVTimeZones getTimezonesForNewEvent(
            PassportUid uid, IcsVEventGroup group, ListF<IcsVTimeZone> timezones)
    {
        IcsVEvent someVevent = group.getMasterOrFirstEvent();
        IcsVTimeZones tzs = IcsVTimeZones.cons(
                timezones, dateTimeManager.getTimeZoneForUid(uid), someVevent.isRepeatingOrRecurrence());

        Option<String> startTzId = someVevent.getStart().getTzId();
        if (startTzId.isPresent()) {
            return tzs.withFallback(tzs.getOrFallbackForDate(startTzId.get(), someVevent.getStart().getLocalDateTime()));
        }
        Option<ParticipantId> organizer = someVevent.getOrganizerEmailSafe()
                .map(eventInvitationManager.getParticipantIdByEmailF());

        if (organizer.isPresent() && organizer.get().isYandexUser()) {
            return tzs.withFallback(dateTimeManager.getTimeZoneForUid(organizer.get().getUid()));
        }
        if (organizer.isPresent() && organizer.get().isResource()) {
            return tzs.withFallback(resourceRoutines.getTimeZoneByResourceId(organizer.get().getResourceId()));
        }
        return tzs;
    }

    private boolean isNotMeetingOrUserIsOrganizer(IcsVEvent vevent, PassportUid uid) {
        Option<ParticipantId> veventOrganizerParticipantId = vevent.getOrganizerEmailSafe()
                .map(eventInvitationManager.getParticipantIdByEmailF());
        return !veventOrganizerParticipantId.isPresent() || veventOrganizerParticipantId.get().isYandexUserWithUid(uid);
    }

    private boolean isNewRecurrenceWithChangedParticipation(EventInfo masterEvent, EventData eventData, PassportUid uid) {
        EventInstanceForUpdate eventInstanceForModifier = eventRoutines.getEventInstanceForModifier(
                Option.of(uid), Option.of(eventData.getEvent().getRecurrenceId().get()),
                masterEvent, Option.<Long>empty());

        EventChangesInfo eventChangesInfo = eventChangesFinder.getEventChangesInfo(eventInstanceForModifier,
                eventData, NotificationsData.notChanged(), Option.of(uid), false, false);

        return eventChangesInfo.isNewRecurrenceWithChangedParticipation(UidOrResourceId.user(uid));
    }

    private InvitationProcessingMode invitationProcessingMode(
            IcsVEvent vevent, PassportUid actorUid, IcsImportMode importMode)
    {
        return invitationProcessingMode(Either.left(vevent), actorUid, importMode);
    }

    private InvitationProcessingMode invitationProcessingMode(
            Either<IcsVEvent, EventWithRelations> event, PassportUid actorUid, IcsImportMode importMode)
    {
        if (!event.fold(IcsVEvent::isMeeting, EventWithRelations::isMeeting)) {
            return InvitationProcessingMode.SAVE_ONLY;
        }
        if (!event.fold(
            v -> isNotMeetingOrUserIsOrganizer(v, actorUid),
            e -> e.getParticipants().getOrganizerIdWithInconsistent().containsTs(ParticipantId.yandexUid(actorUid))))
        { // CAL-4152
            return InvitationProcessingMode.SAVE_ATTACH;
        }
        if (importMode.getActionSource() == ActionSource.CALDAV) {
            return InvitationProcessingMode.SAVE_ATTACH_SEND;
        }
        return InvitationProcessingMode.SAVE_ATTACH;
    }

    private boolean isSendReplyMails(ActionSource actionSource) {
        return Cf.list(ActionSource.CALDAV, ActionSource.MAIL).containsTs(actionSource);
    }

    // see EwsImporter#processOnlyUserRelatedStuff
    private IcsImportMails processOnlyUserRelatedStuff(
            PassportUid actorUid, IcsVEvent vevent, IcsVTimeZones tzs, long eventId,
            EventWithRelations metaEvent, RepetitionInstanceInfo metaRepetition, EventInstanceParameters instance,
            EventData eventData, ActionInfo actionInfo, IcsImportStats stats,
            EventInstanceStatusInfo eventInstanceStatusInfo, IcsImportMode importMode)
    {
        Option<EventUser> oldEventUser = eventUserDao.findEventUserByEventIdAndUid(eventId, actorUid);
        EventUser newEventUser = eventData.getEventUserData().getEventUser();

        EventAttachedLayerId layerId = eventRoutines.createOrUpdateEventLayer(
                userManager.getUserInfo(actorUid), eventData.getLayerId(),
                PassportSid.CALENDAR, eventDbManager.getEventAndRepetitionByIdForUpdate(eventId),
                getEventType(importMode), false, actionInfo);

        // XXX: decide if we should use notification from vevent if user is not participant
        if (!oldEventUser.isPresent()) {
            eventUserRoutines.saveEventUserAndNotification(
                    actorUid, eventId, newEventUser, getNotificationsCreateData(eventData, importMode),
                    layerId.getCurrentLayerId(), actionInfo);
        } else {
            eventUserRoutines.updateEventUserAndNotification(
                    oldEventUser.get().getId(), newEventUser,
                    getNotificationsUpdateData(eventData, importMode), actionInfo);
        }

        // before we could delete actor's event user
        lastUpdateManager.updateTimestampsAsync(metaEvent.getMainEventId(), actionInfo);

        Option<Decision> newDecision = newEventUser.getFieldValueO(EventUserFields.DECISION);
        Option<String> reason = vevent.getPropertyValue(PropertyNames.X_CALENDARSERVER_PRIVATE_COMMENT);
        Option<ParticipantData> participantDataO = findUserParticipantData(actorUid, vevent, eventData);

        if (participantDataO.isPresent()) {
            ParticipantData participantData = participantDataO.get()
                    .withDecision(newDecision.getOrElse(participantDataO.get().getDecision()));

            SequenceAndDtStamp userVersion = IcsUtils.createIcsSynchData(
                    vevent, tzs.withFallback(dateTimeManager.getTimeZoneForUid(actorUid))).getSequenceAndDtstamp();

            eventInvitationManager.updateUserDecision(
                    eventId, actionInfo, eventInstanceStatusInfo, userVersion, participantData, reason);
        }
        // XXX: wrong, may be ignored inside
        stats.addUpdatedEventId(eventId);

        boolean sendEmail = isSendReplyMails(actionInfo.getActionSource())
                && newDecision.isPresent() && !newDecision.isSome(Decision.UNDECIDED)
                && !newDecision.equals(oldEventUser.map(EventUser::getDecision));

        Option<ReplyMessageParameters> decisionMail = sendEmail
                ? eventRoutines.updateDecisionViaEwsOrElseCreateMailIfNeeded(
                        Cf.list(eventId), actorUid, newDecision.get(), Option.empty(), actionInfo).singleO()
                : Option.empty();

        ListF<EventOnLayerChangeMessageParameters> notify = eventsOnLayerChangeHandler.handleEventChange(
                UidOrResourceId.user(actorUid), metaEvent, metaRepetition, instance,
                LayerIdChangesInfo.changes(metaEvent.getLayerIds(), layerId.getAttachedId(), layerId.getDetachedId()),
                EventChangesInfoForMails.EMPTY, actionInfo);

        Email userEmail = participantDataO.map(ParticipantData::getEmail)
                .getOrElse(() -> settingsRoutines.getSettingsByUid(actorUid).getEmail());

        eventsLogger.log(EventChangeLogEvents.updated(ActorId.user(actorUid),
                new EventIdLogDataJson(eventId, metaEvent.getMainEvent(), instance.getOccurrenceId().toOptional()),
                userEmail, oldEventUser, eventUserRoutines.findEventUser(eventId, actorUid), Option.of(layerId)), actionInfo);

        return IcsImportMails.replyAndNotifyMails(decisionMail, notify);
    }

    private Option<IcsAttendee> getAttendeeFromVeventIfPresent(IcsVEvent vevent, PassportUid actorUid) {
        Option<Email> attendeeEmail = eventInvitationManager.getParticipantIdsByEmails(vevent.getAttendeeEmailsSafe())
                .findBy2(ParticipantId.isYandexUserWithUidF(actorUid))
                .map(Tuple2.<Email, ParticipantId>get1F());

        return vevent.getAttendees().find(IcsAttendee.getEmailSafeF().andThen(Cf2.isSomeOfF(attendeeEmail)));
    }

    private ListF<ParticipantId> getAttendeeIds(IcsVEventGroup group) {
        return eventInvitationManager.getParticipantIdsByEmails(
                group.getEvents().flatMap(IcsVEvent::getAttendeeEmailsSafe)).get2().stableUnique();
    }

    private ListF<ParticipantId> getOrganizerIds(IcsVEventGroup group) {
        return eventInvitationManager.getParticipantIdsByEmails(
                group.getEvents().filterMap(IcsVEvent::getOrganizerEmailSafe)).get2().stableUnique();
    }

    private Option<ParticipantId> getOrganizerId(IcsVEvent event) {
        return getOrganizerIds(IcsVEventGroup.single(event)).singleO();
    }

    private boolean organizerMismatches(
            PassportUid actorUid, ListF<ParticipantId> organizerIds, ActionSource actionSource) // CAL-7856
    {
        return organizerMismatches(actorUid, Option.empty(), organizerIds, actionSource);
    }

    boolean organizerMismatches(
            PassportUid actorUid,
            Option<EventWithRelations> event,
            ListF<ParticipantId> dataOrganizerIds,
            ActionSource actionSource
    ) {
        if (!passportAuthDomainsHolder.containsYandexTeamRu() && !actionSource.isCaldav()) {
            if (!event.isPresent()) {
                return false; // let pseudo local copy to be created
            }
            Option<ParticipantId> organizerId = event.get().getOrganizerIdSafe();

            if (organizerId.isPresent()) {
                return organizerId.get().isYandexUser();
            } else {
                return !event.get().getEvent().getCreationSource().isInvitationMailsFree();
            }
        } else {
            ListF<ParticipantId> organizerIds = event
                    .flatMapO(EventWithRelations::getOrganizerIdSafe)
                    .plus(dataOrganizerIds);

            return actionSource.isCaldav()
                    && organizerIds.exists(id -> id.getUidIfYandexUser().exists(uid -> !actorUid.sameAs(uid)));
        }
    }

    public Option<MainEvent> findMainEvent(PassportUid uid, IcsVEventGroup veventGroup, ActionSource actionSource) {
        if (veventGroup.getUid().isPresent()) {
            if (isPublicExternalOrganizedEvent(veventGroup, actionSource)) {
                return eventRoutines.getMainEventBySubjectAndExternalId(
                        UidOrResourceId.user(uid), veventGroup.getUid().get());
            } else {
                return eventRoutines.getMainEventBySubjectIdAndParticipantEmailsAndExternalId(
                        UidOrResourceId.user(uid), veventGroup.getParticipantEmailsSafe(), veventGroup.getUid().get());
            }
        } else {
            return Option.empty();
        }
    }

    private boolean isPublicExternalOrganizedEvent(IcsVEventGroup veventGroup, ActionSource actionSource) {
        return !passportAuthDomainsHolder.containsYandexTeamRu()
                && (!actionSource.isCaldav() || getOrganizerIds(veventGroup).exists(ParticipantId::isExternalUser));
    }

    public boolean isMeetingWithDeletedMaster(PassportUid uid, IcsVEventGroup veventGroup, ActionSource actionSource) {
        if (!veventGroup.getUid().isPresent() || !veventGroup.getEvents().exists(IcsVEvent.isMeetingF())) {
            return false;
        }
        if (isPublicExternalOrganizedEvent(veventGroup, actionSource)) {
            return eventRoutines.findDeletedMasterExistsByUid(veventGroup.getUid().get(), uid);
        }
        return eventRoutines.findDeletedMasterExistsByExternalIdAndParticipantEmails(
                veventGroup.getUid().get(), veventGroup.getParticipantEmailsSafe());
    }

    private IcsVEventGroup excludeExdatesThatAreUserMissedRecurrences(
            PassportUid uid, Optional<Long> layerId,
            MainEventWithRelations existing, IcsVEventGroup group, IcsVTimeZones tzs)
    {
        if (group.getMasterEvents().isEmpty()) return group;

        ListF<EventWithRelations> missedEvents = existing.getRecurrenceEvents().filterNot(layerId.isPresent()
                ? e -> e.findLayerById(layerId.get()).exists(
                        l -> e.findUserEventUser(l.getCreatorUid()).exists(eu -> eu.getDecision() != Decision.NO))
                : e -> e.findOwnUserLayer(uid).exists(
                        l -> e.findUserEventUser(uid).exists(eu -> eu.getDecision() != Decision.NO)));

        if (missedEvents.isEmpty()) return group;
        if (existing.getMasterEvents().isEmpty()) {
            log.info("Returning group because existing.getMasterEvents().isEmpty()..");
            return group;
        }

        DateTimeZone tz = existing.getMasterEvents().first().getTimezone();

        SetF<LocalDate> excluded = missedEvents.map(e -> new LocalDate(e.getRecurrenceId().get(), tz)).unique();

        ListF<IcsVEvent> masters = group.getMasterEvents().map(e ->
                e.withExdates(e.getExDates().flatMap(ex -> ex.getInstants(tzs).filterMap(
                        ts -> Option.when(!excluded.containsTs(new LocalDate(ts, tz)), new IcsExDate(ts))))));

        return new IcsVEventGroup(group.getUid(), masters.plus(group.getRecurrenceEvents()));
    }

    private ListF<Long> createOrUpdateEventGroup(
            PassportUid uid, IcsMethod method, IcsVEventGroup veventGroup, ListF<IcsVTimeZone> tzs,
            Optional<Long> layerId, IcsImportStats stats, IcsImportMode icsImportMode, Optional<String> url)
    {
        ActionSource actionSource = icsImportMode.getActionSource();
        Option<MainEvent> mainEvent = findMainEvent(uid, veventGroup, actionSource);

        if (!mainEvent.isPresent() && isMeetingWithDeletedMaster(uid, veventGroup, actionSource)) {
            String externalId = veventGroup.getUid().get();
            log.info("Skipping group {} that looks like already deleted", LogMarker.EXT_ID.format(externalId));

            stats.addIgnoredEvents(veventGroup.getEvents().size());
            return Cf.list();
        }
        if (mainEvent.isPresent()
            && caldavCheckETag.get()
            && icsImportMode.getNotModifiedSince().exists(mainEvent.get().getLastUpdateTs()::isAfter))
        {
            throw CommandRunException.createSituation("Outdated change", Situation.EVENT_MODIFIED);
        }

        final IcsImportMails mails;

        if (mainEvent.isPresent()) {
            mails = updateEventGroup(uid, method, veventGroup, tzs, layerId, stats, icsImportMode, mainEvent.get(), url);
        } else {
            mails = createEventGroup(uid, method, veventGroup, tzs, layerId, stats, icsImportMode, url);
        }
        ListF<EventMessageParameters> messages = eventInvitationManager.createEventInvitationOrCancelMails(
                ActorId.user(uid), mails.getUpdateMails(), icsImportMode.getActionInfo());

        ListF<EventMessageParameters> totalMails = messages.plus(mails.getEventMails()).plus(mails.getReplyMails()).plus(mails.getNotifyMails());
        eventInvitationManager.sendEventMails(
                totalMails,
                icsImportMode.getActionInfo());
        return totalMails.map(EventMessageParameters::getEventId);
    }

    private static EventType getEventType(IcsImportMode icsImportMode) {
        return icsImportMode.getLayerImportInfo().layerType.getMyEventType();
    }

    private IcsImportCreateInfo createEvent(IcsVEvent vevent, PassportUid actorUid, long mainEventId, EventData eventData,
                                            IcsImportStats icsImportStats, IcsImportMode icsImportMode, Optional<String> url) {
        if (soChecker.isEventSpam(actorUid, vevent, icsImportMode.getActionInfo(), url)) {
            return IcsImportCreateInfo.empty();
        }

        CreateInfo ci = eventRoutines.createUserOrFeedEvent(
                UidOrResourceId.user(actorUid), getEventType(icsImportMode), mainEventId, eventData,
                getNotificationsCreateData(eventData, icsImportMode),
                InvitationProcessingMode.SAVE_ATTACH, icsImportMode.getActionInfo());

        icsImportStats.addNewEventId(ci.getEvent().getId());

        return new IcsImportCreateInfo(
                ci.getEventId(),
                IcsImportMails.updateAndNotifyMails(ci.getSendingInfos(), ci.getLayerNotifyMails()));
    }

    private IcsImportCreateInfo createRecurrence(IcsVEvent vevent, PassportUid actorUid, EventInfo masterEvent,
                                                 EventData eventData, IcsVTimeZones tzs, IcsImportStats icsImportStats,
                                                 IcsImportMode icsImportMode, Optional<String> url) {
        if (soChecker.isEventSpam(actorUid, vevent, icsImportMode.getActionInfo(), url)) {
            return IcsImportCreateInfo.empty();
        }

        long recurrenceEventId = eventRoutines.createNotChangedRecurrence(
                actorUid, masterEvent, eventData.getEvent().getRecurrenceId().get(), icsImportMode.getActionInfo());

        eventData.getEvent().setId(recurrenceEventId);
        IcsEventSynchData syncData = IcsUtils.createIcsSynchData(
                vevent, tzs.withFallback(dateTimeManager.getTimeZoneForUid(actorUid)));

        UpdateInfo updateInfo = eventRoutines.updateEventFromIcsOrExchange(
                UidOrResourceId.user(actorUid), eventData,
                getNotificationsUpdateData(eventData, icsImportMode),
                EventInstanceStatusInfo.needToUpdate(recurrenceEventId),
                syncData.getSequenceAndDtstamp(), icsImportMode.getActionInfo());

        icsImportStats.addNewEventId(recurrenceEventId);

        return new IcsImportCreateInfo(
                recurrenceEventId,
                IcsImportMails.updateAndNotifyMails(updateInfo.getSendingInfos(), updateInfo.getLayerNotifyMails()));
    }

    private IcsImportCreateInfo createRecurrenceWithChangedParticipation(
            IcsVEvent vevent, PassportUid actorUid, EventInfo masterEvent, EventData eventData, IcsVTimeZones tzs,
            IcsImportStats icsImportStats, IcsImportMode icsImportMode, Optional<String> url) {
        if (soChecker.isEventSpam(actorUid, vevent, icsImportMode.getActionInfo(), url)) {
            return IcsImportCreateInfo.empty();
        }

        long recurrenceEventId = eventRoutines.createNotChangedRecurrence(
                actorUid, masterEvent, eventData.getEvent().getRecurrenceId().get(), icsImportMode.getActionInfo());

        icsImportStats.addNewEventId(recurrenceEventId);

        EventInstanceParameters instance = new EventInstanceParameters(
                eventData.getEvent().getStartTs(), eventData.getEvent().getEndTs(),
                Option.of(eventData.getEvent().getRecurrenceId().get()));

        IcsImportMails mails = processOnlyUserRelatedStuff(
                actorUid, vevent, tzs, recurrenceEventId,
                masterEvent.getEventWithRelations(), masterEvent.getRepetitionInstanceInfo(), instance,
                eventData, icsImportMode.getActionInfo(),
                icsImportStats, EventInstanceStatusInfo.needToUpdate(recurrenceEventId), icsImportMode);

        return new IcsImportCreateInfo(recurrenceEventId, mails);
    }

    private IcsImportCreateInfo createLocalEventCopyWithoutAttendees(
            IcsVEvent vevent, PassportUid uid, long mainEventId, EventData eventData, IcsMethod method,
            IcsImportMode icsImportMode, IcsImportStats stats, Optional<String> url) {
        if (soChecker.isEventSpam(uid, vevent, icsImportMode.getActionInfo(), url)) {
            return IcsImportCreateInfo.empty();
        }

        eventData.setInvData(ParticipantsData.notMeeting());
        EventType eType =
            method.sameAs(IcsMethod.REQUEST) ?
            EventType.USER : icsImportMode.getLayerImportInfo().layerType.getMyEventType();
        CreateInfo ci = eventRoutines.createUserOrFeedEvent(UidOrResourceId.user(uid), eType, mainEventId, eventData,
                getNotificationsCreateData(eventData, icsImportMode),
                InvitationProcessingMode.SAVE_ONLY, icsImportMode.getActionInfo());
        stats.addNewEventId(ci.getEvent().getId());

        return new IcsImportCreateInfo(ci.getEventId(), IcsImportMails.notifyMails(ci.getLayerNotifyMails()));
    }

    private IcsImportMails updateEvent(
            PassportUid uid, long eventId, EventData eventData, SequenceAndDtStamp sequenceAndDtStamp,
            EventInstanceStatusInfo status, IcsImportMode icsImportMode, IcsImportStats stats)
    {
        eventData.getEvent().setId(eventId);

        UpdateInfo updateInfo = eventRoutines.updateEventFromIcsOrExchange(
                UidOrResourceId.user(uid), eventData,
                getNotificationsUpdateData(eventData, icsImportMode),
                status, sequenceAndDtStamp, icsImportMode.getActionInfo());

        stats.addUpdatedEventId(eventData.getEvent().getId());

        return IcsImportMails.updateAndNotifyMails(updateInfo.getSendingInfos(), updateInfo.getLayerNotifyMails());
    }

    private Option<Long> getCategoryLayerId(PassportUid actorUid, IcsVEvent ve, IcsImportMode icsImportMode) {
        if (icsImportMode.getLayerImportInfo().getLayerReference().isByCategory()) {
            String cat = IcsUtils.getFirstCategory(ve.toComponent());
            // TODO: search in cache
            return Option.of(layerRoutines.getOrCreateFirstUserLayer(actorUid, StringUtils.notEmptyO(cat)));
            // TODO: put to cache
        }
        return Option.empty();
    }

    private ListF<Long> importEventGroup(
            PassportUid uid, final IcsVEventGroup veventGroup, IcsMethod method, ListF<IcsVTimeZone> tzs,
            IcsImportMode icsImportMode, final Optional<Long> layerId, IcsImportStats stats, Optional<String> url)
    {
        // XXX check permissions! // ssytnik@
        ListF<Long> affectedEventIds = Cf.arrayList();
        if (Cf.list(IcsMethod.PUBLISH, IcsMethod.REQUEST).containsTs(method)) {
            ListF<Long> modifiedEventIds = createOrUpdateEventGroup(uid, method, veventGroup, tzs, layerId, stats, icsImportMode, url);
            affectedEventIds.addAll(modifiedEventIds);
        } else if (IcsMethod.CANCEL.sameAs(method)) {
            affectedEventIds.addAll(icsEventReplyHandler.removeAttendeesMeeting(veventGroup, tzs, icsImportMode, uid));
        } else if (IcsMethod.REPLY.sameAs(method)) {
            if (getAttendeeIds(veventGroup).exists(p -> p.getUidIfYandexUser().exists(u -> !u.isYandexTeamRu()))) {
                log.warn("Ignoring reply from yandex user");
            } else {
                ListF<Long> modifiedEventIds = icsEventReplyHandler.handleReplyOnInternalEvent(veventGroup, uid, tzs, icsImportMode.getActionInfo());
                affectedEventIds.addAll(modifiedEventIds);
            }
        } else {
            throw new NotImplementedException(
                    "Not implemented ics type " + method.getValue());
        }
        return affectedEventIds;
    }

    private ListF<Long> importEventGroupsWithTransactionAndLog(
            final PassportUid uid, final IcsVEventGroup veventGroup, final IcsMethod method, ListF<IcsVTimeZone> tzs,
            final IcsImportMode icsImportMode, final Optional<Long> layerId,
            final IcsImportStats stats, Optional<String> url)
    {
        log.info("Processing {}", LogMarker.EXT_ID.format(veventGroup.getUid().getOrElse("?")));
        final long processedCount = stats.getProcessedCount();

        ListF<Long> allAffectedEventIds = Cf.arrayList();
        try {
            try (var lock = distributedSemaphore.forceTryAcquire("importEventGroup")) {
                if (veventGroup.getUid().isPresent()) {
                    lockTransactionManager.lockAndDoInTransaction(LockResource.event(new ExternalId(veventGroup.getUid().get())), () -> {
                        ListF<Long> affectedEventIds = importEventGroup(uid, veventGroup, method, tzs, icsImportMode,
                                layerId, stats, url);
                        allAffectedEventIds.addAll(affectedEventIds);
                    });
                } else {
                    lockTransactionManager.doInTransaction(() -> {
                        ListF<Long> affectedEventIds = importEventGroup(uid, veventGroup, method, tzs, icsImportMode,
                                layerId, stats, url);
                        allAffectedEventIds.addAll(affectedEventIds);
                    });
                }
            }
        } catch (RuntimeException e) {
            ExceptionUtils.rethrowIfTlt(e);

            if (icsImportMode.getActionSource() == ActionSource.WEB_ICS) {
                log.error("Failed processing {}", veventGroup.getUid().getOrElse("?"), e);
                stats.addIgnoredEvents(veventGroup.getEvents().size() - (int)(stats.getProcessedCount() - processedCount));
                return allAffectedEventIds;
            }
            throw e;
        }

        log.info("Done processing {}", veventGroup.getUid().getOrElse("?"));
        return allAffectedEventIds;
    }

    public void importEvents(PassportUid uid, IcsCalendar iCal, IcsImportMode icsImportMode, IcsImportStats stats, Optional<String> url) {
        final ListF<IcsVEvent> veComponentList = iCal.getEvents();
        stats.setTotalEventsCount(veComponentList.size());

        final Optional<Long> layerId = List.of(IcsMethod.PUBLISH, IcsMethod.REQUEST).contains(iCal.getMethod())
                ? identityLayerId(icsImportMode, iCal, uid).toOptional()
                : Optional.empty();

        val affectedEventIds = StreamEx.of(iCal.getEventsGroupedByUid())
                .flatMap(vEventGroup ->
                        importEventGroupsWithTransactionAndLog(uid, vEventGroup, iCal.getMethod(), iCal.getTimezones(),
                                icsImportMode.withActionInfoNow(Instant.now()), layerId, stats, url).stream())
                .distinct()
                .collect(CollectorsF.toList());

        ListF<Long> affectedLayerIds = StreamEx.of(eventLayerDao.findEventLayersByEventIds(affectedEventIds).map(EventLayer::getLayerId))
                .distinct().collect(CollectorsF.toList());

        if (icsImportMode.getActionSource() == ActionSource.WEB_ICS) {
            lastUpdateManager.updateLayerTimestamps(affectedLayerIds, icsImportMode.getActionInfo());
        }
        xivaNotificationManager.notifyLayersUsersAboutEventsChangeAsynchronously(
                affectedLayerIds, icsImportMode.getActionInfo());
    }


    private Option<Long> identityLayerId(IcsImportMode icsImportMode, IcsCalendar calendar, PassportUid uid) {
        final IcsLayerImportInfo layerImportInfo = icsImportMode.getLayerImportInfo();
        final LayerReference layerReference = layerImportInfo.layerReference;
        switch (layerImportInfo.layerType) {
        case FEED:
            Validate.isFalse(layerReference.isCreate(), "feed layer should be created by IcsFeedManager");
            return Option.of(layerReference.getId());

        case USER:
            if (!layerImportInfo.getLayerReference().isByCategory()) {
                if (layerReference.isCreate()) {
                    Option<String> layerName = calendar.getXWrCalname().orElse(layerImportInfo.getNewLayerNameO());
                    if (layerReference.isGetCreate()) {
                        return Option.of(layerRoutines.getOrCreateFirstUserLayer(uid, layerName));
                    } else if (layerReference.isCreateNew()) {
                        return Option.of(layerRoutines.createUserLayer(uid, layerName));
                    }
                }
                else if (layerReference.isDefaultLayer()) {
                    return Option.of(layerRoutines.getOrCreateDefaultLayer(uid));
                } else if (layerReference.isId()) {
                    return Option.of(layerReference.getId());
                } else if (layerReference.isDefaultLayerIfCreate()) {
                    return Option.empty();
                } else {
                    throw new IllegalStateException("unknown layer reference: " + layerReference);
                }
            } else {
                return Option.empty();
            }

        default:
            String msg =
                "Layer type: " + layerImportInfo.layerType + ". " +
                "Ics could only be imported to layer of type 'user' or 'feed'.";
            throw new CommandRunException(msg);
        }
    }

} //~
