package ru.yandex.calendar.logic.ics.imp;

import java.util.Optional;

import lombok.val;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.LastUpdateManager;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventInvitation;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.ActorId;
import ru.yandex.calendar.logic.event.EventAttachedLayerId;
import ru.yandex.calendar.logic.event.EventAttachedUser;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventInstanceStatusChecker;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventInvitationRoutines;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventUserRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.EventsAttachedLayerIds;
import ru.yandex.calendar.logic.event.EventsOnLayerChangeHandler;
import ru.yandex.calendar.logic.event.IcsEventSynchData;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.ics.EventInstanceStatusInfo;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVEventGroup;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVEvent;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVTimeZone;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsAttendee;
import ru.yandex.calendar.logic.log.EventIdLogDataJson;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.log.change.EventChangeLogEvents;
import ru.yandex.calendar.logic.log.change.EventChangesJson;
import ru.yandex.calendar.logic.log.change.changes.EventFieldsChangesJson;
import ru.yandex.calendar.logic.log.change.changes.UserRelatedChangesJson;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.ReplyInfo;
import ru.yandex.calendar.logic.sharing.participant.EventParticipants;
import ru.yandex.calendar.logic.sharing.participant.ExternalUserParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.UserParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.YandexUserParticipantInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class IcsEventReplyHandler {
    private static final Logger logger = LoggerFactory.getLogger(IcsEventReplyHandler.class);

    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private EventInstanceStatusChecker eventInstanceStatusChecker;
    @Autowired
    private LastUpdateManager lastUpdateManager;
    @Autowired
    private UserManager userManager;
    @Autowired
    private EventUserRoutines eventUserRoutines;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private EventsOnLayerChangeHandler eventsOnLayerChangeHandler;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private EventInvitationRoutines eventInvitationRoutines;
    @Autowired
    private EventsLogger eventsLogger;
    @Autowired
    private IcsEventImporter icsEventImporter;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;


    public ListF<Long> handleReplyOnInternalEvent(
            IcsVEventGroup veventGroup, PassportUid uid, ListF<IcsVTimeZone> tzs, ActionInfo actionInfo)
    {
        ListF<Long> allModifiedEventIds = Cf.arrayList();
        for (IcsVEvent vevent : veventGroup.getEvents()) {
            ListF<IcsAttendee> pl = vevent.getAttendees();
            Validate.isTrue(
                    pl.size() == 1, "Incorrect attendees count: " + pl.size() + ", must be 1");
            IcsAttendee attendee = pl.single();

            ListF<Long> modifiedEventIds = handleAttendeeDecision(uid, vevent, attendee, tzs, actionInfo);
            allModifiedEventIds.addAll(modifiedEventIds);
        }
        return allModifiedEventIds;
    }

    public ListF<Long> handleAttendeeDecision(
            PassportUid uid, IcsVEvent vevent, IcsAttendee attendee, ListF<IcsVTimeZone> tzs, ActionInfo actionInfo)
    {
        ListF<Long> affectedEventIds = Cf.arrayList();
        IcsVTimeZones timezones = IcsVTimeZones.cons(tzs, dateTimeManager.getTimeZoneForUid(uid), false);
        IcsEventSynchData synchData = IcsUtils.createIcsSynchData(vevent, timezones);

        String reason = vevent.getComment().filter(StringUtils::isNotBlank).getOrElse("");
        ReplyInfo replyInfo = new ReplyInfo(IcsUtils.getDecision(attendee), reason, synchData.getSequenceAndDtstamp());

        Option<Event> eventO = eventRoutines.findEventBySubjectIdAndExternalIdAndRecurrenceId(
                UidOrResourceId.user(uid), synchData.externalId.get(), synchData.recurrenceId);

        if (eventO.exists(e -> e.getRepetitionId().isPresent())
                && vevent.getRRules().isEmpty() && !vevent.getRecurrenceId().isPresent())
        {
            Option<Instant> recurrenceId = eventDbManager.getEventAndRepetitionByEvent(eventO.get())
                    .getRepetitionInfo().findRecurrenceIdByStart(vevent.getStart(timezones));

            if (recurrenceId.isPresent()) {
                eventO = eventRoutines.findEventBySubjectIdAndExternalIdAndRecurrenceId(
                        UidOrResourceId.user(uid), synchData.externalId.get(), recurrenceId);
            } else {
                logger.debug("Ignore single event looking like recurrence that does not match");
                eventO = Option.empty();
            }
        }

        Option<Event> masterEventO = eventO;
        if (synchData.recurrenceId.isPresent()) {
            masterEventO = eventRoutines.findMasterEventBySubjectIdAndExternalId(
                    UidOrResourceId.user(uid), synchData.externalId.get());
        }

        Option<Tuple2<Event, UserRelatedChangesJson>> changes = Option.empty();

        if (eventO.isPresent()) {
            long eventId = eventO.get().getId();

            changes = handleReply(uid, eventId, attendee, replyInfo, actionInfo)
                    .map(Tuple2.join(Cf2.f0(eventO::get).asFunction(), c -> c));

            lastUpdateManager.updateTimestampsAsync(eventO.get().getMainEventId(), actionInfo);

        } else if (masterEventO.isPresent() && synchData.recurrenceId.isPresent()) {
            long eventId = eventRoutines.createNotChangedRecurrence(
                    uid, masterEventO.get().getId(),
                    synchData.recurrenceId.get(), actionInfo);

            Event event = eventDbManager.getEventById(eventId);

            changes = handleReply(uid, eventId, attendee, replyInfo, actionInfo)
                    .map(c -> Tuple2.tuple(event, c));

        } else {
            logger.debug("Event couldn't be found by external and recurrence ids");
        }

        EventsAttachedLayerIds layerIds = new EventsAttachedLayerIds(Tuple2List.tuple2List(
                changes.filterMap(c -> Option.ofNullable(c.get2().layerChanges.map(l -> Tuple2.tuple(c.get1(), l)).orElse(null)))));

        eventInvitationManager.sendEventMails(
                eventsOnLayerChangeHandler.handleEventsAttach(uid, layerIds, actionInfo), actionInfo);

        changes.forEach(c -> {
            Long eventId = c.get1().getId();
            Event updatedEvent = eventDbManager.getEventById(eventId);
            affectedEventIds.add(eventId);

            eventsLogger.log(EventChangeLogEvents.updated(
                    ActorId.user(uid), new EventIdLogDataJson(vevent.getUid().get(), c.get1()),
                    EventChangesJson.empty()
                            .withEvent(EventFieldsChangesJson.of(Optional.of(c.get1()), updatedEvent))
                            .withUsersAndLayers(c.get2())), actionInfo);
        });
        return affectedEventIds;
    }

    public ListF<Long> removeAttendeesMeeting(
            IcsVEventGroup veventGroup, ListF<IcsVTimeZone> tzs, IcsImportMode icsImportMode, PassportUid uid)
    {
        UidOrResourceId subjectId = UidOrResourceId.user(uid);
        ActionSource actionSource = icsImportMode.getActionSource();

        ListF<Long> affectedEventIds = Cf.arrayList();
        for (IcsVEvent vevent : veventGroup.getEvents()) {
            IcsEventSynchData synchData = IcsUtils.createIcsSynchData(
                    vevent, IcsVTimeZones.cons(tzs, dateTimeManager.getTimeZoneForUid(uid), false));

            Option<EventWithRelations> eventO = synchData.externalId
                    .filterMap(extId -> eventRoutines.findEvent(Cf.list(subjectId), extId, synchData.recurrenceId))
                    .map(eventDbManager::getEventWithRelationsByEvent);

            EventInstanceStatusInfo instanceStatusInfo = eventO.map(e ->
                    eventInstanceStatusChecker.getStatusBySubjectAndSyncData(subjectId, synchData, e, actionSource)
            ).orElseGet(EventInstanceStatusInfo::notFound);

            if (instanceStatusInfo.isNeedToUpdate()) {
                long eventId = instanceStatusInfo.getEventId();

                EventWithRelations event = eventO.get();
                EventParticipants participants = event.getEventParticipants();

                if (icsEventImporter.organizerMismatches(uid, Option.of(event), Cf.list(), actionSource)) {
                    logger.info("Ignoring organizer mismatched event cancellation");
                    continue;
                }

                lastUpdateManager.updateTimestampsAsync(
                        eventDao.findMainEventIdByEventId(instanceStatusInfo.getEventId()), icsImportMode.getActionInfo());

                ListF<Email> attendeeEmails = vevent.getAttendeeEmailsSafe();
                ListF<ParticipantId> attendeeIds =
                        eventInvitationManager.getParticipantIdsByEmails(attendeeEmails).get2();

                attendeeIds = attendeeIds.plus(attendeeEmails.map(ParticipantId::invitationIdForExternalUser));

                attendeeIds = attendeeIds.filter(id -> participants.getParticipants().getByIdSafe(id).isPresent()
                        || id.getUidIfYandexUser().exists(u -> participants.getEventUser(u).isPresent()));

                if (icsImportMode.getMailRecipientIds().isPresent()) {
                    attendeeIds = attendeeIds.filter(icsImportMode.getMailRecipientIds().get().containsF());
                }
                ActionInfo actionInfo = icsImportMode.getActionInfo();

                ListF<EventAttachedLayerId> layerIds = Cf.arrayList();

                for (ParticipantId participantId : attendeeIds) {
                    layerIds.addAll(eventInvitationManager.removeAttendeeButNotOrganizerByParticipantId(
                            instanceStatusInfo.getEventId(), participantId, actionInfo));
                }
                eventInvitationManager.updateEventUserSequenceAndDtstamp(
                        uid, instanceStatusInfo.getEventId(), synchData.getSequenceAndDtstamp(), actionInfo);

                if (attendeeIds.isNotEmpty()) {
                    EventParticipants cur = eventDbManager.getParticipantsByEventIds(Cf.list(eventId)).single();

                    Event eventById = eventDao.findEventById(eventId);
                    val logId = new EventIdLogDataJson(vevent.getUid().get(), eventById);
                    affectedEventIds.add(eventId);

                    eventsLogger.log(EventChangeLogEvents.updated(
                            ActorId.user(uid), logId, participants, cur, layerIds), actionInfo);
                }

            } else {
                logger.info("Ignoring event " + veventGroup.getUid() + " status: " + instanceStatusInfo);
            }
        }
        return affectedEventIds;
    }

    private Option<UserRelatedChangesJson> handleReply(PassportUid receiverUid,
                                                       long eventId, IcsAttendee attendee, ReplyInfo replyInfo, ActionInfo actionInfo)
    {
        ParticipantId participantId = eventInvitationManager
                .getParticipantIdByEmail(attendee.getEmail());
        Option<ParticipantInfo> participantO = eventInvitationManager
                .getParticipantByEventIdAndParticipantId(eventId, participantId);

        if (participantO.isPresent() && participantId.isAnyUser()) {
            UserParticipantInfo participant = (UserParticipantInfo) participantO.get();
            Availability avail = Availability.byDecision(replyInfo.getDecision());

            Option<EventAttachedLayerId> layerId = Option.empty();

            if (participant.getId().isYandexUser()) {
                eventUserRoutines.updateEventUserAvailability(participant.getUid().get(), eventId, avail, actionInfo);

                EventAndRepetition event = eventDbManager.getEventAndRepetitionByIdForUpdate(participant.getEventId());

                layerId = Option.of(eventDbManager.saveEventLayerIfAbsentForUser(
                        participant.getUid().get(), event, actionInfo));
            }
            // XXX: check per-user sequence and dtstamp
            EventInstanceStatusInfo status = EventInstanceStatusInfo.needToUpdate(eventId);
            eventInvitationManager.updateUserParticipantDecision(participant, replyInfo, actionInfo, status);

            if (participant.getId().isYandexUser()) {
                YandexUserParticipantInfo p = (YandexUserParticipantInfo) participant;

                return Option.of(UserRelatedChangesJson.find(
                        p.getEmail(), Optional.of(p.getEventUser()),
                        eventUserRoutines.findEventUser(eventId, p.getUidSome()).toOptional(), layerId.toOptional()));
            } else {
                ExternalUserParticipantInfo p = (ExternalUserParticipantInfo) participant;

                return Option.of(UserRelatedChangesJson.find(Optional.of(p.getInvitation()),
                        eventInvitationRoutines.findEventInvitation(eventId, p.getEmail()).toOptional()));
            }

        } else if (participantId.isYandexUser()) { // CAL-5683
            EventAttachedUser changes = eventRoutines.attachEventOrMeetingToUser(
                    userManager.getUserInfo(participantId.getUid()), eventId, Option.<Long>empty(),
                    replyInfo.getDecision(), toEventUserDataForNewAttendee(replyInfo),
                    NotificationsData.useLayerDefaultIfCreate(), actionInfo);

            eventDao.updateEventIncrementSequenceById(eventId); // CAL-6760

            return Option.of(UserRelatedChangesJson.find(attendee.getEmail(), changes));

        } else if (participantId.isExternalUser()) { // CAL-5683
            EventInvitation invitation = eventInvitationManager.createExternalUserInvitation(
                    UidOrResourceId.user(receiverUid), eventId,
                    IcsEventDataConverter.attendeeToParticipantData(attendee).get(), actionInfo);

            eventDao.updateEventIncrementSequenceById(eventId); // CAL-6760

            return Option.of(UserRelatedChangesJson.find(Optional.empty(), Optional.of(invitation)));

        } else {
            logger.debug("Ignoring reply from resource");
            return Option.empty();
        }
    }

    private EventUser toEventUserDataForNewAttendee(ReplyInfo reply) {
        EventUser data = new EventUser();
        data.setIsAttendee(true);
        data.setDecision(reply.getDecision());
        data.setReason(reply.getReason());

        data.setAvailability(Availability.byDecision(reply.getDecision()));
        data.setSequence(reply.getSequence());
        data.setDtstamp(reply.getDtstamp());

        return data;
    }
}
