package ru.yandex.calendar.logic.ics.imp;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.layer.LayerType;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.reqid.RequestIdStack;

/**
 * @author Stepan Koltsov
 */
public class IcsImportMode {
    private final Option<Instant> notModifiedSince;
    private final Option<Decision> decision;
    private final Option<ListF<Notification>> notifications;
    private final Option<PassportUid> mailFromUid;
    private final Option<ListF<ParticipantId>> mailRecipientIds;
    private final ActionInfo actionInfo;
    private final IcsLayerImportInfo layerImportInfo;

    private IcsImportMode(ActionInfo actionInfo, IcsLayerImportInfo layerInfo) {
        this(Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.empty(), actionInfo, layerInfo);
    }

    private IcsImportMode(
            Option<Instant> notModifiedSince, Option<Decision> decision, Option<ListF<Notification>> notifications,
            Option<PassportUid> mailFromUid, Option<ListF<ParticipantId>> mailRecipientIds,
            ActionInfo actionInfo, IcsLayerImportInfo layerImportInfo)
    {
        this.notModifiedSince = notModifiedSince;
        this.decision = decision;
        this.notifications = notifications;
        this.mailFromUid = mailFromUid;
        this.mailRecipientIds = mailRecipientIds;
        this.actionInfo = actionInfo;
        this.layerImportInfo = layerImportInfo;
    }

    public Option<Instant> getNotModifiedSince() {
        return notModifiedSince;
    }

    public Option<Decision> getDecision() {
        return decision;
    }

    public Option<ListF<Notification>> getNotifications() {
        return notifications;
    }

    public Option<PassportUid> getMailFromUid() {
        return mailFromUid;
    }

    public Option<ListF<ParticipantId>> getMailRecipientIds() {
        return mailRecipientIds;
    }

    public ActionSource getActionSource() {
        return actionInfo.getActionSource();
    }

    public ActionInfo getActionInfo() {
        return actionInfo;
    }

    public IcsLayerImportInfo getLayerImportInfo() {
        return layerImportInfo;
    }

    private static ActionInfo caldavNow() {
        return actionInfoNow(ActionSource.CALDAV);
    }

    private static ActionInfo actionInfoNow(ActionSource actionSource) {
        return actionInfo(actionSource, new Instant());
    }

    private static ActionInfo actionInfo(ActionSource actionSource, Instant now) {
        return new ActionInfo(actionSource, RequestIdStack.current().getOrElse("?"), now);
    }

    private static ActionInfo actionInfoTest(ActionSource actionSource, Instant now) {
        return ActionInfo.test(actionSource, RequestIdStack.current().getOrElse("?"), now);
    }

    public static IcsImportMode caldavPut(long layerId, Option<Instant> notModifiedSince) {
        return new IcsImportMode(
                notModifiedSince, Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                caldavNow(), new IcsLayerImportInfo(LayerReference.id(layerId), LayerType.USER));
    }

    public static IcsImportMode caldavPut(long layerId, Instant now) {
        return new IcsImportMode(actionInfo(ActionSource.CALDAV, now), new IcsLayerImportInfo(LayerReference.id(layerId), LayerType.USER));
    }

    public static IcsImportMode caldavPutToDefaultLayerForTest() {
        return new IcsImportMode(actionInfoTest(ActionSource.CALDAV,
                Instant.now()), new IcsLayerImportInfo(LayerReference.defaultLayer(), LayerType.USER));
    }

    public static IcsImportMode caldavPutToDefaultLayerForTest(Instant now) {
        return new IcsImportMode(actionInfoTest(ActionSource.CALDAV, now), new IcsLayerImportInfo(LayerReference.defaultLayer(), LayerType.USER));
    }

    private static ActionInfo webIcsNow() {
        return actionInfoNow(ActionSource.WEB_ICS);
    }

    public static IcsImportMode updateFeed(long layerId) {
        return new IcsImportMode(webIcsNow(), new IcsLayerImportInfo(LayerReference.id(layerId), LayerType.FEED));
    }

    public static IcsImportMode importFile(LayerReference layerReference) {
        return new IcsImportMode(webIcsNow(), new IcsLayerImportInfo(layerReference, LayerType.USER));
    }

    public static IcsImportMode importFile(LayerReference layerReference, Instant now) {
        return new IcsImportMode(actionInfo(ActionSource.WEB_ICS, now), new IcsLayerImportInfo(layerReference, LayerType.USER));
    }

    public static IcsImportMode importFile(LayerReference layerReference, String newLayerName) {
        return new IcsImportMode(webIcsNow(), new IcsLayerImportInfo(layerReference, LayerType.USER, Option.of(newLayerName)));
    }

    public static IcsImportMode importFileForTest(LayerReference layerReference, Instant now) {
        return new IcsImportMode(actionInfoTest(ActionSource.WEB_ICS, now), new IcsLayerImportInfo(layerReference, LayerType.USER));
    }

    public static IcsImportMode incomingEmailFromMailhook() {
        return incomingEmailFromMailhook(Option.empty(), Cf.list(), Instant.now());
    }

    public static IcsImportMode incomingEmailFromMailhook(
            Option<PassportUid> fromUid, ListF<ParticipantId> recipientIds, Instant now)
    {
        return new IcsImportMode(
                Option.empty(), Option.empty(), Option.empty(),
                fromUid, Option.of(recipientIds),
                actionInfo(ActionSource.MAILHOOK, now), IcsLayerImportInfo.DEFAULT_USER_IF_CREATE);
    }

    public static IcsImportMode incomingEmailFromMailhook(Instant now) {
        return new IcsImportMode(actionInfo(ActionSource.MAILHOOK, now), IcsLayerImportInfo.DEFAULT_USER_IF_CREATE);
    }

    public static IcsImportMode mailWidget(PassportUid recipient, Decision decision, ActionInfo actionInfo) {
        return new IcsImportMode(Option.empty(), Option.of(decision), Option.empty(), Option.empty(),
                Option.of(Cf.list(ParticipantId.yandexUid(recipient))),
                actionInfo, IcsLayerImportInfo.DEFAULT_USER_IF_CREATE);
    }

    public static IcsImportMode mailWidget(PassportUid recipient, ActionInfo actionInfo) {
        return new IcsImportMode(Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                Option.of(Cf.list(ParticipantId.yandexUid(recipient))),
                actionInfo, IcsLayerImportInfo.DEFAULT_USER_IF_CREATE);
    }

    public IcsImportMode withActionInfoNow(Instant now) {
        return new IcsImportMode(
                notModifiedSince, decision, notifications,
                mailFromUid, mailRecipientIds, actionInfo.withNow(now), layerImportInfo);
    }

    public IcsImportMode withActionInfoFreezedNowForTest() {
        return new IcsImportMode(
                notModifiedSince, decision, notifications,
                mailFromUid, mailRecipientIds, actionInfo.freezeNowForTest(), layerImportInfo);
    }

} //~
