package ru.yandex.calendar.logic.ics.imp;

import java.text.MessageFormat;

import lombok.Getter;
import org.jdom.Element;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.calendar.util.xml.CalendarXmlizer;

/**
 * Holds results of the ics events import.
 * XXX sum of additives != totalEventsCount. This can be reproduced with
 * {@link IcsImporterAttendeeUpdatesMeetingTest#attendeeChangesDecisionToYesInExistingEvent()} test.
 */
public class IcsImportStats {
    // total count of events in ics
    private int totalEventCount = 0;

    // Id set of new created events
    private SetF<Long> newEventIds = Cf.hashSet();
    // Id set of updated events
    private SetF<Long> updatedEventIds = Cf.hashSet();
    // Id set of attached events
    private SetF<Long> attachedEventIds = Cf.hashSet();
    private SetF<Long> deletedEventIds = Cf.hashSet();
    // Id set of events, replies on which invitation were handled
    private SetF<Long> repliedEventIds = Cf.hashSet();
    // Id set of ignored events
    // Such events was properly parsed, but version in the store is
    // newer than in ics
    private SetF<Long> ignoredEventIds = Cf.hashSet();
    // Count of ignored events from ics
    @Getter private long ignoredEventCount = 0;
    private IcsTodoImportStats todoImportStats;

    protected IcsImportStats() {
    }

    public void setTotalEventsCount(int count) { this.totalEventCount = count; }

    public void addNewEventId(long id) { newEventIds.add(id); }
    public void addUpdatedEventId(long id) { updatedEventIds.add(id); }

    public void addIgnoredEventId(long id) {
        ignoredEventIds.add(id);
        ignoredEventCount++;
    }

    public void addIgnoredEvent() {
        ignoredEventCount++;
    }

    public void addIgnoredEvents(int count) {
        ignoredEventCount += count;
    }

    public void addDeletedEventId(long id) {
        deletedEventIds.add(id);
    }

    public IcsTodoImportStats getTodoImportStats() {
        return todoImportStats;
    }

    public SetF<Long> getDeletedEventIds() {
        return deletedEventIds;
    }

    public void setTodoImportStats(IcsTodoImportStats todoImportStats) {
        this.todoImportStats = todoImportStats;
    }

    public boolean wasChange() {
        return !(newEventIds.isEmpty() && updatedEventIds.isEmpty() && attachedEventIds.isEmpty() && repliedEventIds.isEmpty());
    }

    public SetF<Long> getProcessedEventIds() {
        SetF<Long> res = Cf.<Long>hashSet();
        res.addAll(newEventIds);
        res.addAll(updatedEventIds);
        res.addAll(attachedEventIds);
        res.addAll(ignoredEventIds);
        res.addAll(repliedEventIds);
        return res;
    }

    public Element createEventStatElem() {
        Element res = new Element("events");
        // TODO: ssytnik: out all fields separately here for makeup to show
        CalendarXmlizer.setAttr(res, "processed", getProcessedCount());
        CalendarXmlizer.setAttr(res, "total", getTotalCount());
        return res;
    }

    public long getProcessedCount() {
        return newEventIds.size() + updatedEventIds.size() + attachedEventIds.size() +
        repliedEventIds.size() + ignoredEventCount;
    }

    public int getTotalCount() {
        return totalEventCount;
    }

    public String getLogMsg() {
        return MessageFormat.format(
            "Ics import statistic - " +
            "events: " +
                "added: {0,number}, " +
                "updated: {1,number}, " +
                "attached: {2,number}, " +
                "replies handled: {3, number}, " +
                "ignored: {4,number}, " +
                "total: {5,number}; " +
            "todos: " +
                "added: {6,number}, " +
                "updated: {7,number}, " +
                "ignored: {8,number}",
            newEventIds.size(), updatedEventIds.size(), attachedEventIds.size(),
            repliedEventIds.size(), ignoredEventCount, totalEventCount,
            todoImportStats.getNewTodoItemCount(),
            todoImportStats.getUpdatedTodoItemCount(),
            todoImportStats.getIgnoredTodoItemCount()
        );
    }

    public SetF<Long> getNewEventIds() {
        return newEventIds;
    }

    public SetF<Long> getUpdatedEventIds() {
        return updatedEventIds;
    }
}
