package ru.yandex.calendar.logic.ics.imp;

import java.util.Optional;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsMethod;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.file.File2;

/**
 * This class is used for import data from ics.
 */
@Slf4j
public class IcsImporter {
    @Autowired
    private IcsTodoImporter icsTodoImporter;
    @Autowired
    private IcsEventImporter icsEventImporter;

    public IcsImportStats importIcsStuff(PassportUid uid, IcsCalendar iCal, IcsImportMode icsImportMode) {
        return importIcsStuff(uid, iCal, icsImportMode, Optional.empty());
    }

    public IcsImportStats importIcsStuff(PassportUid uid, IcsCalendar iCal, IcsImportMode icsImportMode, Optional<String> url) {
        val stats = new IcsImportStats();
        log.debug("Start ics import, {}", LogMarker.ICS_METHOD.format(iCal.getMethod().getValue()));

        if (icsImportMode.getActionSource() != ActionSource.WEB_ICS) {
            log.debug("\n {}", iCal.serializeToString());
        }

        if (!Cf.list(IcsMethod.PUBLISH, IcsMethod.REQUEST, IcsMethod.CANCEL, IcsMethod.REPLY).containsTs(iCal.getMethod())) {
            val msg = "Processing of iCalendar method - " + iCal.getMethod().getValue() + " is not implemented";
            throw new IllegalArgumentException(msg);
        }

        icsEventImporter.importEvents(uid, iCal, icsImportMode, stats, url);
        stats.setTodoImportStats(icsTodoImporter.importTodos(uid, iCal, icsImportMode.getActionInfo()));

        log.debug(stats.getLogMsg());
        return stats;

    }

    public void importIcsFromMailhook(
            PassportUid uid, Option<PassportUid> fromUid,
            ListF<ParticipantId> recipientIds, IcsCalendar iCal)
    {
        if (iCal.getMethod().sameMethodAs(IcsMethod.REQUEST)
                || iCal.getMethod().sameMethodAs(IcsMethod.REPLY)
                || iCal.getMethod().sameMethodAs(IcsMethod.CANCEL))
        {
            importIcsStuff(uid, iCal, IcsImportMode.incomingEmailFromMailhook(fromUid, recipientIds, Instant.now()));
        } else {
            log.info("Ics from mailhook is ignored because method is {}", iCal.getMethod());
        }
    }

    public void importIcsFromFsLikeWebIcs(PassportUid uid, File2 file) {
        importIcsStuff(uid, IcsCalendar.parse(file), IcsImportMode.importFile(LayerReference.defaultLayer()));
    }
}
