package ru.yandex.calendar.logic.ics.imp;

import net.fortuna.ical4j.model.Property;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.beans.generated.TodoItemFields;
import ru.yandex.calendar.logic.beans.generated.TodoList;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.calendar.logic.ics.exp.IcsTodoExporter;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.PropertyNames;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVAlarm;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVToDo;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsDtStart;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsDue;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsProperty;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.dateTime.IcsDateTimeFormats;
import ru.yandex.calendar.logic.ics.iv5j.support.IvParser;
import ru.yandex.calendar.logic.todo.TodoDao;
import ru.yandex.calendar.logic.todo.TodoRoutines;
import ru.yandex.calendar.logic.todo.TodoStatus;
import ru.yandex.calendar.logic.todo.id.ListIdOrExternalId;
import ru.yandex.calendar.util.color.ColorUtils;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Stepan Koltsov
 * @author akirakozov
 * @author ssytnik
 */
public class IcsTodoImporter {

    private static final Logger logger = LoggerFactory.getLogger(IcsTodoImporter.class);

    @Autowired
    private TodoRoutines todoRoutines;
    @Autowired
    private TodoDao todoDao;
    @Autowired
    private DateTimeManager dateTimeManager;

    private enum TodoItemStatus {
        NOT_FOUND,
        UPDATED,
        NEED_TO_UPDATE,
    };

    private Tuple2<TodoItemStatus, Option<Long>> getTodoItemStatus(
            PassportUid uid, IcsVToDo icsTodo, IcsVTimeZones tzs, ActionInfo actionInfo)
    {
        if (icsTodo.getUid().isPresent()) {
            Option<TodoItem> todoItemO = todoDao.findNotDeletedNotArchivedTodoItemByExternalIdAndCreatorUid(
                    icsTodo.getUid().get(), uid);
            if (todoItemO.isPresent()) {
                TodoItem todoItem = todoItemO.get();
                Instant dtstamp = icsTodo.getDtStampInstant(tzs).getOrElse(new Instant(0));

                TodoItemStatus status = TodoItemStatus.UPDATED;
                if (actionInfo.getActionSource() == ActionSource.CALDAV // CAL-5915
                    || todoItem.getLastUpdateTs().isBefore(dtstamp))
                {
                    status = TodoItemStatus.NEED_TO_UPDATE;
                }
                return Tuple2.tuple(status, Option.of(todoItem.getId()));
            }
        }
        return Tuple2.tuple(TodoItemStatus.NOT_FOUND, Option.<Long>empty());
    }

    public IcsTodoImportStats importTodos(final PassportUid uid, IcsCalendar iCal, ActionInfo actionInfo) {
        if (iCal.getTodos().isNotEmpty()) {
            return importTodos(uid, iCal, actionInfo, prepareListForImport(uid, iCal, actionInfo));
        }
        return new IcsTodoImportStats(0, 0, 0);
    }

    public IcsTodoImportStats importTodos(PassportUid uid, IcsCalendar iCal, ActionInfo actionInfo, long todoListId) {
        int newTodoItemCount = 0;
        int updatedTodoItemCount = 0;
        int ignoredTodoItemCount = 0;

        IcsVTimeZones tzs = IcsVTimeZones.cons(iCal.getTimezones(), dateTimeManager.getTimeZoneForUid(uid), false);

        // Do todo import
        for (IcsVToDo icsTodo : iCal.getTodos()) {
            Tuple2<TodoItemStatus, Option<Long>> statusRes = getTodoItemStatus(uid, icsTodo, tzs, actionInfo);
            TodoItemStatus status = statusRes.get1();
            Option<Long> todoItemIdO = statusRes.get2();

            if (TodoItemStatus.NOT_FOUND == status) {
                TodoItem todoItemData = convertTodoItemData(uid, icsTodo, tzs);
                todoItemData.setTodoListId(todoListId);
                todoRoutines.createTodoItem(uid, todoItemData, actionInfo);
                ++newTodoItemCount;
            } else if (TodoItemStatus.NEED_TO_UPDATE == status) {
                TodoItem todoItemData = convertTodoItemData(uid, icsTodo, tzs);
                todoItemData.unsetField(TodoItemFields.EXTERNAL_ID);
                todoItemData.setTodoListId(todoListId);
                todoRoutines.updateTodoItem(uid, todoItemData, todoItemIdO.get(), actionInfo);
                ++updatedTodoItemCount;
            } else {
                ++ignoredTodoItemCount;
            }
        }

        return new IcsTodoImportStats(newTodoItemCount, updatedTodoItemCount, ignoredTodoItemCount);
    }

    private TodoItem convertTodoItemData(final PassportUid uid, IcsVToDo icsTodo, IcsVTimeZones tzs) {
        TodoItem todoItemData = new TodoItem();
        if (icsTodo.getUid().isPresent()) {
            todoItemData.setExternalId(icsTodo.getUid().get());
        }
        String summary = icsTodo.getSummary().getOrElse("");
        if (TodoItemFields.TITLE.getCut().isPresent()) {
            summary = StringUtils.take(summary, TodoItemFields.TITLE.getCut().get());
        }
        todoItemData.setTitle(summary);
        String description = icsTodo.getDescription().getOrElse("");
        if (TodoItemFields.DESCRIPTION.getCut().isPresent()) {
            description = StringUtils.take(description, TodoItemFields.DESCRIPTION.getCut().get());
        }
        todoItemData.setDescription(description);
        todoItemData.setLocation(icsTodo.getPropertyValue(Property.LOCATION).getOrElse(""));

        todoItemData.setStartTs(icsTodo.getDtStart().map(IcsDtStart.getInstantF(tzs)));
        todoItemData.setCompletionTs(icsTodo.getCompletedInstant(tzs));
        todoItemData.setDueTs(icsTodo.getDueInstant(tzs));

        todoItemData.setIsAllDay(
                icsTodo.getDtStart().exists(IcsDtStart.isDateF()) || icsTodo.getDue().exists(IcsDue.isDateF()));

        todoItemData.setNotificationTs(findFirstNotAcknowledgedNotGeoNotificationInstant(icsTodo));
        todoItemData.setIcalGeoNotification(getGeoNotification(icsTodo));
        todoItemData.setStatus(TodoStatus.fromValueSafe(icsTodo.getPropertyValue(Property.STATUS).getOrElse("")));

        Function<String, Option<Integer>> parseIntF = Cf.Integer.parseSafeF();
        todoItemData.setSequence(icsTodo.getPropertyValue(Property.SEQUENCE).filterMap(parseIntF).getOrElse(0));
        todoItemData.setCompletePercent(icsTodo.getPropertyValue(Property.PERCENT_COMPLETE).filterMap(parseIntF));
        todoItemData.setPriority(icsTodo.getPropertyValue(Property.PRIORITY).filterMap(parseIntF));

        // lightning use these to track alarm dismissal and snooze
        // http://hg.mozilla.org/releases/comm-1.9.2/file/2d9fb6c132aa/calendar/base/src/calAlarm.js
        // http://hg.mozilla.org/releases/comm-1.9.2/file/2d9fb6c132aa/calendar/base/src/calAlarmService.js
        Option<Instant> lastack = icsTodo.getProperty(PropertyNames.X_MOZ_LASTACK)
                .map(IcsProperty.valueF().andThen(IcsUtils.parseTimestampF()));
        Option<Instant> snoozeTime = icsTodo.getProperty(PropertyNames.X_MOZ_SNOOZE_TIME)
                .map(IcsProperty.valueF().andThen(IcsDateTimeFormats.parseDateTimeF()));
        todoItemData.setIcalXMozLastack(lastack);
        todoItemData.setIcalXMozSnoozeTime(snoozeTime);

        Option<Long> order = icsTodo.getPropertyValue(PropertyNames.X_APPLE_SORT_ORDER).filterMap(Cf.Long.parseSafeF());
        todoItemData.setIcalXAppleSortOrder(order);

        // TODO pos
        Option<String> xColor = icsTodo.getPropertyValue("X-COLOR");
        if (xColor.isPresent()) {
            todoItemData.setColor(ColorUtils.unformatColor(xColor.get()));
        }
        return todoItemData;
    }

    private Option<Instant> findFirstNotAcknowledgedNotGeoNotificationInstant(IcsVToDo todo) {
        Option<IcsVAlarm> alarm = todo.getVAlarms().find(
                IcsVAlarm.isAcknowledgedF().notF()
                        .andF(IcsVAlarm.hasRelativeTriggerF().notF())
                        .andF(IcsVAlarm.isGeoF().notF()));

        return alarm.isPresent() ? Option.of(alarm.get().getMainNotificationTimestamp()) : Option.<Instant>empty();
    }

    private Option<String> getGeoNotification(IcsVToDo todo) {
        Option<IcsVAlarm> alarm = todo.getVAlarms().find(IcsVAlarm.isGeoF());

        if (alarm.isPresent()) {
            String serialized = alarm.get().getProperties().map(IcsProperty.serializeF()).mkString("");
            try {
                IvParser.parseIcsProperties(serialized);
                return Option.of(serialized);

            } catch (RuntimeException e) {
                logger.error("Failed to reparse serialized valarm: {}", alarm.get());
            }
        }
        return Option.empty();
    }

    private long prepareListForImport(PassportUid uid, IcsCalendar iCal, ActionInfo actionInfo) {
        Option<String> listNameO = iCal.getXWrCalname()
                .orElse(iCal.getPropertyValue(IcsTodoExporter.X_YANDEX_TODO_LIST_TITLE));

        String listName = listNameO.getOrElse(TodoRoutines.DEFAULT_LIST_NAME);

        final long listId;

        if (actionInfo.getActionSource() != ActionSource.WEB_ICS) {
            listId = todoRoutines.findFirstCreatedListOrCreateNewWithName(uid, listName, actionInfo);

        } else if (iCal.hasProperty(IcsTodoExporter.X_YANDEX_TODO_LIST_UID)) {
            String externalId = iCal.getPropertyValue(IcsTodoExporter.X_YANDEX_TODO_LIST_UID).get();

            if (todoDao.findNotDeletedTodoListExistsById(ListIdOrExternalId.externalId(externalId, uid))) {
                listId = todoRoutines.getTodoListId(ListIdOrExternalId.externalId(externalId, uid));

                todoDao.updateTodoListTitleAndDescription(listId, listName, "", actionInfo);

            } else {
                TodoList todoListData = new TodoList();
                todoListData.setTitle(listName);
                todoListData.setExternalId(externalId);

                listId = todoRoutines.createTodoList(uid, todoListData, actionInfo);
            }
        } else {
            ListF<TodoList> lists = todoRoutines.getNonDeletedTodoLists(uid);
            Option<TodoList> list = lists.find((l) -> listNameO.containsTs(l.getTitle()));

            if (list.isPresent()) {
                listId = list.get().getId();
            } else if (lists.isNotEmpty() && !listNameO.isPresent()) {
                listId = lists.min(TodoList.getCreationTsF().andThenNaturalComparator()).getId();
            } else {
                listId = todoRoutines.createTodoListDefault(uid, listNameO, actionInfo);
            }
        }
        return listId;
    }

} //~
