package ru.yandex.calendar.logic.ics.iv5j.ical;

import java.io.InputStream;
import java.io.Reader;
import java.io.StringReader;
import java.util.List;

import lombok.val;
import net.fortuna.ical4j.model.Calendar;
import net.fortuna.ical4j.model.Property;
import net.fortuna.ical4j.model.component.CalendarComponent;
import net.fortuna.ical4j.model.component.VEvent;
import net.fortuna.ical4j.model.component.VTimeZone;
import net.fortuna.ical4j.model.property.DtEnd;
import one.util.streamex.StreamEx;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.ComponentsMeta;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsComponent;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVEvent;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVTimeZone;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVToDo;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsMethod;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsProperty;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsXProperty;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsXWrCalname;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.PropertiesMeta;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.ReaderSource;
import ru.yandex.misc.lang.CharsetUtils;

public class IcsCalendar extends PropertiesComponentsContainer {

    public IcsCalendar(ListF<? extends IcsComponent> components, ListF<? extends IcsProperty> properties) {
        super(properties, components);
    }

    public IcsCalendar() {
        super();
    }

    public IcsCalendar(ListF<? extends IcsComponent> components) {
        this(components, Cf.list());
    }

    public IcsCalendar addProperty(IcsProperty property) {
        return addProperties(Cf.list(property));
    }

    public IcsCalendar addProperties(ListF<IcsProperty> properties) {
        return new IcsCalendar(components, this.properties.plus(properties));
    }

    public IcsCalendar addComponent(IcsComponent component) {
        return addComponents(Cf.list(component));
    }

    public IcsCalendar addComponents(ListF<? extends IcsComponent> components) {
        return new IcsCalendar(this.components.plus(components), properties);
    }

    public IcsMethod getMethod() {
        return (IcsMethod) getProperties(Property.METHOD).singleO().getOrElse(IcsMethod.PUBLISH);
    }

    public ListF<IcsVEvent> getEvents() {
        ListF<IcsVEvent> events = getComponents(VEvent.VEVENT);
        return events.filter(IcsVEvent.hasPropertyF(PropertyNames.X_MOZ_FAKED_MASTER).notF()); // CAL-5294
    }

    public List<IcsComponent> getComponentsExceptEvents() {
        return StreamEx.of(getComponents())
                .removeBy(IcsComponent::getName, VEvent.VEVENT)
                .toImmutableList();
    }

    public ListF<IcsVToDo> getTodos() {
        return getComponents(VEvent.VTODO);
    }

    public ListF<IcsVTimeZone> getTimezones() {
        return getComponents(VTimeZone.VTIMEZONE);
    }

    static Comparator<IcsVEvent> eventByRecurrenceIdComparator() {
        return (o1, o2) -> {
            if (!o1.isRecurrence() || !o2.isRecurrence()) {
                return Comparator.<Boolean>naturalComparator().compare(o1.isRecurrence(), o2.isRecurrence());
            }
            val recurrence1 = o1.getRecurrenceId().get().getInstant(IcsVTimeZones.fallback(DateTimeZone.UTC));
            val recurrence2 = o2.getRecurrenceId().get().getInstant(IcsVTimeZones.fallback(DateTimeZone.UTC));
            return Comparator.<Instant>naturalComparator().compare(recurrence1, recurrence2);
        };
    }

    /**
     * Events grouped by UID and sorted main event first
     */
    public ListF<IcsVEventGroup> getEventsGroupedByUid() {
        Function1B<IcsVEvent> hasUidF = e -> e.getUid().isPresent();
        Tuple2<ListF<IcsVEvent>, ListF<IcsVEvent>> partitioned = getEvents().partition(hasUidF);
        ListF<IcsVEvent> withUid = partitioned._1;
        ListF<IcsVEvent> withoutUid = partitioned._2;

        ListF<IcsVEventGroup> withUidGrouped = withUid
            .groupBy(IcsVEvent::getUid)
            .mapValues(es -> es.sorted(eventByRecurrenceIdComparator()))
            .mapEntries(IcsVEventGroup.consF());

        return withUidGrouped.plus(withoutUid.map(IcsVEventGroup.cons1F()));
    }

    public ListF<IcsVEventGroup> getEventsGroupedByUidWithExpiredFiltered() {
        Function1B<IcsVEvent> hasUidF = e -> e.getUid().isPresent();
        ListF<IcsVEvent> events = getEvents().filter(ev -> {
            boolean eventIsActual = ev.hasProperty(DtEnd.DTEND) && ev.getEnd().getDateTime(
                    IcsVTimeZones.cons(Cf.list(IcsTimeZones.icsVTimeZoneForIdFull("UTC")), DateTimeZone.forID("UTC"), false)
            ).isAfter(Instant.now().minus(Duration.standardDays(1))) && !ev.isAllDay();
            boolean recurrenceIsActual = !ev.getRRules().isEmpty() || ev.getRecurrenceIdInstantUtc().isPresent() && ev.getRecurrenceIdInstantUtc().get().isAfter(
                    Instant.now().minus(Duration.standardDays(1))
            );
            return eventIsActual || recurrenceIsActual;
        });
        Tuple2<ListF<IcsVEvent>, ListF<IcsVEvent>> partitioned = events.partition(hasUidF);
        ListF<IcsVEvent> withUid = partitioned._1;
        ListF<IcsVEvent> withoutUid = partitioned._2;

        ListF<IcsVEventGroup> withUidGrouped = withUid
                .groupBy(IcsVEvent::getUid)
                .mapValues(es -> es.sorted(eventByRecurrenceIdComparator()))
                .mapEntries(IcsVEventGroup.consF());

        return withUidGrouped.plus(withoutUid.map(IcsVEventGroup.cons1F()));
    }

    public Option<String> getXWrCalname() {
        return getPropertyValue(IcsXWrCalname.X_WR_CALNAME);
    }

    public Option<String> getProdId() {
        return getPropertyValue(Property.PRODID);
    }

    public IcsCalendar withProperties(ListF<IcsProperty> properties) {
        return new IcsCalendar(components, properties);
    }

    public IcsCalendar removeProperties(String name) {
        return filterProperties(IcsProperty.nameF().andThenEquals(name).notF());
    }

    public IcsCalendar filterProperties(Function1B<? super IcsProperty> f) {
        return withProperties(properties.filter(f));
    }

    public IcsCalendar withProperty(IcsProperty property) {
        return this
            .removeProperties(property.getName())
            .addProperty(property)
            ;
    }

    public IcsCalendar withMethod(IcsMethod method) {
        return withProperty(method);
    }

    public IcsCalendar withXYandexMailType(String type) {
        return withProperty(new IcsXProperty(PropertyNames.X_YANDEX_MAIL_TYPE, type));
    }

    public Option<String> getXYandexMailType() {
        return getPropertyValue(PropertyNames.X_YANDEX_MAIL_TYPE);
    }

    public IcsCalendar withXYandexEnvironment(String value) {
        return withProperty(new IcsXProperty(PropertyNames.X_YANDEX_ENVIRONMENT, value));
    }

    public Option<String> getXYandexEnvironment() {
        return getPropertyValue(PropertyNames.X_YANDEX_ENVIRONMENT);
    }

    public Calendar toCalendar() {
        Calendar r = new Calendar();
        for (IcsProperty property : properties) {
            r.getProperties().add(property.toProperty());
        }
        for (IcsComponent component : components) {
            r.getComponents().add((CalendarComponent) component.toComponent());
        }
        return r;
    }

    public String serializeToString() {
        Calendar r = new Calendar();
        for (IcsProperty property : properties) {
            r.getProperties().add(property.toPropertyForSerialization());
        }
        for (IcsComponent component : components) {
            r.getComponents().add((CalendarComponent) component.toComponentForSerialization());
        }
        return r.toString();
    }

    @Override
    public String toString() {
        // XXX: evil
        return toCalendar().toString();
    }

    @SuppressWarnings("unchecked")
    public static IcsCalendar fromIcal4j(Calendar calendar) {
        ListF<IcsComponent> components = Cf.<CalendarComponent>x(calendar.getComponents()).map(ComponentsMeta.M.fromIcal4jF());
        ListF<IcsProperty> properties = Cf.<Property>x(calendar.getProperties()).map(PropertiesMeta.M.fromIcal4jF());
        return new IcsCalendar(components, properties);
    }

    public static IcsCalendar parse(Reader reader) {
        return IcsUtils.parse(reader);
    }

    public static IcsCalendar parse(InputStream is) {
        return IcsUtils.parse(is);
    }

    public static IcsCalendar parse(ReaderSource reader) {
        return IcsUtils.parse(reader);
    }

    public static IcsCalendar parse(InputStreamSource is) {
        return IcsUtils.parse(is);
    }

    public static IcsCalendar parseString(String string) {
        return parse(new StringReader(string));
    }

    public byte[] serializeToBytes() {
        return CharsetUtils.encodeUtf8ToArray(serializeToString());
    }

    public static Function<IcsCalendar, String> serializeToStringF() {
        return IcsCalendar::serializeToString;
    }

    public static Function<IcsCalendar, byte[]> serializeToBytesF() {
        return IcsCalendar::serializeToBytes;
    }

} //~
