package ru.yandex.calendar.logic.ics.iv5j.ical;

import net.fortuna.ical4j.model.Date;
import net.fortuna.ical4j.model.component.Observance;
import net.fortuna.ical4j.model.component.VTimeZone;
import net.fortuna.ical4j.util.Dates;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author dbrylev
 */
public class IcsSuitableTimeZoneFinder {

    private static final LocalDateTime summer;
    private static final LocalDateTime winter;

    static final MapF<DstOffsets, ListF<DstTzKey>> dstTzKeysByOffsets;

    static {
        LocalDateTime today = new LocalDate(DateTimeZone.UTC).toLocalDateTime(new LocalTime(12, 0));
        LocalDateTime smm = new LocalDateTime(today.getYear(), 6, 1, 12, 0);
        LocalDateTime wnt = new LocalDateTime(today.getYear(), 12, 1, 12, 0);

        smm = today.isBefore(smm) ? smm : smm.plusYears(1);
        wnt = today.isBefore(wnt) ? wnt : wnt.plusYears(1);

        summer = smm.isBefore(wnt) ? smm : smm.minusYears(1);
        winter = wnt.isBefore(smm) ? wnt : wnt.minusYears(1);

        dstTzKeysByOffsets = TimeZoneRegistry2.outlookTimeZones
                .getVTimeZones(Cf.x(DateTimeZone.getAvailableIDs()))
                .filterMap(IcsSuitableTimeZoneFinder::getDstTzKeyIfDst)
                .groupBy(DstTzKey::getOffsets);
    }

    public static DateTimeZone findSuitableTz(VTimeZone tz) {
        return findSuitableTz(tz, new LocalDate(DateTimeZone.UTC).toLocalDateTime(new LocalTime(12, 0)));
    }

    public static DateTimeZone findSuitableTz(VTimeZone tz, LocalDateTime forDate) {
        Option<DateTimeZone> found = findSuitableDstOrFixedTz(tz);

        return found.isPresent() ? found.get() : DateTimeZone.forOffsetMillis(getOffset(tz, forDate));
    }

    public static Option<DateTimeZone> findSuitableDstOrFixedTz(VTimeZone tz) {
        Option<DstTzKey> dstKey = getDstTzKeyIfDst(tz);

        if (dstKey.isPresent() && dstTzKeysByOffsets.containsKeyTs(dstKey.get().getOffsets())) {
            Function2<LocalDateTime, LocalDateTime, Long> minutesBetweenF = (a, b) -> Math.abs(
                    new Duration(a.toDateTime(DateTimeZone.UTC), b.toDateTime(DateTimeZone.UTC)).getStandardMinutes());

            Function<DstTzKey, Long> distanceF = (key) -> {
                long summerDist = minutesBetweenF.apply(key.summerTransition, dstKey.get().summerTransition);
                long winterDist = minutesBetweenF.apply(key.winterTransition, dstKey.get().winterTransition);

                return summerDist * summerDist + winterDist * winterDist;
            };
            ListF<DstTzKey> keys = dstTzKeysByOffsets.getOrThrow(dstKey.get().getOffsets());
            return Option.of(DateTimeZone.forID(keys.min(distanceF.andThenNaturalComparator()).tzId));
        }

        Option<Observance> fixed = getObservanceIfFixed(tz);
        if (fixed.isPresent()) {
            return Option.of(DateTimeZone.forOffsetMillis(getOffset(fixed.get())));
        }

        return Option.empty();
    }

    public static int getOffset(Observance observance) {
        return (int) (observance.getOffsetTo().getOffset().getTotalSeconds() * Dates.MILLIS_PER_SECOND);
    }

    public static int getOffset(VTimeZone tz, LocalDateTime date) {
        return getOffset(getObservance(tz, date));
    }

    public static Observance getObservance(VTimeZone tz, LocalDateTime date) {
        return getObservanceAndStartDateTime(tz, date).get1();
    }

    public static Option<LocalDateTime> getObservanceStartDateTime(VTimeZone tz, LocalDateTime date) {
        return getObservanceAndStartDateTime(tz, date).get2();
    }

    public static Option<LocalDateTime> getObservanceStartDateTime(Observance observance, LocalDate date) {
        Date result = observance.getLatestOnset(toDate(date.toLocalDateTime(LocalTime.MIDNIGHT)));

        if (result != null) {
            return Option.of(new LocalDate(result, DateTimeZone.UTC)
                    .toLocalDateTime(new LocalTime(observance.getStartDate().getDate().getTime(), DateTimeZone.UTC)));
        }
        return Option.empty();
    }

    private static Tuple2<Observance, Option<LocalDateTime>> getObservanceAndStartDateTime(
            VTimeZone tz, LocalDateTime date)
    {
        Observance prev = tz.getApplicableObservance(toDate(date.withFields(LocalTime.MIDNIGHT)));
        Observance next = tz.getApplicableObservance(toDate(date.withFields(LocalTime.MIDNIGHT).plusDays(1)));

        if (prev == next) return Tuple2.tuple(prev, getObservanceStartDateTime(prev, date.toLocalDate()));

        Option<LocalDateTime> prevStart = getObservanceStartDateTime(prev, date.toLocalDate());
        Option<LocalDateTime> nextStart = getObservanceStartDateTime(next, date.plusDays(1).toLocalDate());

        if (prevStart.isPresent() && prevStart.get().isBefore(date)) {
            return Tuple2.tuple(next, nextStart);
        } else {
            return Tuple2.tuple(prev, prevStart);
        }
    }

    static Option<DstTzKey> getDstTzKeyIfDst(VTimeZone tz) {
        Tuple2<Observance, Option<LocalDateTime>> summerTzAndStart = getObservanceAndStartDateTime(tz, summer);
        Tuple2<Observance, Option<LocalDateTime>> winterTzAndStart = getObservanceAndStartDateTime(tz, winter);

        Observance summerTz = summerTzAndStart.get1();
        Observance winterTz = winterTzAndStart.get1();

        if (!summerTz.getClass().equals(winterTz.getClass())
            && summerTz == getObservance(tz, summer.plusYears(1))
            && winterTz == getObservance(tz, winter.plusYears(1)))
        {
            int summerOffset = getOffset(summerTz);
            int winterOffset = getOffset(winterTz);

            if (summerOffset != winterOffset) {
                LocalDateTime summerTransition = summerTzAndStart.get2().get();
                LocalDateTime winterTransition = winterTzAndStart.get2().get();

                return Option.of(new DstTzKey(
                        tz.getTimeZoneId().getValue(),
                        summerOffset, winterOffset, summerTransition, winterTransition));
            }
        }
        return Option.empty();
    }

    static Option<Observance> getObservanceIfFixed(VTimeZone tz) {
        Observance summerTz = getObservance(tz, summer);
        Observance winterTz = getObservance(tz, winter);
        Observance nextTz = getObservance(tz, (summer.isBefore(winter) ? summer : winter).plusYears(1));

        return Option.when(summerTz == winterTz && winterTz == nextTz, nextTz);
    }

    private static Date toDate(LocalDateTime date) {
        return new Date(date.toDateTime(DateTimeZone.UTC).getMillis());
    }

    static class DstOffsets extends DefaultObject {
        private final int summerOffset;
        private final int winterOffset;

        public DstOffsets(int summerOffset, int winterOffset) {
            this.summerOffset = summerOffset;
            this.winterOffset = winterOffset;
        }
    }

    static class DstTzKey {
        private final String tzId;
        private final DstOffsets offsets;
        private final LocalDateTime summerTransition;
        private final LocalDateTime winterTransition;

        public DstTzKey(
                String tzId, int summerOffset, int winterOffset,
                LocalDateTime summerTransition, LocalDateTime winterTransition)
        {
            this.tzId = tzId;
            this.offsets = new DstOffsets(summerOffset, winterOffset);
            this.summerTransition = summerTransition;
            this.winterTransition = winterTransition;
        }

        public DstOffsets getOffsets() {
            return offsets;
        }
    }
}
