package ru.yandex.calendar.logic.ics.iv5j.ical;

import net.fortuna.ical4j.model.component.VTimeZone;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDateTime;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVTimeZone;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author dbrylev
 */
public class IcsVTimeZones {
    private static final Logger logger = LoggerFactory.getLogger(IcsVTimeZones.class);

    private final MapF<String, DateTimeZone> suitableTzById;
    private final MapF<String, VTimeZone> floatingTzById;
    private final DateTimeZone fallbackTz;

    public IcsVTimeZones(
            MapF<String, DateTimeZone> suitableTzById,
            MapF<String, VTimeZone> floatingTzById, DateTimeZone fallbackTz)
    {
        this.suitableTzById = suitableTzById;
        this.floatingTzById = floatingTzById;
        this.fallbackTz = fallbackTz;
    }

    public static IcsVTimeZones cons(ListF<IcsVTimeZone> timezones, DateTimeZone fallbackTz, boolean useSuitableTzs) {
        timezones = timezones.filterNot(tz -> IcsTimeZones.forId(tz.getTzId()).isPresent());

        MapF<String, DateTimeZone> suitableTzById = Cf.hashMap();
        MapF<String, VTimeZone> floatingTzById = Cf.hashMap();

        timezones.forEach(tz -> {
            VTimeZone vtz = tz.toComponent();
            if (!useSuitableTzs) {
                logger.debug("Timezone is not recognized by id {}. Will use fixed offsets", tz.getTzId());
                floatingTzById.put(tz.getTzId(), vtz);
                return;
            }
            Option<DateTimeZone> suitable = IcsSuitableTimeZoneFinder.findSuitableDstOrFixedTz(vtz);

            if (suitable.isPresent()) {
                logger.debug("Timezone is not recognized by id {}. Will use suitable {} instead",
                        tz.getTzId(), suitable.get().getID());
                suitableTzById.put(tz.getTzId(), suitable.get());
            } else {
                logger.debug("Timezone is not recognized by id {} and no suitable timezone found", tz.getTzId());
                floatingTzById.put(tz.getTzId(), vtz);
            }
        });
        return new IcsVTimeZones(suitableTzById, floatingTzById, fallbackTz);
    }

    public static IcsVTimeZones fallback(DateTimeZone timezone) {
        return new IcsVTimeZones(Cf.map(), Cf.map(), timezone);
    }

    public IcsVTimeZones withFallback(DateTimeZone timezone) {
        return new IcsVTimeZones(suitableTzById, floatingTzById, timezone);
    }

    public DateTime getDateTime(LocalDateTime localDateTime, String tzId) {
        return localDateTime.toDateTime(getOrFallbackForDate(tzId, localDateTime));
    }

    public Option<DateTimeZone> getForDate(String tzId, LocalDateTime dateTime) {
        Option<DateTimeZone> byIdO = IcsTimeZones.forId(tzId);
        if (byIdO.isPresent()) {
            return byIdO;
        }
        Option<DateTimeZone> suitableO = suitableTzById.getO(tzId);
        if (suitableO.isPresent()) {
            return suitableO;
        }
        if (floatingTzById.containsKeyTs(tzId)) {
            return Option.of(DateTimeZone.forOffsetMillis(
                    IcsSuitableTimeZoneFinder.getOffset(floatingTzById.getOrThrow(tzId), dateTime)));
        }
        return Option.empty();
    }

    public DateTimeZone getOrFallbackForDate(String tzId, LocalDateTime dateTime) {
        return getForDate(tzId, dateTime).getOrElse(fallbackTz);
    }

    public DateTimeZone getFallbackTz() {
        return fallbackTz;
    }
}
