package ru.yandex.calendar.logic.ics.iv5j.ical;

import net.fortuna.ical4j.data.CalendarBuilder;
import net.fortuna.ical4j.model.Calendar;
import net.fortuna.ical4j.model.Component;
import net.fortuna.ical4j.model.component.VTimeZone;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.TimeZones;
import ru.yandex.calendar.util.exception.ExceptionUtils;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.property.PropertiesUtils;

/**
 * Remake of {@link net.fortuna.ical4j.model.TimeZoneRegistryImpl}
 * to have multiple instances with different timezone definitions.
 *
 * @author shinderuk
 */
public class TimeZoneRegistry2 {

    public static final TimeZoneRegistry2 fullTimeZones =
            new TimeZoneRegistry2("ru/yandex/calendar/logic/ics/iv5j/ical/zoneinfo/");

    public static final TimeZoneRegistry2 outlookTimeZones =
            new TimeZoneRegistry2("ru/yandex/calendar/logic/ics/iv5j/ical/zoneinfo-outlook/");


    private static final Logger logger = LoggerFactory.getLogger(TimeZoneRegistry2.class);

    private static final MapF<String, String> legacyTzNames =
            Cf.x(PropertiesUtils.load(new ClassPathResourceInputStreamSource(
                    "ru/yandex/calendar/logic/ics/iv5j/ical/legacy-tz-names.properties")));

    private static final MapF<String, String> timeZoneAliases =
            Cf.x(PropertiesUtils.load(new ClassPathResourceInputStreamSource("net/fortuna/ical4j/model/tz.alias")))
                    .filter(Tuple2List.fromPairs(
                            "Etc/GMT", "Europe/London",
                            "Etc/UCT", "Europe/London",
                            "Etc/UTC", "Europe/London").containsF().notF())
                    .plus(Cf.toMap(Tuple2List.fromPairs(
                            "America/Montreal", "America/Toronto",
                            "America/Shiprock", "America/Denver",
                            "America/Santa_Isabel", "America/Tijuana",
                            "Antarctica/South_Pole", "Antarctica/McMurdo",
                            "Asia/Chongqing", "Asia/Shanghai",
                            "Asia/Chungking", "Asia/Shanghai",
                            "Asia/Harbin", "Asia/Shanghai",
                            "Asia/Kashgar", "Asia/Urumqi",
                            "Asia/Rangoon", "Asia/Yangon"
                    )));

    private final MapF<String, VTimeZone> vTimeZonesById = Cf.hashMap();

    private final String resourcePrefix;

    private TimeZoneRegistry2(String resourcePrefix) {
        this.resourcePrefix = resourcePrefix;
    }

    public synchronized Option<VTimeZone> getVTimeZone(String id) {
        Option<VTimeZone> vTimeZone = vTimeZonesById.getO(id);

        if (vTimeZone.isPresent()) {
            return vTimeZone;
        }

        Option<VTimeZone> vTimeZoneFromUtcOffset = parseVTimeZoneFromUtcOffset(id);
        if (vTimeZoneFromUtcOffset.isPresent()) {
            return vTimeZoneFromUtcOffset;
        }

        Option<DateTimeZone> tz = Option.when(id.startsWith("Etc/GMT"), id).filterMap(AuxDateTime::getVerifyDateTimeZoneSafe);
        if (tz.isPresent()) {
            return Option.of(VTimeZones.vTimeZoneForUtcOffset(Duration.millis(tz.get().getOffset(0))));
        }

        Option<String> nonLegacyName = legacyTzNames.getO(id);
        if (nonLegacyName.isPresent()) {
            return getVTimeZone(nonLegacyName.get());
        }

        Option<String> alias = timeZoneAliases.getO(id);
        if (alias.isPresent()) {
            return getVTimeZone(alias.get());
        }

        // load vTimeZone from .ics file
        vTimeZone = loadVTimeZoneSafe(id);
        if (vTimeZone.isPresent()) {
            vTimeZonesById.put(id, vTimeZone.get());
            return vTimeZone;
        }

        return Option.empty();
    }

    public ListF<VTimeZone> getVTimeZones(CollectionF<String> ids) {
        return ids.filterMap(this::getVTimeZone);
    }

    private Option<VTimeZone> parseVTimeZoneFromUtcOffset(String id) {
        Option<Duration> offset = TimeZones.parseUtcOffset(id);
        if (offset.isPresent()) {
            return Option.of(VTimeZones.vTimeZoneForUtcOffset(offset.get()));
        } else {
            return Option.empty();
        }
    }

    private Option<VTimeZone> loadVTimeZoneSafe(String id) {
        try {
            return loadVTimeZone(id);
        } catch (Exception e) {
            logger.error("Error loading timezone " + id + ": " + e, e);
            return Option.empty();
        }
    }

    private Option<VTimeZone> loadVTimeZone(String id) {
        InputStreamSource icsFile = new ClassPathResourceInputStreamSource(resourcePrefix + id + ".ics");
        if (icsFile.exists()) {
            Calendar calendar = parseCalendar(icsFile);
            VTimeZone vTimeZone = (VTimeZone) calendar.getComponent(Component.VTIMEZONE);
            return Option.of(vTimeZone);
        }
        return Option.empty();
    }

    private Calendar parseCalendar(InputStreamSource icsFile) {
        return icsFile.read(input -> {
            try {
                return new CalendarBuilder().build(input);
            } catch (Exception e) {
                throw ExceptionUtils.translate(e);
            }
        });
    }
}
