package ru.yandex.calendar.logic.ics.iv5j.ical.component;

import net.fortuna.ical4j.model.Property;
import net.fortuna.ical4j.model.component.CalendarComponent;
import net.fortuna.ical4j.model.component.VAlarm;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.ics.iv5j.ical.PropertyNames;
import ru.yandex.calendar.logic.ics.iv5j.ical.meta.MappingToIcal4j;
import ru.yandex.calendar.logic.ics.iv5j.ical.parameter.IcsRelated;
import ru.yandex.calendar.logic.ics.iv5j.ical.parameter.IcsValue;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsAction;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsDescription;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsDuration;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsProperty;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsRepeat;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsSummary;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsTrigger;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.IcsDurationValue;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.dateTime.IcsDateTimeFormats;
import ru.yandex.misc.lang.Validate;

/**
 * @author Stepan Koltsov
 */
@MappingToIcal4j(name=CalendarComponent.VALARM, theirClass=VAlarm.class)
public class IcsVAlarm extends IcsComponentBase<IcsVAlarm, VAlarm> {

    public IcsVAlarm(ListF<IcsProperty> properties, ListF<? extends IcsComponent> components) {
        super(CalendarComponent.VALARM, properties, components);
    }

    public IcsVAlarm() {
        this(Cf.<IcsProperty>list(), Cf.<IcsComponent>list());
    }

    @Override
    public IcsVAlarm withProperties(ListF<IcsProperty> properties) {
        return new IcsVAlarm(properties, getComponents());
    }

    public IcsVAlarm withAction(IcsAction icsAction) {
        return this
            .removeProperties(Property.ACTION)
            .addProperty(icsAction);
    }

    public IcsVAlarm withTrigger(IcsTrigger icsTrigger) {
        return this
            .removeProperties(Property.TRIGGER)
            .addProperty(icsTrigger);
    }

    public IcsVAlarm withSummary(String summary) {
        return this
            .removeProperties(Property.SUMMARY)
            .addProperty(new IcsSummary(summary));
    }

    public IcsVAlarm withDescription(String description) {
        return this
            .removeProperties(Property.DESCRIPTION)
            .addProperty(new IcsDescription(description));
    }

    public IcsTrigger getTrigger() {
        return (IcsTrigger) getProperty(Property.TRIGGER).get();
    }

    public IcsAction getAction() {
        return (IcsAction) getProperty(Property.ACTION).get();
    }

    public Option<IcsRepeat> getRepeat() {
        return getProperty(Property.REPEAT);
    }

    public Option<IcsDuration> getDuration() {
        return getProperty(Property.DURATION);
    }

    public boolean hasRelativeTrigger() {
        return getTrigger().getValueTypeOrGetDefault().sameAs(IcsValue.DURATION);
    }

    public IcsRelated getTriggerRelation() {
        Validate.isTrue(hasRelativeTrigger());
        return getTrigger().getRelationOrGetDefault();
    }

    public Instant getMainNotificationTimestamp() {
        Validate.isFalse(hasRelativeTrigger());
        return IcsDateTimeFormats.parseDateTime(getTrigger().getValue());
    }

    public Duration getMainNotificationOffset() {
        Validate.isTrue(hasRelativeTrigger());
        return IcsDurationValue.parse(getTrigger().getValue()).getPeriod().toStandardDuration();
    }

    public ListF<Duration> getNotificationOffsets() {
        Duration firstOffset = getMainNotificationOffset();
        return Cf.list(firstOffset).plus(getRepetitionOffsets().map(firstOffset::plus));
    }

    public ListF<Instant> getNotificationTimestamps() {
        Instant firstTs = getMainNotificationTimestamp();
        return Cf.list(firstTs).plus(getRepetitionOffsets().map(firstTs::plus));
    }

    private ListF<Duration> getRepetitionOffsets() {
        Validate.isTrue(getRepeat().isPresent() == getDuration().isPresent());
        if (!getRepeat().isPresent()) return Cf.list();

        int count = getRepeat().get().getCount();
        Duration offset = getDuration().get().getPeriod().toStandardDuration();

        ListF<Duration> result = Cf.arrayList(offset);
        for (int i = 1; i < count; ++i) {
            result.add(result.last().plus(offset));
        }
        return result;
    }

    public Option<String> getUid() {
        return getPropertyValue(Property.UID).orElse(getPropertyValue(PropertyNames.X_WR_ALARMUID));
    }

    public Option<String> getRelatedTo() {
        return getPropertyValue(Property.RELATED_TO);
    }

    public Option<Instant> getAcknowledgedTimestamp() {
        return getPropertyValue(PropertyNames.ACKNOWLEDGED).map(IcsDateTimeFormats.parseDateTimeF());
    }

    public boolean isDefault() {
        return getPropertyValue(PropertyNames.DEFAULT_ALARM)
                .orElse(getPropertyValue(PropertyNames.X_APPLE_DEFAULT_ALARM)).isSome("TRUE");
    }

    public static Function<IcsVAlarm, Option<String>> getUidF() {
        return new Function<IcsVAlarm, Option<String>>() {
            public Option<String> apply(IcsVAlarm a) {
                return a.getUid();
            }
        };
    }

    public static Function<IcsVAlarm, Option<String>> getRelatedToF() {
        return new Function<IcsVAlarm, Option<String>>() {
            public Option<String> apply(IcsVAlarm a) {
                return a.getRelatedTo();
            }
        };
    }

    public static Function1B<IcsVAlarm> isDefaultF() {
        return new Function1B<IcsVAlarm>() {
            public boolean apply(IcsVAlarm a) {
                return a.isDefault();
            }
        };
    }

    public static Function1B<IcsVAlarm> hasRelativeTriggerF() {
        return new Function1B<IcsVAlarm>() {
            public boolean apply(IcsVAlarm a) {
                return a.hasRelativeTrigger();
            }
        };
    }

    public static Function1B<IcsVAlarm> isAcknowledgedF() {
        return IcsVAlarm.hasPropertyF(PropertyNames.ACKNOWLEDGED).uncheckedCast();
    }

    public static Function1B<IcsVAlarm> isGeoF() {
        return IcsVAlarm.hasPropertyF(PropertyNames.X_APPLE_PROXIMITY)
                .orF(IcsVAlarm.hasPropertyF(PropertyNames.X_APPLE_STRUCTURED_LOCATION))
                .uncheckedCast();
    }

} //~
