package ru.yandex.calendar.logic.ics.iv5j.ical.component;

import java.util.Optional;

import net.fortuna.ical4j.model.Component;
import net.fortuna.ical4j.model.Property;
import net.fortuna.ical4j.model.component.VEvent;
import net.fortuna.ical4j.model.property.RecurrenceId;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.Period;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.meta.MappingToIcal4j;
import ru.yandex.calendar.logic.ics.iv5j.ical.parameter.IcsPartStat;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsAttendee;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsDescription;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsDtEnd;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsDtStamp;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsDtStart;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsDuration;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsExDate;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsLocation;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsOrganizer;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsProperty;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsRDate;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsRecurrenceId;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsSequence;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsSummary;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsTransp;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsUid;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.dateTime.IcsDateTime;
import ru.yandex.misc.email.Email;

@MappingToIcal4j(name=Component.VEVENT, theirClass=VEvent.class)
public class IcsVEvent extends IcsComponentBase<IcsVEvent, VEvent> {

    public IcsVEvent(ListF<IcsProperty> properties, ListF<? extends IcsComponent> components) {
        super(Component.VEVENT, properties, components);
    }

    public ListF<IcsVAlarm> getAlarms() {
        return getComponents().uncheckedCast();
    }

    public IcsVEvent() {
        this(Cf.list(), Cf.<IcsVAlarm>list());
    }

    @Override
    public IcsVEvent withProperties(ListF<IcsProperty> properties) {
        return new IcsVEvent(properties, getComponents());
    }

    public IcsCalendar makeCalendar() {
        return new IcsCalendar(Cf.list(this));
    }

    public IcsVEvent withOrganizer(Email email) {
        return this
            .removeProperties(Property.ORGANIZER)
            .addProperty(new IcsOrganizer(email));
    }

    public IcsVEvent removeAttendee(Email email) {
        Function1B<IcsProperty> f =
            IcsProperty.nameIsF(Property.ATTENDEE).<IcsProperty>uncheckedCast()
                .andF(IcsAttendee.emailIsF(email).uncheckedCast());
        return filterProperties(f.notF());
    }

    public IcsVEvent updateAttendee(IcsAttendee attendee) {
        return this
            .removeAttendee(attendee.getEmail())
            .addProperty(attendee);
    }

    public IcsVEvent addAttendee(Email email) {
        return this.addProperty(new IcsAttendee(email));
    }

    public IcsVEvent addAttendee(Email email, IcsPartStat partStat) {
        return this.addProperty(new IcsAttendee(email, partStat));
    }

    public IcsVEvent addVAlarm(IcsVAlarm alarm) {
        return this.addComponent(alarm);
    }

    public IcsVEvent withSummary(String summary) {
        return this
            .removeProperties(Property.SUMMARY)
            .addProperty(new IcsSummary(summary));
    }

    public IcsVEvent withUid(String uid) {
        return this
            .removeProperties(Property.UID)
            .addProperty(new IcsUid(uid));
    }

    public IcsVEvent withLocation(String location) {
        return this
                .removeProperties(Property.LOCATION)
                .addProperty(new IcsLocation(location));
    }

    public IcsVEvent withDescription(String description) {
        return this
                .removeProperties(Property.DESCRIPTION)
                .addProperty(new IcsDescription(description));
    }

    public IcsVEvent withDtStart(IcsDtStart start) {
        return this
            .removeProperties(Property.DTSTART)
            .addProperty(start);
    }

    public IcsVEvent withDtStart(IcsDateTime start) {
        return withDtStart(new IcsDtStart(start));
    }

    public IcsVEvent withDtStart(Instant start) {
        return withDtStart(IcsDateTime.instant(start));
    }

    public IcsVEvent withDtStart(DateTime start) {
        return withDtStart(IcsDateTime.dateTime(start));
    }

    public IcsVEvent withDtStart(LocalDate start) {
        return withDtStart(IcsDateTime.localDate(start));
    }

    public IcsVEvent withDtEnd(IcsDtEnd end) {
        return this
            .removeProperties(Property.DTEND)
            .addProperty(end);
    }

    public IcsVEvent withDtEnd(IcsDateTime end) {
        return withDtEnd(new IcsDtEnd(end));
    }

    public IcsVEvent withDtEnd(Instant end) {
        return withDtEnd(IcsDateTime.instant(end));
    }

    public IcsVEvent withDtEnd(DateTime end) {
        return withDtEnd(IcsDateTime.dateTime(end));
    }

    public IcsVEvent withDtEnd(LocalDate end) {
        return withDtEnd(IcsDateTime.localDate(end));
    }

    public IcsVEvent withRecurrence(IcsRecurrenceId recurrenceId) {
        return this
                .removeProperties(Property.RECURRENCE_ID)
                .addProperty(recurrenceId);

    }

    public IcsVEvent withRecurrence(Instant recurrenceId) {
        return withRecurrence(new IcsRecurrenceId(recurrenceId));
    }

    public IcsVEvent addRDate(Instant rdate) {
        return this.addProperty(new IcsRDate(rdate));
    }

    public IcsVEvent withExdates(ListF<IcsExDate> exdates) {
        return withProperties(getProperties()
                .filterNot(p -> p.getName().equals(Property.EXDATE)).plus(exdates));
    }

    public IcsVEvent withSequenece(int sequence) {
        return this
            .removeProperties(Property.SEQUENCE)
            .addProperty(new IcsSequence(sequence));
    }

    public Option<Integer> getSequence() {
        return getPropertyValue(Property.SEQUENCE).map(Cf.Integer::parse);
    }

    public IcsVEvent withDtStampNow() {
        return withDtStamp(new Instant());
    }

    public IcsVEvent withDtStamp(Instant dtStamp) {
        return withDtStamp(new IcsDtStamp(dtStamp));
    }

    public IcsVEvent withDtStamp(IcsDtStamp dtStamp) {
        return this
            .removeProperties(Property.DTSTAMP)
            .addProperty(dtStamp);
    }

    public IcsVEvent withTransp(IcsTransp icsTransp) {
        return this
            .removeProperties(Property.TRANSP)
            .addProperty(icsTransp);
    }

    public Option<String> getUid() {
        return getPropertyValue(Property.UID);
    }

    public static Function<IcsVEvent, Option<String>> getUidF() {
        return IcsVEvent::getUid;
    }

    public Optional<String> getSummary() {
        return getPropertyValue(Property.SUMMARY).toOptional();
    }

    public Optional<String> getDescription() {
        return getPropertyValue(Property.DESCRIPTION).toOptional();
    }

    public Optional<String> getLocation() {
        return getPropertyValue(Property.LOCATION).toOptional();
    }

    public Option<IcsRecurrenceId> getRecurrenceId() {
        return getProperty(RecurrenceId.RECURRENCE_ID);
    }

    public Option<Instant> getRecurrenceIdInstant(IcsVTimeZones tz) {
        return getRecurrenceId().map(IcsRecurrenceId.getInstantF(tz));
    }

    public Option<Instant> getRecurrenceIdInstantUtc() {
        return getRecurrenceIdInstant(IcsVTimeZones.fallback(DateTimeZone.UTC));
    }

    public static Function1B<IcsVEvent> isRecurrenceF() {
        return a -> a.getRecurrenceId().isPresent();
    }

    public boolean isRepeatingOrRecurrence() {
        return getRRules().isNotEmpty() || getRDates().isNotEmpty() || isRecurrence();
    }

    public boolean isRecurrence() {
        return isRecurrenceF().apply(this);
    }

    public static IcsVEvent fromIcal4j(VEvent ve) {
        return ComponentsMeta.M.fromIcal4j(ve);
    }

    public Instant getStart(IcsVTimeZones tz) {
        return getStart().getInstant(tz);
    }

    public IcsDateTime getStart() {
        return getDtStart().get().getIcsDateTime();
    }

    public Option<String> getComment() {
        return getPropertyValue(Property.COMMENT);
    }

    public static Function<IcsVEvent, IcsDateTime> getStartF() {
        return IcsVEvent::getStart;
    }

    public boolean isAllDay() {
        return getDtStart().isPresent() && getDtStart().get().isDate();
    }

    public IcsDateTime getEnd() {
        Option<IcsDtEnd> dtEnd = getDtEnd();
        if (dtEnd.isPresent()) {
            return dtEnd.get().getIcsDateTime();
        }

        Option<IcsDuration> duration = getDuration();
        if (duration.isPresent()) {
            return getStart().plus(duration.get().getPeriod());
        }

        if (getStart().isDate()) {
            return getStart().plus(Period.days(1));
        } else {
            return getStart();
        }
    }

    public Instant getEnd(IcsVTimeZones tz) {
        return getEnd().getInstant(tz);
    }

    public ListF<Email> getAttendeeEmailsSafe() {
        return getAttendees().flatMap(IcsAttendee.getEmailSafeF());
    }

    public Option<Email> getOrganizerEmailSafe() {
        Option<IcsOrganizer> organizer = getOrganizer();
        if (!organizer.isPresent() || !organizer.get().isEmail()) {
            return Option.empty();
        } else {
            return Option.of(organizer.get().getEmail());
        }
    }

    public ListF<Email> getParticipantEmailsSafe() {
        return getOrganizerEmailSafe().plus(getAttendeeEmailsSafe()).stableUnique();
    }

    public boolean isMeeting() {
        return getOrganizer().isPresent() || getAttendees().isNotEmpty();
    }

    public static Function1B<IcsVEvent> isMeetingF() {
        return IcsVEvent::isMeeting;
    }

    public static Function<IcsVEvent, ListF<Email>> getParticipantEmailsSafeF() {
        return IcsVEvent::getParticipantEmailsSafe;
    }
} //~
