package ru.yandex.calendar.logic.ics.iv5j.ical.type.dateTime;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.ReadablePeriod;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.VTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.parameter.IcsParameter;
import ru.yandex.calendar.logic.ics.iv5j.ical.parameter.IcsTzId;
import ru.yandex.calendar.logic.ics.iv5j.ical.parameter.IcsValue;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author Stepan Koltsov
 */
public abstract class IcsDateTime extends DefaultObject {
    // sadly we have no Either3

    private UnsupportedOperationException unsupportedOperationException() {
        return new UnsupportedOperationException("unsupported operation on " + this);
    }

    public Option<String> getTzId() {
        return this instanceof DateTimeImpl ? Option.of(((DateTimeImpl) this).tzId) : Option.empty();
    }

    public LocalDate getLocalDateRaw() {
        if (this instanceof LocalDateImpl) {
            return ((LocalDateImpl) this).localDate;
        } else {
            throw unsupportedOperationException();
        }
    }

    public LocalDateTime getLocalDateTimeRaw() {
        if (this instanceof LocalDateTimeImpl) {
            return ((LocalDateTimeImpl) this).localDateTime;
        } else {
            throw unsupportedOperationException();
        }
    }

    public interface Reducer<T> {
        T dateTime(LocalDateTime localDateTime, String tzId);
        T localDateTime(LocalDateTime localDateTime);
        T localDate(LocalDate localDate);
    }

    public <T> T reduce(Reducer<T> r) {
        if (this instanceof DateTimeImpl) {
            DateTimeImpl impl = (DateTimeImpl) this;
            return r.dateTime(impl.dateTime, impl.tzId);
        } else if (this instanceof LocalDateTimeImpl) {
            return r.localDateTime(getLocalDateTimeRaw());
        } else if (this instanceof LocalDateImpl) {
            return r.localDate(getLocalDateRaw());
        } else {
            throw new IllegalStateException();
        }
    }

    public Instant getInstant(IcsVTimeZones timezones) {
        return getDateTime(timezones).toInstant();
    }

    public DateTime getDateTime(IcsVTimeZones timezones) {
        return reduce(new Reducer<DateTime>() {

            @Override
            public DateTime dateTime(LocalDateTime dateTime, String tzId) {
                return timezones.getDateTime(dateTime, tzId);
            }

            @Override
            public DateTime localDateTime(LocalDateTime localDateTime) {
                return AuxDateTime.toDateTimeIgnoreGap(localDateTime, timezones.getFallbackTz());
            }

            @Override
            public DateTime localDate(LocalDate localDate) {
                return localDate.toDateTimeAtStartOfDay(timezones.getFallbackTz());
            }
        });
    }

    public Either<LocalDate, LocalDateTime> getLocalDateOrLocalDateTime() {
        return reduce(new Reducer<Either<LocalDate, LocalDateTime>>() {
            @Override
            public Either<LocalDate, LocalDateTime> dateTime(LocalDateTime dateTime, String tzId) {
                return Either.right(dateTime);
            }

            @Override
            public Either<LocalDate, LocalDateTime> localDate(LocalDate localDate) {
                return Either.left(localDate);
            }

            @Override
            public Either<LocalDate, LocalDateTime> localDateTime(LocalDateTime localDateTime) {
                return Either.right(localDateTime);
            }
        });
    }

    public LocalDateTime getLocalDateTime() {
        return getLocalDateOrLocalDateTime().fold(TimeUtils.localDate.localDateTimeMidnightF(), Function.<LocalDateTime>identityF());
    }

    public boolean isDate() {
        return this instanceof LocalDateImpl;
    }

    public String toPropertyValue() {
        return reduce(new Reducer<String>() {

            @Override
            public String dateTime(LocalDateTime dateTime, String tzId) {
                if (DateTimeZone.UTC.getID().equals(tzId)) {
                    return IcsDateTimeFormats.DATE_TIME_FORMATTER.print(dateTime) + "Z";
                } else {
                    return IcsDateTimeFormats.DATE_TIME_FORMATTER.print(dateTime);
                }
            }

            @Override
            public String localDateTime(LocalDateTime localDateTime) {
                return IcsDateTimeFormats.DATE_TIME_FORMATTER.print(localDateTime);
            }

            @Override
            public String localDate(LocalDate localDate) {
                return IcsDateTimeFormats.DATE_FORMATTER.print(localDate);
            }

        });
    }

    public ListF<IcsParameter> toPropertyParameters() {
        return reduce(new Reducer<ListF<IcsParameter>>() {

            @Override
            public ListF<IcsParameter> dateTime(LocalDateTime dateTime, String tzId) {
                if (DateTimeZone.UTC.getID().equals(tzId)) {
                    return Cf.list();

                } else {
                    Option<DateTimeZone> tzO = AuxDateTime.getVerifyDateTimeZoneSafe(tzId);
                    return Cf.<IcsParameter>list(tzO.isPresent() && tzO.get().isFixed()
                            ? new IcsTzId(VTimeZones.utcTzId(Duration.millis(tzO.get().getOffset(0))))
                            : new IcsTzId(tzId));
                }
            }

            @Override
            public ListF<IcsParameter> localDateTime(LocalDateTime localDateTime) {
                return Cf.list();
            }

            @Override
            public ListF<IcsParameter> localDate(LocalDate localDate) {
                return Cf.<IcsParameter>list(IcsValue.DATE);
            }
        });
    }

    private static final class DateTimeImpl extends IcsDateTime {
        private final LocalDateTime dateTime;
        private final String tzId;

        private DateTimeImpl(LocalDateTime dateTime, String tzId) {
            this.dateTime = dateTime;
            this.tzId = tzId;
        }
    }

    private static final class LocalDateImpl extends IcsDateTime {
        private final LocalDate localDate;

        private LocalDateImpl(LocalDate localDate) {
            this.localDate = localDate;
        }
    }

    private static final class LocalDateTimeImpl extends IcsDateTime {
        private final LocalDateTime localDateTime;

        private LocalDateTimeImpl(LocalDateTime localDateTime) {
            this.localDateTime = localDateTime;
        }
    }

    public static IcsDateTime dateTime(LocalDateTime dateTime, String tzId) {
        return new DateTimeImpl(dateTime, tzId);
    }

    public static IcsDateTime dateTime(DateTime dateTime) {
        return dateTime(dateTime.toLocalDateTime(), dateTime.getZone().getID());
    }

    public static IcsDateTime instant(Instant instant) {
        return dateTime(instant.toDateTime(DateTimeZone.UTC));
    }

    public static IcsDateTime localDateTime(LocalDateTime localDateTime) {
        return new LocalDateTimeImpl(localDateTime);
    }

    public static IcsDateTime localDate(LocalDate localDate) {
        return new LocalDateImpl(localDate);
    }

    public static IcsDateTime parse(String string, Option<String> tzId) {
        if (string.length() == 8) {
            return localDate(IcsDateTimeFormats.DATE_FORMATTER.parseDateTime(string).toLocalDate());
        } else if (string.endsWith("Z")) {
            String substring = string.substring(0, string.length() - 1);
            return parse(substring, Option.of(DateTimeZone.UTC.getID()));
        } else {
            LocalDateTime localDateTime = IcsDateTimeFormats.DATE_TIME_FORMATTER.parseDateTime(string).toLocalDateTime();
            if (tzId.isPresent()) {
                return dateTime(localDateTime, tzId.get());
            } else {
                return localDateTime(localDateTime);
            }
        }
    }

    public IcsDateTime plus(final ReadablePeriod period) {
        return reduce(new Reducer<IcsDateTime>() {

            @Override
            public IcsDateTime dateTime(LocalDateTime dateTime, String tzId) {
                return IcsDateTime.dateTime(dateTime.plus(period), tzId);
            }

            @Override
            public IcsDateTime localDateTime(LocalDateTime localDateTime) {
                return IcsDateTime.localDateTime(localDateTime.plus(period));
            }

            @Override
            public IcsDateTime localDate(LocalDate localDate) {
                return IcsDateTime.localDate(localDate.plus(period));
            }
        });
    }

} //~
