package ru.yandex.calendar.logic.ics.iv5j.ical.type.recur;

import net.fortuna.ical4j.model.WeekDay;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartByDay;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartByHour;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartByMinute;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartByMonth;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartByMonthDay;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartByNumberList;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartBySecond;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartBySetPos;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartByWeekNo;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartByYearDay;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartCount;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartFreq;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartInterval;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartUntil;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartWkSt;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.type.IcsRecurRulePartXExclude;
import ru.yandex.calendar.logic.ics.iv5j.support.IvObject;
import ru.yandex.misc.lang.Validate;

/**
 * @author Stepan Koltsov
 */
public class IcsRecur {

    private final ListF<IcsRecurRulePart> parts;

    public IcsRecur(Freq frequency) {
        this.parts = Cf.<IcsRecurRulePart>list(new IcsRecurRulePartFreq(frequency));
    }

    public IcsRecur(ListF<IcsRecurRulePart> parts) {
        this.parts = parts;
        Validate.some(getPartByType(IcsRecurRulePartFreq.class), "Frequency is required");
        boolean isCountDefined = getPartByType(IcsRecurRulePartCount.class).isPresent();
        boolean isUntilDefined = getPartByType(IcsRecurRulePartUntil.class).isPresent();
        Validate.isFalse(isCountDefined && isUntilDefined, "Until and count must not be occur in the same recur");
        Validate.sameSize(parts, parts.map(IvObject.nameF()).stableUnique(), "Parts names should be unique");
    }

    public ListF<IcsRecurRulePart> getParts() {
        return parts;
    }

    public String getValue() {
        return parts.map(Function.<IcsRecurRulePart>toStringF()).mkString(";");
    }

    public <A extends IcsRecurRulePart> Option<A> getPartByType(Class<A> type) {
        return parts.filterByType(type).singleO();
    }

    public Option<String> getValueByType(Class<? extends IcsRecurRulePart> type) {
        return getPartByType(type).map(IcsRecurRulePart.getValueF());
    }

    public Option<IcsRecurRulePart> getPartByName(String name) {
        return parts.filter(IvObject.nameIsF(name)).singleO();
    }

    public Option<String> getValueByName(String name) {
        return getPartByName(name).map(IcsRecurRulePart.getValueF());
    }

    private ListF<IcsRecurRulePart> getPartsWithoutPart(String name) {
        return parts.filter(IvObject.nameIsF(name).notF());
    }

    public IcsRecur withoutPart(String name) {
        return new IcsRecur(getPartsWithoutPart(name));
    }

    public IcsRecur withPart(IcsRecurRulePart part) {
        return new IcsRecur(getPartsWithoutPart(part.getName()).plus(part));
    }

    public IcsRecur withCount(int count) {
        return withPart(new IcsRecurRulePartCount(count));
    }

    public IcsRecur withUntilTs(Instant instant) {
        return withPart(new IcsRecurRulePartUntil(IcsRecurUntil.dateTime(instant)));
    }

    public IcsRecur withUntilDate(LocalDate date) {
        return withPart(new IcsRecurRulePartUntil(IcsRecurUntil.date(date)));
    }

    public IcsRecur withInterval(int interval) {
        return withPart(new IcsRecurRulePartInterval(interval));
    }

    public IcsRecur withWeekStartDay(WeekDay day) {
        return withPart(new IcsRecurRulePartWkSt(day));
    }

    public IcsRecur withXExclude(CollectionF<Instant> excludes) {
        return excludes.isNotEmpty() ? withPart(new IcsRecurRulePartXExclude(excludes.toList())) : this;
    }

    public Freq getFreq() {
        return getPartByType(IcsRecurRulePartFreq.class).single().getFreq();
    }

    public Option<Integer> getInterval() {
        return getPartByType(IcsRecurRulePartInterval.class).map(IcsRecurRulePartInterval.getIntervalF());
    }

    public Option<Integer> getCount() {
        return getPartByType(IcsRecurRulePartCount.class).map(IcsRecurRulePartCount.getCountF());
    }

    public Option<IcsRecurUntil> getUntil() {
        return getPartByType(IcsRecurRulePartUntil.class).map(IcsRecurRulePartUntil.getUntilF());
    }

    public Option<WeekDay> getWeekStartDay() {
        return getPartByType(IcsRecurRulePartWkSt.class).map(IcsRecurRulePartWkSt.getDayF());
    }

    private ListF<Integer> getNumberList(Class<? extends IcsRecurRulePartByNumberList> part) {
        return getPartByType(part).flatMap(IcsRecurRulePartByNumberList.getNumbersF());
    }

    public ListF<Integer> getSecondList() {
        return getNumberList(IcsRecurRulePartBySecond.class);
    }

    public ListF<Integer> getMinuteList() {
        return getNumberList(IcsRecurRulePartByMinute.class);
    }

    public ListF<Integer> getHourList() {
        return getNumberList(IcsRecurRulePartByHour.class);
    }

    public ListF<WeekDay> getDayList() {
        return getPartByType(IcsRecurRulePartByDay.class).flatMap(IcsRecurRulePartByDay.getDaysF());
    }

    public ListF<Integer> getMonthDayList() {
        return getNumberList(IcsRecurRulePartByMonthDay.class);
    }

    public ListF<Integer> getYearDayList() {
        return getNumberList(IcsRecurRulePartByYearDay.class);
    }

    public ListF<Integer> getWeekNoList() {
        return getNumberList(IcsRecurRulePartByWeekNo.class);
    }

    public ListF<Integer> getMonthList() {
        return getNumberList(IcsRecurRulePartByMonth.class);
    }

    public ListF<Integer> getSetPosList() {
        return getNumberList(IcsRecurRulePartBySetPos.class);
    }

    public ListF<Instant> getXExclude() {
        return getPartByType(IcsRecurRulePartXExclude.class).map(IcsRecurRulePartXExclude::getExcludes).getOrElse(Cf.list());
    }

} //~
