package ru.yandex.calendar.logic.ics.iv5j.support;

import java.io.StringReader;

import net.fortuna.ical4j.data.CalendarParser;
import net.fortuna.ical4j.data.CalendarParserFactory;
import net.fortuna.ical4j.data.ContentHandler;
import net.fortuna.ical4j.util.Strings;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.ComponentsMeta;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsComponent;
import ru.yandex.calendar.logic.ics.iv5j.ical.parameter.IcsParameter;
import ru.yandex.calendar.logic.ics.iv5j.ical.parameter.ParametersMeta;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsProperty;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.PropertiesMeta;
import ru.yandex.calendar.logic.ics.iv5j.vcard.VcfVCard;
import ru.yandex.calendar.logic.ics.iv5j.vcard.parameter.VcfParameter;
import ru.yandex.calendar.logic.ics.iv5j.vcard.parameter.VcfParameters;
import ru.yandex.calendar.logic.ics.iv5j.vcard.property.VcfProperties;
import ru.yandex.calendar.logic.ics.iv5j.vcard.property.VcfProperty;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.lang.Validate;

/**
 * @author dbrylev
 */
public class IvParser {

    public static VcfVCard parseVCard(String input) {
        return parseSingleComponent(input, vCardBuilder, vcfPropertyBuilder, vcfParameterBuilder);
    }

    public static IcsComponent parseSingleIcsComponent(String input) {
        return parseSingleComponent(input, icsComponentBuilder, icsPropertyBuilder, icsParamterBuilder);
    }

    public static IcsProperty parseIcsProperty(String input) {
        return parseIcsProperties(input).single();
    }

    public static ListF<IcsProperty> parseIcsProperties(String input) {
        return parseSingleIcsComponent("BEGIN:C\n" + endWithLineBreak(input) + "END:C").getProperties();
    }

    private static String endWithLineBreak(String string) {
        return string.endsWith("\n") ? string : string + "\n";
    }

    private static final CalendarParser calendarParser = CalendarParserFactory.getInstance().get();

    private static <C, P, T> C parseSingleComponent(
            String input,
            ComponentBuilder<C, P> componentBuilder,
            PropertyBuilder<P, T> propertyBuilder,
            ParameterBuilder<T> parameterBuilder)
    {
        String wrappedInput = "BEGIN:VCALENDAR\n" + endWithLineBreak(input) + "END:VCALENDAR";
        SingleComponentContentHandler<C, P, T> handler = new SingleComponentContentHandler<C, P, T>(
                componentBuilder, propertyBuilder, parameterBuilder);

        try {
            calendarParser.parse(new StringReader(wrappedInput), handler);
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
        return handler.getComponent();
    }

    private static class SingleComponentContentHandler<C, P, T> implements ContentHandler {
        private final ComponentBuilder<C, P> componentBuilder;
        private final PropertyBuilder<P, T> propertyBuilder;
        private final ParameterBuilder<T> parameterBuilder;

        private SingleComponentContentHandler(
                ComponentBuilder<C, P> componentBuilder,
                PropertyBuilder<P, T> propertyBuilder,
                ParameterBuilder<T> parameterBuilder)
        {
            this.componentBuilder = componentBuilder;
            this.propertyBuilder = propertyBuilder;
            this.parameterBuilder = parameterBuilder;
        }

        private Option<String> componentName = Option.empty();
        private ListF<P> componentProperties = Cf.arrayList();

        private Option<String> propertyValue = Option.empty();
        private ListF<T> propertyParameters = Cf.arrayList();

        public C getComponent() {
            return componentBuilder.build(componentName.get(), componentProperties, Cf.<C>list());
        }

        @Override
        public void startCalendar() {}

        @Override
        public void endCalendar() {}

        @Override
        public void startComponent(String name) {
            Validate.none(componentName, "Expected single component");
            componentName = Option.of(name);
        }

        @Override
        public void endComponent(String name) {}

        @Override
        public void startProperty(String name) {
            propertyValue = Option.empty();
            propertyParameters = Cf.arrayList();
        }

        @Override
        public void propertyValue(String value) {
            propertyValue = Option.of(value);
        }

        @Override
        public void endProperty(String name) {
            componentProperties.add(propertyBuilder.build(name, propertyValue.getOrElse(""), propertyParameters));
        }

        @Override
        public void parameter(String name, String value) {
            propertyParameters.add(parameterBuilder.build(name, Strings.unquote(value)));
        }
    }

    private static interface ComponentBuilder<C, P> {
        C build(String name, ListF<P> properties, ListF<C> components);
    }

    private static interface PropertyBuilder<P, T> {
        P build(String name, String value, ListF<T> parameters);
    }

    private static interface ParameterBuilder<T> {
        T build(String name, String value);
    }

    private static final ComponentBuilder<VcfVCard, VcfProperty> vCardBuilder =
            new ComponentBuilder<VcfVCard, VcfProperty>() {
                public VcfVCard build(String name, ListF<VcfProperty> properties, ListF<VcfVCard> components) {
                    Validate.equals("VCARD", name);
                    Validate.isEmpty(components);

                    return new VcfVCard(properties);
                }
            };

    private static final PropertyBuilder<VcfProperty, VcfParameter> vcfPropertyBuilder =
            new PropertyBuilder<VcfProperty, VcfParameter>() {
                public VcfProperty build(String name, String value, ListF<VcfParameter> parameters) {
                    return VcfProperties.P.newProperty(name, value, parameters);
                }
            };

    private static final ParameterBuilder<VcfParameter> vcfParameterBuilder =
            new ParameterBuilder<VcfParameter>() {
                public VcfParameter build(String name, String value) {
                    return VcfParameters.P.newParameter(name, value);
                }
            };

    private static final ComponentBuilder<IcsComponent, IcsProperty> icsComponentBuilder =
            new ComponentBuilder<IcsComponent, IcsProperty>() {
                public IcsComponent build(String name, ListF<IcsProperty> properties, ListF<IcsComponent> components) {
                    return ComponentsMeta.M.newOur(name, properties, components);
                }
            };

    private static final PropertyBuilder<IcsProperty, IcsParameter> icsPropertyBuilder =
            new PropertyBuilder<IcsProperty, IcsParameter>() {
                public IcsProperty build(String name, String value, ListF<IcsParameter> parameters) {
                    return PropertiesMeta.M.newOur(name, value, parameters);
                }
            };

    private static final ParameterBuilder<IcsParameter> icsParamterBuilder =
            new ParameterBuilder<IcsParameter>() {
                public IcsParameter build(String name, String value) {
                    return ParametersMeta.M.newOurParameter(name, value);
                }
            };
}
