package ru.yandex.calendar.logic.layer;

import java.util.List;
import java.util.NoSuchElementException;

import lombok.val;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.dao.IncorrectResultSizeDataAccessException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.generated.EventLayer;
import ru.yandex.calendar.logic.beans.generated.EventLayerHelper;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.LayerFields;
import ru.yandex.calendar.logic.beans.generated.LayerHelper;
import ru.yandex.calendar.logic.beans.generated.LayerUser;
import ru.yandex.calendar.logic.beans.generated.LayerUserFields;
import ru.yandex.calendar.logic.beans.generated.LayerUserHelper;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.util.db.BeanRowMapper;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlQueryUtils;

public class LayerDao extends CalendarJdbcDaoSupport {

    @Autowired
    private GenericBeanDao genericBeanDao;

    /**
     * @see EventDao#findEventById(long)
     */
    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public Layer findLayerById(long layerId) {
        String q = "SELECT * FROM layer WHERE id = ?";
        return getJdbcTemplate().queryForObject(q, Layer.class, layerId);
    }

    @RunWithRandomTest
    public Option<Layer> findLayerByIdSafe(long layerId) {
        String q = "SELECT * FROM layer WHERE id = ?";
        return getJdbcTemplate().queryForOption(q, Layer.class, layerId);
    }

    /**
     * @see EventDao#findEventsByIds(ListF)
     */
    @RunWithRandomTest(possible=IncorrectResultSizeDataAccessException.class)
    public ListF<Layer> findLayersByIds(ListF<Long> ids) {
        SqlCondition c = LayerFields.ID.column().inSet(ids);
        String q = "SELECT * FROM layer WHERE " + c.sql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        MapF<Long, Layer> layersById = getJdbcTemplate().queryForList(q, Layer.class, c.args()).toMapMappingToKey(LayerFields.ID.getF());
        try {
            return ids.map(layersById::getOrThrow);
        } catch (NoSuchElementException e) {
            throw new IncorrectResultSizeDataAccessException("layers not found by some of ids " + ids, ids.size(), layersById.size());
        }
    }

    @RunWithRandomTest
    public Tuple2List<Long, PassportUid> findLayerCreatorUids(ListF<Long> layerIds) {
        SqlCondition c = LayerFields.ID.column().inSet(layerIds);
        String q = "SELECT id, creator_uid FROM layer" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Tuple2List.tuple2List();

        return getJdbcTemplate().query2(
                q, (rs, rn) -> rs.getLong(1), (rs, rn) -> PassportUid.cons(rs.getLong(2)), c.args());
    }

    @RunWithRandomTest
    public ListF<Layer> findLayers(SqlCondition... conditions) {
        SqlCondition c = SqlCondition.all(conditions);
        String q = "SELECT * FROM layer WHERE " + c.sql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, Layer.class, c.args());
    }

    @RunWithRandomTest
    public Option<Layer> findAbsenceLayerByUser(PassportUid yandexUid) {
        String q = "SELECT * FROM layer WHERE creator_uid = ? AND type = ?";
        return getJdbcTemplate().queryForOption(q, Layer.class, yandexUid.getUid(), LayerType.ABSENCE);
    }

    @RunWithRandomTest
    public Option<Long> findServiceLayerIdByUidAndSid(PassportUid uid, PassportSid sid) {
        String sql = "SELECT id FROM layer WHERE creator_uid = ? AND type = ? AND sid = ?";
        return getJdbcTemplate().queryForOption(sql, Long.class, uid, LayerType.SERVICE, sid);
    }

    @RunWithRandomTest
    public int findUserLayerCount(PassportUid creatorUid) {
        String sql = "SELECT COUNT(*) FROM layer WHERE creator_uid = ? AND type = ?";
        return getJdbcTemplate().queryForInt(sql, creatorUid, LayerType.USER);
    }

    @RunWithRandomTest
    public Tuple2List<PassportUid, Option<Long>> findFirstCreatedUserLayerIds(ListF<PassportUid> uids) {
        SqlCondition c = LayerFields.TYPE.eq(LayerType.USER)
                .and(LayerFields.CREATOR_UID.column().inSet(uids));

        String q = "SELECT creator_uid, MIN(id) FROM layer" + c.whereSql() + " GROUP BY creator_uid";
        if (skipQuery(uids, q, c.args())) return Tuple2List.tuple2List();

        return uids.zipWith(getJdbcTemplate().query2(
                q, (rs, rn) -> PassportUid.cons(rs.getLong(1)), (rs, rn) -> rs.getLong(2), c.args()).toMap()::getO);
    }

    @RunWithRandomTest
    public ListF<Layer> findLayersByUid(ListF<PassportUid> uid) {
        String q = "SELECT * FROM layer WHERE creator_uid " + SqlQueryUtils.inSet(uid);

        if (skipQuery(uid, q)) return Cf.list();

        return getJdbcTemplate().queryForList(q, Layer.class);
    }

    @RunWithRandomTest
    public List<Long> findLayerIdsByUid(PassportUid uid) {
        val q = "SELECT id FROM layer WHERE creator_uid = ?";
        return getJdbcTemplate().queryForList(q, Long.class, uid);
    }

    @RunWithRandomTest
    public ListF<Layer> findLayersByLayerUserUid(PassportUid uid) {
        val q = "SELECT l.* FROM layer l INNER JOIN layer_user lu ON l.id = lu.layer_id" +
                " WHERE lu.uid = ?";
        return getJdbcTemplate().queryForList(q, Layer.class, uid);
    }

    @RunWithRandomTest
    public ListF<Layer> findLayersByTypesAndLayerUserUid(PassportUid uid, ListF<LayerType> types) {
        SqlCondition layerCondition = LayerFields.TYPE.column().inSet(types);
        SqlCondition layerUserCondition = LayerUserFields.UID.eq(uid);
        String q = "SELECT l.* FROM layer l" +
                " INNER JOIN layer_user lu ON l.id = lu.layer_id" +
                " WHERE " + layerCondition.sqlForTable("l") + " AND " + layerUserCondition.sqlForTable("lu");

        if (skipQuery(layerCondition.and(layerUserCondition), q, layerCondition.args(), layerUserCondition.args())) {
            return Cf.list();
        }

        return getJdbcTemplate().queryForList(q, Layer.class, layerCondition.args(), layerUserCondition.args());
    }

    @RunWithRandomTest
    public Tuple2List<Layer, EventLayer> findLayersByEventIds(ListF<Long> eventIds) {
        BeanRowMapper<Layer> lRm = LayerHelper.INSTANCE.offsetRowMapper(0);
        BeanRowMapper<EventLayer> elRm = EventLayerHelper.INSTANCE.offsetRowMapper(lRm.nextOffset());

        String q = "SELECT " + lRm.columns("l.") + ", " + elRm.columns("el.") + " FROM layer l" +
                " INNER JOIN event_layer el ON l.id = el.layer_id" +
                " WHERE el.event_id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q)) return Tuple2List.tuple2List();

        return getJdbcTemplate().query2(q, lRm, elRm);
    }

    private Tuple2List<LayerUser, Layer> findLayerUserLayerPairs(SqlCondition layerUserC, SqlCondition layerC) {
        BeanRowMapper<LayerUser> luRm = LayerUserHelper.INSTANCE.offsetRowMapper(0);
        BeanRowMapper<Layer> lRm = LayerHelper.INSTANCE.offsetRowMapper(luRm.nextOffset());

        String q = "SELECT " + luRm.columns("lu.") + ", " + lRm.columns("l.") +
                " FROM layer_user lu " +
                " INNER JOIN layer l ON l.id = lu.layer_id" +
                " WHERE " + layerUserC.sqlForTable("lu") + " AND " + layerC.sqlForTable("l");

        if (skipQuery(layerC.and(layerUserC), q, layerUserC.args(), layerC.args())) return Tuple2List.tuple2List();

        return getJdbcTemplate().query2(q, luRm, lRm, layerUserC.args().plus(layerC.args()));
    }

    @RunWithRandomTest
    public Tuple2List<LayerUser, Layer> findLayerUsersWithRelationsByLayerUserUid(PassportUid layerUserOwner) {
        SqlCondition layerUserC = LayerUserFields.UID.eq(layerUserOwner);
        return findLayerUserLayerPairs(layerUserC, SqlCondition.trueCondition());
    }

    @RunWithRandomTest
    public Tuple2List<LayerUser, Layer> findLayerUsersWithRelationsByLayerIds(ListF<Long> ids) {
        SqlCondition layerC = LayerFields.ID.column().inSet(ids);
        return findLayerUserLayerPairs(SqlCondition.trueCondition(), layerC);
    }

    @RunWithRandomTest
    public ListF<PassportUid> findLayerUserUidsByLayerIdsModifiedSince(ListF<Long> layerIds, Instant since) {
        SqlCondition layerC = LayerFields.ID.column().inSet(layerIds)
                .and(LayerFields.COLL_LAST_UPDATE_TS.ge(since));

        String q = "SELECT DISTINCT uid FROM layer_user lu, layer l" +
                " WHERE lu.layer_id = l.id" + layerC.andSqlForTable("l");

        if (skipQuery(layerC, q, layerC.args())) return Cf.list();

        return getJdbcTemplate().query(q, (rs, rn) -> PassportUid.cons(rs.getLong(1)), layerC.args());
    }

    @RunWithRandomTest
    public Tuple2List<LayerUser, Layer> findLayerUsersWithRelationsByLayerUserUids(ListF<PassportUid> layerUserOwners) {
        SqlCondition layerUserC = LayerUserFields.UID.column().inSet(layerUserOwners);
        return findLayerUserLayerPairs(layerUserC, SqlCondition.trueCondition());
    }

    @RunWithRandomTest
    public Option<Tuple2<LayerUser, Layer>> findLayerUserWithRelations(PassportUid uid, long layerId) {
        return findLayerUserLayerPairs(LayerUserFields.UID.eq(uid), LayerFields.ID.eq(layerId)).singleO();
    }

    @RunWithRandomTest
    public ListF<Layer> findLayersWithoutLayerUsers() {
        String q =
            "SELECT l.* FROM layer l " +
            "WHERE NOT EXISTS (SELECT lu.* FROM layer_user lu WHERE lu.layer_id = l.id)";
        return getJdbcTemplate().queryForList(q, Layer.class);
    }

    public void updateLayerEventsClosedByDefault(long layerId, boolean isClosed) {
        String q = "UPDATE layer SET is_events_closed_by_default = ? WHERE id = ?";
        getJdbcTemplate().updateRow(q, isClosed, layerId);
    }

    public long saveLayer(Layer layer) {
        return genericBeanDao.insertBeanGetGeneratedKey(layer);
    }

    public void updateLayer(Layer layer) {
        genericBeanDao.updateBean(layer);
    }

    /**
     * Attention! coll_last_update_ts MUST be updated after any event update, create or delete,
     * otherwise CalDAV won't flush event cache.
     *
     * @see EventDbManager#updateEvent(ru.yandex.calendar.logic.beans.generated.Event)
     */
    @RunWithRandomTest
    public void updateLayersCollLastUpdateTsByEventIds(Instant now, ListF<Long> eventIds) {
        String q = "UPDATE layer l SET coll_last_update_ts = ?" +
                " FROM event_layer el" +
                " WHERE l.id = el.layer_id AND el.event_id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q, now)) return;

        getJdbcTemplate().update(q, now);
    }

    @RunWithRandomTest
    public void updateLayerCollLastUpdateTsByLayerId(Instant now, long layerId) {
        String q = "UPDATE layer SET coll_last_update_ts = ? WHERE id = ?";
        getJdbcTemplate().update(q, now, layerId);
    }

    @RunWithRandomTest
    public void updateLayerCollLastUpdateTsByLayerIds(Instant now, ListF<Long> layerIds) {
        String q = "UPDATE layer SET coll_last_update_ts = ? WHERE id " + SqlQueryUtils.inSet(layerIds);

        if (skipQuery(layerIds, q, now)) return;

        getJdbcTemplate().update(q, now);
    }

    @RunWithRandomTest
    public ListF<Long> updateStaleCollLastUpdateTsByMainEventIds(ListF<Long> mainEventIds, Instant ts) {
        String q = "UPDATE layer SET coll_last_update_ts = ?"
                + " FROM (SELECT l.id AS lid FROM layer l, event_layer el, event e"
                + " WHERE l.id = el.layer_id AND e.id = el.event_id"
                + " AND coll_last_update_ts < ?"
                + " AND e.main_event_id " + SqlQueryUtils.inSet(mainEventIds)
                + " ORDER BY l.id FOR NO KEY UPDATE OF l) x"
                + " WHERE id = lid AND coll_last_update_ts < ?"
                + " RETURNING id";

        if (skipQuery(mainEventIds, q, ts, ts, ts)) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, ts, ts, ts);
    }

    @RunWithRandomTest
    public ListF<Long> updateStaleCollLastUpdateTsByMainEventIdsUsingEventLayersDeletedAt(
            ListF<Long> mainEventIds, Instant ts)
    {
        String q = "UPDATE layer SET coll_last_update_ts = ?"
                + " FROM (SELECT l.id AS lid FROM layer l, deleted_event_layer el, event e"
                + " WHERE l.id = el.layer_id AND e.id = el.event_id"
                + " AND el.deletion_ts = ?"
                + " AND coll_last_update_ts < ?"
                + " AND e.main_event_id " + SqlQueryUtils.inSet(mainEventIds)
                + " ORDER BY l.id FOR NO KEY UPDATE OF l) x"
                + " WHERE id = lid AND coll_last_update_ts < ?"
                + " RETURNING id";

        if (skipQuery(mainEventIds, q, ts, ts, ts, ts)) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, ts, ts, ts, ts);
    }

    @RunWithRandomTest(possible=DataIntegrityViolationException.class)
    public void deleteLayerById(ListF<Long> id) {
        String q = "DELETE FROM layer WHERE id " + SqlQueryUtils.inSet(id);

        if (skipQuery(id, q)) return;

        getJdbcTemplate().update(q);
    }

    @RunWithRandomTest
    public Option<Layer> findLayerByPrivateToken(String privateToken) {
        String q = "SELECT * FROM layer WHERE private_token = ?";
        return getJdbcTemplate().queryForOption(q, Layer.class, privateToken);
    }

    @RunWithRandomTest
    public void updateLayerSetCreatorUidById(long layerId, PassportUid newCreator, ActionInfo actionInfo) {
        val q = "UPDATE layer SET creator_uid = ?, last_update_ts = ? WHERE id = ?";
        getJdbcTemplate().update(q, newCreator, actionInfo.getNow(), layerId);
    }
}
