package ru.yandex.calendar.logic.layer;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.LayerInvitation;
import ru.yandex.calendar.logic.contact.ContactRoutines;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActorId;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.sending.LayerSendingInfo;
import ru.yandex.calendar.logic.sending.param.LayerInvitationMessageParameters;
import ru.yandex.calendar.logic.sending.param.Sender;
import ru.yandex.calendar.logic.sending.real.MailSender;
import ru.yandex.calendar.logic.sending.so.SoChecker;
import ru.yandex.calendar.logic.sharing.Nick;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.sharing.perm.LayerActionClass;
import ru.yandex.calendar.logic.sharing.perm.LayerInfoForPermsCheck;
import ru.yandex.calendar.logic.svc.SvcRoutines;
import ru.yandex.calendar.logic.user.KarmaCheckAction;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.micro.perm.LayerAction;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.base.UidGen;
import ru.yandex.calendar.util.exception.ExceptionUtils;
import ru.yandex.commune.mail.MailAddress;
import ru.yandex.inside.passport.AbstractPassportUid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

import static java.util.Objects.requireNonNull;

@Slf4j
public class LayerInvitationManager {
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private SvcRoutines svcRoutines;
    @Autowired
    private MailSender mailSender;
    @Autowired
    private LayerInvitationDao layerInvitationDao;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private ContactRoutines contactRoutines;
    @Autowired
    private LayerUserDao layerUserDao;
    @Autowired
    private UserManager userManager;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private SoChecker soChecker;

    public ListF<LayerInvitationMessageParameters> createLayerInvitationMails(
            PassportUid senderUid, ListF<LayerSendingInfo> sendingInfoList)
    {
        Sender sender = eventInvitationManager.getSender(ActorId.user(senderUid));

        return sendingInfoList.filterMap(a -> {
            try {
                return Option.of(createInvitationMailXmlForLayer(sender, a));
            } catch (Exception e) {
                handleInvitationMailError(a.getInvitation().getPrivateToken().getOrNull(), e);
                return Option.empty();
            }
        });
    }

    public void sendLayerInvitationMails(ListF<LayerInvitationMessageParameters> layerInvitationMails, ActionInfo actionInfo) {
        soChecker.checkNoSpamLayers(layerInvitationMails, actionInfo);

        mailSender.sendEmailsViaTask(layerInvitationMails, actionInfo);
    }

    private void handleInvitationMailError(@Nullable String privateToken, Exception e) {
        log.error("Error occurred while creating or sending layer invitation mail", e);
        ExceptionUtils.rethrowIfTlt(e);
        try {
            if (StringUtils.isNotEmpty(privateToken)) {
                layerInvitationDao.deleteLayerInvitationByPrivateToken(privateToken);
            }
        } catch (Exception e2) {
            log.error("Error occurred while trying to rollback invitation insert", e2);
            ExceptionUtils.rethrowIfTlt(e2);
        }
    }

    private LayerInvitationMessageParameters createInvitationMailXmlForLayer(Sender sender, LayerSendingInfo layerSendingInfo) {
        final LayerInvitation invitation = layerSendingInfo.getInvitation();
        Validate.some(invitation.getPrivateToken());

        long layerId = invitation.getLayerId();
        Layer layer = layerRoutines.getLayerById(layerId);

        MailAddress recipient = new MailAddress(invitation.getEmail(), Nick.getRecipient(invitation.getName()));
        Email fromEmail = svcRoutines.getCalendarInfoEmail(PassportAuthDomain.byEmail(sender.getEmail()));

        Option<PassportUid> recipientUid = invitation.getUid();

        Language language = settingsRoutines.chooseMailLanguage(sender.getUid(), recipientUid);

        final String calendarUrl;
        if (recipientUid.isPresent()) {
            calendarUrl = svcRoutines.getCalendarUrlForUid(recipientUid.get());
        } else {
            calendarUrl = svcRoutines.getCalendarUrlForDomain(PassportAuthDomain.byUid(layer.getCreatorUid()), language);
        }

        return new LayerInvitationMessageParameters(
                language,
                layerRoutines.evalLayerName(layer, Option.empty()), layer.getId(), invitation.getPerm(),
                sender, recipient, fromEmail, recipientUid,
                calendarUrl, invitation.getPrivateToken().getOrNull(), layerSendingInfo.isAutoAccept());
    }

    public void removeLayerInvitation(
            PassportUid clientUid, LayerInvitation inv, ActionInfo actionInfo)
    {
        boolean isShared = false;
        if (inv.getUid().isPresent()) {
            isShared = layerUserDao
                    .findLayerUserByLayerIdAndUid(inv.getLayerId(), inv.getUid().get())
                    .isPresent();
        }

        if (isShared) {
            layerRoutines.detach(clientUid, inv.getUid().get(), inv.getLayerId(), actionInfo);
        } else {
            layerInvitationDao.deleteLayerInvitationById(LayerInvitationId.of(inv));
        }
    }

    private Option<LayerSendingInfo> createLayerInvitationSendingInfo(LayerInvitation invitation) {
        long layerId = invitation.getLayerId();
        Email guestEmail = invitation.getEmail();
        Option<Long> resourceIdO = resourceRoutines.parseIdFromEmail(guestEmail);

        if (resourceIdO.isPresent()) {
            throw new IllegalStateException();
        }

        Option<PassportUid> subjectIdO = invitation.getUid();

        final boolean autoAccept = subjectIdO.exists(AbstractPassportUid::isYandexTeamRu)
                || subjectIdO.exists(settingsRoutines::isAutoAcceptInvitations);

        if (autoAccept) {
            PassportUid subjectId = subjectIdO.get();
            val layerActionClass = invitation.getPerm();
            layerRoutines.startNewSharing(subjectId, layerId, layerActionClass);
        }

        LayerSendingInfo eventSendingInfo = new LayerSendingInfo(invitation, autoAccept);
        return resourceIdO.isPresent() ? Option.empty() : Option.of(eventSendingInfo);
    }

    private void createAndSendLayerInvitationMails(
            PassportUid uid, ListF<LayerInvitation> newInvitations, ActionInfo actionInfo)
    {
        ListF<LayerSendingInfo> sendingInfos = Cf.arrayList();
        if (newInvitations.isNotEmpty()) {
            for (LayerInvitation inv : newInvitations) {
                sendingInfos.addAll(createLayerInvitationSendingInfo(inv));
            }
        }
        ListF<LayerInvitationMessageParameters> mails = createLayerInvitationMails(uid, sendingInfos);
        sendLayerInvitationMails(mails, actionInfo);
    }

    public void updateLayerSharing(UserInfo userInfo, long layerId, Map<Email, LayerActionClass> nowPerms,
                                   boolean isEventsClosedByDefault, ActionInfo actionInfo) {
        val uid = userInfo.getUid();
        val layer = layerRoutines.getLayerById(layerId);
        val layerPermInfo = LayerInfoForPermsCheck.fromLayer(layer);
        authorizer.ensureCanPerformLayerActionType(userInfo, layerPermInfo, LayerAction.GRANT, actionInfo.getActionSource(),
            LayerType.USER);

        LayerInvitationChangesInfo changes = layerChangesAndStoreContacts(uid, layerId, nowPerms, actionInfo);

        ListF<PassportUid> nonInvitedUids = layerUserDao.findNonInvitedLayerUsers(layerId);
        Tuple2<ListF<LayerInvitation>, ListF<LayerInvitation>> p = changes.getNewInvitations()
                .partition(LayerInvitation.getUidF().andThen(Cf2.isSomeOfF(nonInvitedUids)));

        ListF<LayerInvitation> newInvitations = p._2;
        ListF<LayerInvitation> missedInvitations = p._1;

        if (newInvitations.isNotEmpty()) {
            userManager.checkPublicUserKarma(KarmaCheckAction.inviteLayer(
                    uid, layerRoutines.evalLayerSName(layerRoutines.getLayerById(layerId)), actionInfo));
        }

        layerInvitationDao.saveLayerInvitations(newInvitations.plus(missedInvitations));
        layerInvitationDao.updateLayerInvitations(changes.getUpdatedInvitations());
        layerInvitationDao.deleteLayerInvitationsByIds(changes.getRemovedInvitations().map(LayerInvitationId::of));

        for (LayerInvitation newYt : newInvitations.filter(i -> i.getUid().exists(PassportUid::isYandexTeamRu))) {
            layerRoutines.startNewSharing(newYt.getUid().get(), layerId, requireNonNull(nowPerms.get(newYt.getEmail())));
        }

        for (LayerInvitation updated : changes.getUpdatedInvitations().plus(missedInvitations)) {
            // XXX email about changed permissions?
            updated.getUid().ifPresent(passportUid -> {
                val perm = requireNonNull(nowPerms.get(updated.getEmail()));
                layerRoutines.updateLayerUserPerm(passportUid, updated.getLayerId(), perm);
            });
        }

        layerRoutines.detach(uid, changes.getRemovedInvitationUids(), layerId, actionInfo);

        layerRoutines.updateLayerEventsClosedByDefault(layerId, isEventsClosedByDefault);
        createAndSendLayerInvitationMails(uid, newInvitations, actionInfo);
    }

    public void createLayerInvitationIfAbsentForUser(
            long layerId, PassportUid uid, LayerActionClass perm, ActionInfo actionInfo)
    {
        if (!layerInvitationDao.findLayerInvitationExistsByLayerIdAndUid(layerId, uid)) {
            PassportUid layerCreatorUid = layerRoutines.getCreatorUid(layerId);
            Email userEmail = userManager.getEmailByUid(uid).get();

            Option<LayerInvitation> invitation = layerInvitationDao.findInvitationByLayerIdAndEmail(layerId, userEmail);
            if (invitation.isPresent()) {
                LayerInvitation data = new LayerInvitation();
                data.setId(invitation.get().getId());
                data.setUid(uid);
                data.setModificationReqId(actionInfo.getRequestIdWithHostId());
                data.setModificationSource(actionInfo.getActionSource());

                layerInvitationDao.updateLayerInvitation(data);
            } else {
                layerInvitationDao.saveLayerInvitations(createInvitationCreateDatas(
                        layerCreatorUid, layerId, Cf.list(userEmail), Map.of(userEmail, perm), Cf.set(), actionInfo));
            }
        }
    }

    /**
     * Creates and fills invitation changes info structure
     * @param uid user who invites
     */
    private LayerInvitationChangesInfo layerChangesAndStoreContacts(
            PassportUid uid, long layerId, Map<Email, LayerActionClass> nowPerms, ActionInfo actionInfo) {
        ListF<LayerInvitation> existingInvitations = layerInvitationDao.findLayerInvitationByLayerId(layerId);
        ListF<Email> existingInvitationsEmails = existingInvitations.map(LayerInvitation.getEmailF());

        val newEmails = StreamEx.of(nowPerms.keySet())
                .remove(existingInvitationsEmails::contains)
                .toImmutableList();
        SetF<Email> remEmails = existingInvitationsEmails.unique().minus(nowPerms.keySet());

        contactRoutines.exportUserContactsEmails(uid, Cf.toList(newEmails));

        ListF<LayerInvitation> newInvitations =
                createInvitationCreateDatas(uid, layerId, newEmails, nowPerms,
                        existingInvitations.filterMap(LayerInvitation::getUid).unique(), actionInfo);

        Tuple2<ListF<LayerInvitation>, ListF<LayerInvitation>> p =
                existingInvitations.partition(LayerInvitation.getEmailF().andThen(remEmails.containsF()));

        ListF<LayerInvitation> removedInvitations = p._1;
        ListF<LayerInvitation> survivedInvitations = p._2;

        ListF<LayerInvitation> updatedInvitations =
                createInvitationUpdateDatasIfChanged(survivedInvitations, nowPerms, actionInfo);

        Function1B<LayerInvitation> notLayerCreatorInvitationF = LayerInvitation.getUidF()
                .andThen(Cf2.isSomeF(layerRoutines.getCreatorUid(layerId)).notF());

        return new LayerInvitationChangesInfo(
                newInvitations.filter(notLayerCreatorInvitationF),
                updatedInvitations.filter(notLayerCreatorInvitationF),
                removedInvitations.filter(notLayerCreatorInvitationF));
    }

    private ListF<LayerInvitation> createInvitationCreateDatas(PassportUid uid, long layerId, List<Email> emails,
                                                               Map<Email, LayerActionClass> permByEmail,
                                                               SetF<PassportUid> filteringUids, ActionInfo actionInfo) {
        val uidByEmail = userManager.getUidsByEmails(Cf.toList(emails)).toMap();
        val nameByEmail = contactRoutines.getNamesByEmails(uid, Cf.toList(emails)).toMap();

        ListF<LayerInvitation> invitations = Cf.arrayList();
        for (Email email : emails) {
            LayerInvitation invitation = new LayerInvitation();
            invitation.setLayerId(layerId);
            invitation.setEmail(email);
            invitation.setUid(uidByEmail.getOrThrow(email).filterNot(filteringUids::containsTs)); // XXX drop db unique key
            invitation.setName(nameByEmail.getOrThrow(email).getOrElse(""));
            invitation.setPerm(requireNonNull(permByEmail.get(email)));

            invitation.setPrivateToken(UidGen.createPrivateToken());
            invitation.setCreationTs(actionInfo.getNow());
            invitation.setCreationReqId(actionInfo.getRequestIdWithHostId());
            invitation.setCreationSource(actionInfo.getActionSource());
            invitation.setCreatorUid(uid);

            invitations.add(invitation);
        }
        return invitations;
    }

    private ListF<LayerInvitation> createInvitationUpdateDatasIfChanged(
            List<LayerInvitation> existing, Map<Email, LayerActionClass> permByEmail, ActionInfo actionInfo) {
        ListF<LayerInvitation> updates = Cf.arrayList();
        for (LayerInvitation invitation : existing) {
            if (!invitation.getPerm().equals(requireNonNull(permByEmail.get(invitation.getEmail())))) {
                LayerInvitation update = invitation.copy();

                update.setPerm(requireNonNull(permByEmail.get(invitation.getEmail())));
                update.setModificationReqId(actionInfo.getRequestIdWithHostId());
                update.setModificationSource(actionInfo.getActionSource());

                updates.add(update);
            }
        }
        return updates;
    }

    public LayerInvitation getLayerInvitationByPrivateToken(String privateToken, Option<PassportUid> uid) {
        Option<LayerInvitation> r = layerInvitationDao.findInvitationByPrivateToken(privateToken);
        if (!r.isPresent()) {
            throw CommandRunException.createSituation(
                    "layer invitation not found by private token: " + privateToken, Situation.INV_IS_MISSING);
        }
        EventInvitationManager.checkInvUid(r.get().getUid(), uid);
        return r.get();
    }

    public ListF<LayerInvitation> findLayerInvitations(long layerId) {
        return layerInvitationDao.findLayerInvitationByLayerId(layerId);
    }

    public MapF<Long, Integer> findLayerInvitationsCounts(ListF<Long> layerIds) {
        return layerInvitationDao.findLayerInvitationsCountByLayerIds(layerIds);
    }
}
