package ru.yandex.calendar.logic.layer;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.Lazy;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function2;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.frontend.web.cmd.RequestDataConverter;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.PermissionDeniedUserException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.logic.LastUpdateManager;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.LayerFields;
import ru.yandex.calendar.logic.beans.generated.LayerUser;
import ru.yandex.calendar.logic.beans.generated.LayerUserFields;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.beans.generated.SettingsFields;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventInfoDbLoader;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventLoadLimits;
import ru.yandex.calendar.logic.event.EventOrLayerRoutines;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventUserRoutines;
import ru.yandex.calendar.logic.event.avail.absence.AbsenceType;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.EventLayerDao;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.InfiniteInterval;
import ru.yandex.calendar.logic.ics.feed.IcsFeedDao;
import ru.yandex.calendar.logic.ics.feed.IcsFeedManager;
import ru.yandex.calendar.logic.notification.ControlDataNotification;
import ru.yandex.calendar.logic.notification.EventNotificationChangesInfo;
import ru.yandex.calendar.logic.notification.EventNotifications;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.notification.NotificationChangesFinder;
import ru.yandex.calendar.logic.notification.NotificationDbManager;
import ru.yandex.calendar.logic.notification.NotificationRoutines;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.sending.param.EventMessageParameters;
import ru.yandex.calendar.logic.sharing.InvitationProcessingMode;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.sharing.perm.LayerActionClass;
import ru.yandex.calendar.logic.sharing.perm.LayerInfoForPermsCheck;
import ru.yandex.calendar.logic.svc.DbSvcRoutines;
import ru.yandex.calendar.logic.svc.Service;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.micro.perm.LayerAction;
import ru.yandex.calendar.util.base.Binary;
import ru.yandex.calendar.util.base.UidGen;
import ru.yandex.calendar.util.color.Color;
import ru.yandex.calendar.util.color.ColorUtils;
import ru.yandex.calendar.util.data.DataProvider;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.calendar.util.idlent.YandexUser;
import ru.yandex.calendar.util.resources.UStringLiteral;
import ru.yandex.inside.passport.PassportAuthDomain;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.cache.Cache;
import ru.yandex.misc.cache.tl.TlCache;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

@Slf4j
public class LayerRoutines extends EventOrLayerRoutines<Layer, LayerUser> {
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private NotificationRoutines notificationRoutines;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private LayerDao layerDao;
    @Autowired
    private LayerUserDao layerUserDao;
    @Autowired
    private EventLayerDao eventLayerDao;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private LayerInvitationDao layerInvitationDao;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EventUserRoutines eventUserRoutines;
    @Autowired
    private LayerDbManager layerDbManager;
    @Autowired
    private IcsFeedManager icsFeedManager;
    @Autowired
    private IcsFeedDao icsFeedDao;
    @Autowired
    private EventInfoDbLoader eventInfoDbLoader;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private LastUpdateManager lastUpdateManager;
    @Autowired
    private NotificationDbManager notificationDbManager;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private UserManager userManager;

    //    private static final int MAX_CUSTOM_LAYERS_PER_USER = 5; // temp (due to ics)!
    static final int MAX_NAME_LEN = 100;
    static final int MAX_CSS_CLASS_LEN = 40;

    public static final NameI18n DEFAULT_USER_LAYER_NAME = new NameI18n(
            UStringLiteral.MY_EVENTS, UStringLiteral.MY_EVENTS_EN);

    private static final String DEFAULT_LAYER_CSS_CLASS = "calendar";
    private static final String DEFAULT_LAYER_HEAD_BG_COLOR = "#4887e1";
    private static final String DEFAULT_LAYER_HEAD_FG_COLOR = "#FFFFFF";
    private static final String DEFAULT_LAYER_BODY_BG_COLOR = "#FBFBED";

    private static final String DEFAULT_LAYER_CSS_CLASS_PUBLIC = "custom";
    private static final String DEFAULT_LAYER_HEAD_BG_COLOR_PUBLIC = "#4887e1";
    private static final String DEFAULT_LAYER_HEAD_FG_COLOR_PUBLIC = "#FFFFFF";
    private static final String DEFAULT_LAYER_BODY_BG_COLOR_PUBLIC = "#FDFFD7";

    /**
     * Ensures that service layer for given uid and service id exists.
     * NOTE: caller should take care of SID existence itself
     * (e.g. through {@link DbSvcRoutines#exists} ).
     * @param sid service id
     * @return found or created layer id
     */
    public long getOrCreateServiceLayer(PassportUid uid, PassportSid sid) {
        Option<Long> res = layerDao.findServiceLayerIdByUidAndSid(uid, sid);
        if (!res.isPresent()) {
            Layer layer = new Layer();
            layer.setSid(sid);

            LayerUser layerUserOverrides = new LayerUser();
            return createLayerWithLayerUser(uid, LayerType.SERVICE, layer, layerUserOverrides,
                    notificationRoutines.getDefaultNotifications());
        } else {
            return res.get();
        }
    }

    public long getOrCreateAbsenceLayer(PassportUid uid, AbsenceType type) {
        Option<Layer> res = layerDao.findLayers(
                LayerFields.CREATOR_UID.eq(uid).and(LayerFields.TYPE.eq(LayerType.ABSENCE))).singleO();

        if (!res.isPresent()) {
            Layer layer = new Layer();
            layer.setName(type.getLayerName());

            LayerUser layerUser = createDefaultLayerUserOverrides(uid.getDomain());

            return createLayerWithLayerUser(uid, LayerType.ABSENCE, layer, layerUser, Cf.list());
        } else {
            return res.get().getId();
        }
    }

    public Option<Long> getAbsenceLayerIdIfPresent(PassportUid uid, AbsenceType type) {
        Option<Layer> res = layerDao.findLayers(
                LayerFields.CREATOR_UID.eq(uid).and(LayerFields.TYPE.eq(LayerType.ABSENCE))
        ).singleO();
        return res.map(Layer::getId);
    }

    /**
     * Looks up for the first user layer, optionally bounded with a given name
     * and returns its id (or null if nothing is found)
     */
    // XXX: make private, should use getDefaultLayerId
    public Option<Long> getFirstUserLayerId(PassportUid creatorUid, Option<String> layerNameO) {
        // Check if at least 1 user layer (maybe bounded by given name) already exists

        SqlCondition layerNameCondition = layerNameO.isPresent() ?
                LayerFields.NAME.eq(layerNameO.get()) :
                SqlCondition.trueCondition(); // XXX ssytnik: see comment below
        // XXX ssytnik: every call of this method with empty 'layerNameO' given
        // would like that we first check if there is a default layer value set
        // in the 'settings.layerId' first and, if so, immediately return it with
        // 'getDefaultLayerIdFromSettings(Option.some(creatorUid))'.
        // (NOTE: not 'getDefaultLayerId(uid)' in order to prevent us from loop).

        SqlCondition c = SqlCondition.all(
            LayerFields.CREATOR_UID.eq(creatorUid),
            LayerFields.TYPE.eq(LayerType.USER),
            layerNameCondition
        );
        String sql = "SELECT id FROM layer WHERE " + c.sql() + " ORDER BY creation_ts LIMIT 1";
        return getJdbcTemplate().queryForOption(sql, Long.class, c.args());
    }

    public long createUserLayer(PassportUid uid, Layer layerOverrides, LayerUser layerUserOverrides) {
        return createUserLayer(uid, notificationRoutines.getDefaultNotifications(),
                layerOverrides, false, layerUserOverrides);
    }

    /**
     * Creates new USER layer with its default values, optionally overridden by given layer name
     */
    public long createUserLayer(PassportUid uid, Option<String> layerNameO) {
        Layer layerOverrides = new Layer();
        layerOverrides.setName(layerNameO.getOrNull());

        LayerUser layerUserOverrides = createDefaultLayerUserOverrides(uid.getDomain());

        return createUserLayer(uid, notificationRoutines.getDefaultNotifications(),
                layerOverrides, false, layerUserOverrides);
    }

    public long createUserLayer(PassportUid uid) {
        return createUserLayer(uid, Option.empty());
    }

    private long createAbsenceLayer(PassportUid uid) {
        Layer layer = new Layer();
        layer.setName("Отсутствия");

        // XXX: create default absence layer parameters
        LayerUser layerUserOverrides = createDefaultLayerUserOverrides(uid.getDomain());
        return createLayerWithLayerUser(uid, LayerType.ABSENCE, layer, layerUserOverrides, Cf.list());
    }

    public long getOrCreateAndGetAbsenceLayerId(PassportUid uid) {
        Validate.isTrue(uid.isYandexTeamRu());
        Option<Layer> layerO = layerDao.findAbsenceLayerByUser(uid);
        if (layerO.isPresent()) {
            return layerO.get().getId();
        } else {
            return createAbsenceLayer(uid);
        }
    }

    /**
     * Create USER layer with given overrides (including notification), and takes care of is-default flag
     */
    public long createUserLayer(
            PassportUid uid, ListF<Notification> notifications, Layer layerOverrides,
            boolean isLayerDefault, LayerUser layerUserOverrides)
    {
        long layerId = createLayerWithLayerUser(uid, LayerType.USER, layerOverrides, layerUserOverrides, notifications);
        updateDefaultLayer(uid, layerId, isLayerDefault);
        return layerId;
    }

    /**
     * Creates FEED layer for user with given name, and layer-user (with empty notification unless specified)
     */
    public long createFeedLayer(PassportUid uid, String layerName) {
        Layer layer = new Layer();
        layer.setName(layerName);

        LayerUser layerUserOverrides = createDefaultLayerUserOverrides(uid.getDomain());
        layerUserOverrides.setAffectsAvailability(!uid.isYandexTeamRu());

        if (!uid.isYandexTeamRu()) settingsRoutines.createSettingsIfNotExistsForUids(Cf.list(uid));

        return createLayerWithLayerUser(
                uid, LayerType.FEED, layer, layerUserOverrides, notificationRoutines.getDefaultNotifications());
    }

    /**
     * Creates layer and (optionally) a single creator layer-user (with empty notification unless specified)
     * @param uid layer creator
     * @param layerType layer type
     * @param layerOverrides fields required: name; ignored: creator_uid, creation_ts, type, last_data_modify_ts
     * @param layerUserOverrides if defined: fields ignored: uid, layer_id, notification_id
     * @return layer id
     */
    public long createLayerWithLayerUser(PassportUid uid, LayerType layerType,
            Layer layerOverrides, LayerUser layerUserOverrides, ListF<Notification> notifications)
    {
        long layerId = createLayer(uid, layerType, layerOverrides);
        layerUserOverrides.setPerm(LayerActionClass.ADMIN);
        createLayerUserForUserAndLayer(uid, layerId, layerUserOverrides, notifications);
        return layerId;
    }

    public long createLayer(PassportUid uid, LayerType type, Layer layerOverrides) {
        Layer layer = new Layer();
        layer.setFields(layerOverrides);
        layer.setCreatorUid(uid);
        layer.setCreationTs(new Instant());
        layer.setType(type);
        layer.setLastUpdateTs(new Instant());
        layer.setCollLastUpdateTs(new Instant());

        return layerDao.saveLayer(layer);
    }

    public long createLayerUserForUserAndLayer(PassportUid uid, long layerId,
                                               LayerUser layerUserOverrides, ListF<Notification> notifications) {
        val layerUser = new LayerUser();
        layerUser.setFields(layerUserOverrides);
        layerUser.setUid(uid);
        layerUser.setLayerId(layerId);
        layerUser.setFieldDefaults(createDefaultLayerUserOverrides(uid.getDomain()));
        val layerUserId = layerUserDao.saveLayerUser(layerUser);
        notificationDbManager.saveLayerNotifications(layerUserId, notifications);
        return layerUserId;
    }

    public LayerUser createDefaultLayerUserOverrides(PassportAuthDomain domain) {
        LayerUser layerUser = new LayerUser();

        if (domain == PassportAuthDomain.YANDEX_TEAM_RU) {
            layerUser.setCssClass(DEFAULT_LAYER_CSS_CLASS);

            layerUser.setHeadBgColor(ColorUtils.unformatColor(DEFAULT_LAYER_HEAD_BG_COLOR));
            layerUser.setHeadFgColor(ColorUtils.unformatColor(DEFAULT_LAYER_HEAD_FG_COLOR));
            layerUser.setBodyBgColor(ColorUtils.unformatColor(DEFAULT_LAYER_BODY_BG_COLOR));
        } else {
            layerUser.setCssClass(DEFAULT_LAYER_CSS_CLASS_PUBLIC);

            layerUser.setHeadBgColor(ColorUtils.unformatColor(DEFAULT_LAYER_HEAD_BG_COLOR_PUBLIC));
            layerUser.setHeadFgColor(ColorUtils.unformatColor(DEFAULT_LAYER_HEAD_FG_COLOR_PUBLIC));
            layerUser.setBodyBgColor(ColorUtils.unformatColor(DEFAULT_LAYER_BODY_BG_COLOR_PUBLIC));
        }
        layerUser.setAffectsAvailability(true);
        return layerUser;
    }

    public void updateServiceLayer(PassportUid uid, DataProvider dp, ActionInfo actionInfo) {
        DataProvider layerDataProvider = dp.getDataProvider("layer", true);
        DataProvider layerUserDataProvider = dp.getDataProvider("user", true);
        PassportSid sid = PassportSid.cons(Integer.parseInt(layerDataProvider.getText("sid", true)));
        long layerId = getOrCreateServiceLayer(uid, sid);
        boolean applyToEvents = Binary.parseBoolean(dp.getText("apply-to-events", false));
        val layer = getLayerById(layerId);
        authorizer.ensureLayerType(LayerInfoForPermsCheck.fromLayer(layer), LayerType.SERVICE);
        updateLayerNotification(actionInfo, uid, layer, layerUserDataProvider, applyToEvents);
    }

    // Updates existing layer.
    // For SERVICE layers, only notification is updated. Requires: l=1, l_id, u=1, u_*.
    // For all other layers, also name, css/colors, default layer can be updated.
    // NOTE: to edit layer properly, user has to have LayerUser for layer being edited.
    public void updateLayer(UserInfo user, DataProvider dp, ActionInfo actionInfo) {
        Validate.notNull(dp, "layer data is missing");
        PassportUid uid = user.getUid();
        // Get data and do major checks
        DataProvider layerDataProvider = dp.getDataProvider("layer", true); // lDp: id, is-default
        long layerId = Long.parseLong(layerDataProvider.getText("id", true));
        DataProvider layerUserDataProvider = dp.getDataProvider("user", true); // luDp: css-class, colors, process-ntf + <ntf>

        val layer = getLayerById(layerId);
        authorizer.ensureLayerType(LayerInfoForPermsCheck.fromLayer(layer), LayerType.USER, LayerType.FEED, LayerType.SERVICE); // XXX: resource?

        // Update for USER, FEED layer types
        if (!isService(layer)) {
            Option<Long> feedIdO = Cf.Long.parseSafe(layerDataProvider.getText("feed-id", false));
            if (feedIdO.isPresent()) {
                String feedUrl = layerDataProvider.getText("feed-url", true);
                icsFeedManager.updateFeedUrl(uid, feedIdO.get(), feedUrl);
            }

            // Update layer (only name of layer can be changed)
            Layer layerData = LayerRequestDataConverter.convertLayerFromDataProvider(layerDataProvider);
            layerData.setId(layerId);
            layerData = keepNullNameOnUpdate(layerData, user.getUid());

            // Check permission if layer name was changed
            if (layerData.getName().isPresent() && !layerData.getName().equals(layer.getName())) {
                val layerPermInfo = LayerInfoForPermsCheck.fromLayer(layer);
                authorizer.ensureCanPerformLayerAction(user, layerPermInfo, Optional.empty(), LayerAction.EDIT, actionInfo.getActionSource());
                layerDao.updateLayer(layerData);
            }
            // Handle 'is-default'
            // ssytnik: no permission check (even LayerAction.CREATE_EVENT): for now, let user be aware of it
            updateDefaultLayer(uid, layerId, Binary.parseBoolean(layerDataProvider.getText("is-default", false)));
            // Update layer-user css class / colors
            // ssytnik: no permission check is needed for editing layer-user
            LayerUser layerUserData = LayerRequestDataConverter.convertLayerUserFromDataProvider(layerUserDataProvider);
            layerUserDao.updateLayerUserByUidAndLayerId(layerUserData, uid, layerId);
            lastUpdateManager.updateLayerTimestamps(Cf.list(layerId), actionInfo);
        }
        // Update for all layer types, including SERVICE
        boolean applyToEvents = Binary.parseBoolean(dp.getText("apply-to-events", false));
        updateLayerNotification(actionInfo, uid, layer, layerUserDataProvider, applyToEvents);
    }

    private void updateLayerNotification(
            ActionInfo actionInfo, PassportUid uid,
            Layer layer, DataProvider layerUserDataProvider, boolean applyToEvents)
    {
        if (Binary.parseBoolean(layerUserDataProvider.getText("process-ntf", false))) {
            if (isService(layer) && ControlDataNotification.isSupportedBy(layer.getSid())) {
                final String msg = "Can't change notification(s) for a layer with sid: " + layer.getSid();
                throw new PermissionDeniedUserException(msg);
            }
            ListF<Notification> notifications = RequestDataConverter
                    .convertNotifications(layerUserDataProvider.getDataProvider("notification", false));

            updateNotification(uid, layer.getId(), NotificationsData.updateFromWeb(notifications), applyToEvents, actionInfo);
        }
    }

    public ListF<Notification> getNotificationOrGetDefaultBySid(PassportUid uid, PassportSid sid) {
        if (sid.sameAs(PassportSid.CALENDAR)) {
            Option<Long> layerIdO = getDefaultLayerId(uid);
            return layerIdO.isPresent()
                    ? getNotificationsOrGetDefault(uid, layerIdO.get())
                    : notificationRoutines.getDefaultNotifications();
        }
        Option<Long> svcLayerIdO = layerDao.findServiceLayerIdByUidAndSid(uid, sid);
        return (svcLayerIdO.isPresent()
                ? getNotificationsOrGetDefault(uid, svcLayerIdO.get())
                : notificationRoutines.getDefaultNotifications()
        );
    }

    public ListF<Notification> getNotificationsOrGetDefault(PassportUid uid, long layerId) {
        Option<Long> layerUserIdO = layerUserDao.findLayerUserByLayerIdAndUid(layerId, uid).map(LayerUser.getIdF());
        if (layerUserIdO.isPresent()) {
            return notificationDbManager.getNotificationsByLayerUserId(layerUserIdO.get());
        } else {
            return notificationRoutines.getDefaultNotifications();
        }
    }

    public void updateNotification(
            final PassportUid uid, final long layerId,
            NotificationsData.Update notificationsData, boolean applyToEvents, ActionInfo actionInfo)
    {
        long layerUserId = layerUserDao.findLayerUserByLayerIdAndUid(layerId, uid).get().getId();
        ListF<Notification> notificationsCurrent = notificationDbManager.getNotificationsByLayerUserId(layerUserId);

        notificationDbManager.updateLayerNotifications(layerUserId, notificationsData);

        if (applyToEvents) {
            Instant since = actionInfo.getNow().minus(Option.of(Duration.ZERO)
                    .plus(notificationsCurrent.map(Notification::getOffset))
                    .plus(notificationsData.getNotifications().map(Notification::getOffset))
                    .max());

            ListF<EventAndRepetition> events = eventInfoDbLoader.getEventsOnLayers(Cf.list(layerId),
                    EventLoadLimits.startsInInterval(new InfiniteInterval(since, Option.empty())));

            ListF<Long> eventIds = events.map(EventAndRepetition::getEventId);

            ListF<EventUser> eventUsers = eventUserDao.findEventUsersByEventIdsAndUid(eventIds, uid);

            ListF<Long> eventUserIds = eventUsers.map(EventUser::getId);

            ListF<EventNotifications> notificationss = notificationDbManager.getNotificationsByEventUserIds(eventUserIds);
            ListF<EventNotificationChangesInfo> changes = NotificationChangesFinder.changes(notificationss, notificationsData);
            notificationDbManager.updateEventNotifications(changes);

            ListF<Long> eIds = eventIds.filterNot(eventUsers.map(EventUser::getEventId).unique()::containsTs);
            if (eIds.isNotEmpty()) {
                log.debug("Number of events without event user: {}", eIds.size());
                ListF<Notification> notifications = notificationDbManager.getNotificationsByLayerUserId(layerUserId);
                for (long eId : eIds) {
                    eventUserRoutines.saveEventUser(uid, eId, new EventUser(), notifications, actionInfo);
                }
            }

            notificationRoutines.recalcUserNextSendTs(uid, events, actionInfo);

            lastUpdateManager.updateMainEventAndLayerTimestampsByLayerId(layerId, actionInfo);
        }

    }

    public void deleteLayer(UserInfo user, long layerId, ActionInfo actionInfo) {
        val layer = getLayerById(layerId);
        val layerPermInfo = LayerInfoForPermsCheck.fromLayer(layer);
        authorizer.ensureCanPerformLayerAction(user, layerPermInfo, Optional.empty(), LayerAction.DELETE, actionInfo.getActionSource());

        // remove meetings from this layer which are organized by layer creator
        PassportUid layerCreatorUid = layer.getCreatorUid();
        ListF<Long> meetingIds = eventLayerDao.findEventIdsByLayerIdAndUserOrganizerId(
                layerId, layerCreatorUid);
        ListF<EventMessageParameters> meetingMails = eventRoutines.deleteEvents(
                Option.of(user), meetingIds,
                InvitationProcessingMode.SAVE_ATTACH_SEND, actionInfo);

        // remove events which lie on this layer only
        ListF<Long> eventIdsInThisLayer = eventLayerDao.findEventLayerEventIdsByLayerId(layerId);
        ListF<Long> eventIdsOnlyInThisLayer = eventLayerDao.findEventIdsWithOnlyOneEventLayer(eventIdsInThisLayer);
        ListF<EventMessageParameters> eventMails = eventRoutines.deleteEventsSafe(
                Option.of(user), eventIdsOnlyInThisLayer, InvitationProcessingMode.SAVE_ONLY, actionInfo);

        // XXX: remove all layer_users here?
        eventRoutines.storeAndDeleteEventLayersByLayerId(layerId, actionInfo);
        if (layer.getType() == LayerType.FEED) {
            icsFeedDao.deleteIcsFeedByLayerId(layerId);
        }
        updateDefaultLayer(user.getUid(), layerId, false);
        layerDbManager.deleteOnlyLayerWithNotification(layerId);

        eventInvitationManager.sendEventMails(meetingMails.plus(eventMails), actionInfo);
    }

    // Moves all events from one user layer to another, then deletes source layer
    public void deleteMoveEvents(UserInfo user, long oldId, long newId, ActionInfo actionInfo) {
        Validate.isTrue(oldId != newId,
                "Attempted to illegally move events from layer (id: " + oldId + ") to itself");
        // NOTE: No need to ensure LayerAction.[CREATE_EVENT, ATTACH_EVENT]
        //       on destination layer as we move events to own layers only
        // Checks: ensure that old and new layers are of type 'user' and belong to 'uid',
        //         also check that 'uid' has permissions to delete old layer
        log.debug("LR.deleteMoveEvents() from l.id {} to l.id  {}...", oldId, newId);

        val source = actionInfo.getActionSource();
        authorizer.ensureMoveAction(user, oldId, newId, LayerAction.DELETE, source);

        eventDao.lockEventsByLayerId(oldId);
        eventLayerDao.updateIgnoreEventLayerSetLayerIdAndLayerCreatorUidByLayerId(
                newId, getLayerById(newId).getCreatorUid(), oldId, actionInfo);
        eventLayerDao.deleteEventLayersByLayerIds(Cf.list(oldId));

        updateDefaultLayer(user.getUid(), oldId, false);
        layerDbManager.deleteOnlyLayerWithNotification(oldId);

        lastUpdateManager.updateMainEventAndLayerTimestampsByLayerId(newId, actionInfo);
    }

    public void detach(PassportUid clientUid, PassportUid subjectUid, long layerId, ActionInfo actionInfo) {
        detach(clientUid, Cf.list(subjectUid), layerId, actionInfo);
    }

    public void detach(PassportUid clientUid, ListF<PassportUid> subjectUids, long layerId, ActionInfo actionInfo) {
        if (subjectUids.isEmpty()) {
            return;
        }

        Validate.isFalse(subjectUids.containsTs(getCreatorUid(layerId)), "cannot detach layer from creator");

        eventInvitationManager.removeMeetingsFromLayer(clientUid, layerId, subjectUids, actionInfo);

        ListF<Long> layerUserIds = layerUserDao.findLayerUserIdsByLayerIdAndUids(layerId, subjectUids);
        notificationDbManager.deleteNotificationsByLayerUserIds(layerUserIds);

        layerUserDao.deleteLayerUsersByIds(layerUserIds);
        layerUserCache.flush();

        layerInvitationDao.deleteLayerInvitationByLayerIdAndUids(layerId, subjectUids);
    }

    public String obtainPtk(PassportUid uid, long id, boolean forceNew) {
        String res = null;
        // Both creator check and probably source for existing private token
        Layer l = layerDao.findLayerById(id).copy();
        if (!l.getCreatorUid().sameAs(uid)) {
            throw new IllegalStateException();
        }
        if (!forceNew) {
            res = l.getPrivateToken().getOrNull();
        }
        if (forceNew || StringUtils.isEmpty(res)) {
            // XXX do it safely
            res = UidGen.createPrivateToken();

            Layer temp = new Layer();
            temp.setId(id);
            temp.setPrivateToken(res);
            layerDao.updateLayer(temp);
        }
        return res;
    }

    public Layer getByPrivateToken(String privateToken) {
        return layerDao.findLayerByPrivateToken(privateToken).getOrThrow(CommandRunException.createSituationF(
                "Bean cannot be found by private token: " + privateToken, Situation.INVALID_TOKEN));
    }

    @Override
    public PassportUid getCreatorUid(long layerId) {
        return getLayerById(layerId).getCreatorUid();
    }

    /**
     * @return true if any changes were made, false otherwise (sharing was pointless)
     */
    public boolean startNewSharing(PassportUid uid, long layerId, LayerActionClass layerActionClass) {
        val layerUserOpt = findOwnerBean(uid, layerId).toOptional();
        val layerCreatorUid = getCreatorUid(layerId);

        if (!uid.isYandexTeamRu()) {
            settingsRoutines.createSettingsIfNotExistsForUids(Cf.list(uid));
        }

        return layerUserOpt.map(layerUser -> {
            return updateLayerUserPermForAnonymous(uid, layerUser, layerActionClass);
        })
        .orElseGet(() -> {
            saveLayerUser(uid, layerId, layerCreatorUid, layerActionClass);
            return true;
        });
    }

    private boolean updateLayerUserPermForAnonymous(PassportUid uid, LayerUser layerUser, LayerActionClass newLayerActionClass) {
        val isAnonymousSharing = layerUser.getPerm() == LayerActionClass.ACCESS;

        if (isAnonymousSharing) {
            var layerId = layerUser.getLayerId();
            updateLayerUserPerm(uid, layerId, newLayerActionClass);
        }

        return isAnonymousSharing;
    }

    public void updateLayerUserPerm(PassportUid uid, long layerId, LayerActionClass layerActionClass) {
        Validate.isFalse(getCreatorUid(layerId).sameAs(uid), "perm cannot be updated, as user is layer creator");
        layerUserDao.updateLayerUserPerm(layerId, uid, layerActionClass);
        layerUserCache.removeFromCache(Tuple2.tuple(layerId, uid));
    }

    public void updateLayerEventsClosedByDefault(long layerId, boolean isClosed) {
        layerDao.updateLayerEventsClosedByDefault(layerId, isClosed);
        layerByIdCache.removeFromCache(layerId);
    }

    public void saveLayerUser(PassportUid uid, long layerId, PassportUid layerCreatorUid, LayerActionClass layerActionClass) {
        LayerUser layerUser;
        layerUser = new LayerUser();
        layerUser.setUid(uid);
        layerUser.setLayerId(layerId);

        layerUser.setPerm(layerActionClass);

        LayerUser luCreator = findOwnerBean(layerCreatorUid, layerId).get();
        layerUser.setCssClass(luCreator.getCssClass().getOrNull());
        layerUser.setHeadBgColor(luCreator.getHeadBgColor().getOrNull());
        layerUser.setHeadFgColor(luCreator.getHeadFgColor().getOrNull());
        layerUser.setBodyBgColor(luCreator.getBodyBgColor().getOrNull());
        layerUserDao.saveLayerUser(layerUser);

        layerUserCache.removeFromCache(Tuple2.tuple(layerId, uid));
    }


    private final Cache<Long, Layer> layerByIdCache =
            TlCache.asCache(LayerRoutines.class.getName() + ".layerById");

    private final Cache<Tuple2<Long, PassportUid>, Option<LayerUser>> layerUserCache =
            TlCache.asCache(LayerRoutines.class.getName() + ".layerUser");

    public void putLayersToCache(ListF<Layer> layers) {
        layerByIdCache.putInCacheBatch(layers.zipWith(LayerFields.ID.getF()).invert());
    }

    private void putLayerUsersWithRelationsToCache(ListF<LayerUserWithRelations> layerUsers) {
        putLayersToCache(layerUsers.map(LayerUserWithRelations::getLayer));
        putLayerUsersToCache(layerUsers.map(LayerUserWithRelations::getLayerUser));
    }

    public ListF<Layer> getLayersById(ListF<Long> ids) {
        return layerByIdCache.getFromCacheSomeBatch(ids, ids1 -> layerDao.findLayersByIds(ids1)).get2();
    }

    public ListF<Layer> findLayersByIdsOrTypesOrTokens(
            ListF<Long> ids, Option<PassportUid> uid, ListF<LayerType> types, ListF<String> tokens)
    {
        ListF<Layer> byIds = layerByIdCache.getFromCacheBatch(ids, is -> is.map(
                layerDao.findLayers(LayerFields.ID.column().inSet(is)).toMapMappingToKey(Layer::getId)::getOptional)
        ).filterMapOptional(Tuple2::get2);

        ListF<Layer> byTokens = tokens.isNotEmpty()
                ? layerDao.findLayers(LayerFields.PRIVATE_TOKEN.column().inSet(tokens))
                : Cf.list();

        layerByIdCache.putInCacheBatch(byTokens.toMapMappingToKey(Layer.getIdF()).entries());

        ListF<Layer> byTypes = uid.isPresent() && types.isNotEmpty()
                ? layerDao.findLayers(LayerFields.CREATOR_UID.eq(uid.get()).and(LayerFields.TYPE.column().inSet(types)))
                : Cf.list();

        layerByIdCache.putInCacheBatch(byTypes.toMapMappingToKey(Layer::getId).entries());

        return byIds.plus(byTokens).plus(byTypes);
    }

    public Layer getLayerById(long id) {
        return getLayersById(Cf.list(id)).single();
    }

    public Option<Layer> findLayerById(long id) {
        Option<Layer> layer = layerDao.findLayerByIdSafe(id);

        layer.forEach(l -> layerByIdCache.putInCache(id, l));

        return layer;
    }

    public Option<LayerUser> getLayerUser(final long layerId, final PassportUid uid) {
        return layerUserCache.getFromCacheSome(Tuple2.tuple(layerId, uid), new Function0<Option<LayerUser>>() {
            public Option<LayerUser> apply() {
                return layerUserDao.findLayerUserByLayerIdAndUid(layerId, uid);
            }
        });
    }

    private void putLayerUsersToCache(ListF<LayerUser> layerUsers) {
        for (LayerUser layerUser : layerUsers) {
            layerUserCache.putInCache(Tuple2.tuple(layerUser.getLayerId(), layerUser.getUid()), Option.of(layerUser));
        }
    }

    public ListF<LayerUser> getLayerUsers(SqlCondition c) {
        ListF<LayerUser> r = layerUserDao.findLayerUsers(c);
        putLayerUsersToCache(r);
        return r;
    }

    public Map<PassportUid, LayerActionClass> getLayerPermissions(long layerId) {
        return layerUserDao.getLayerPermissions(layerId);
    }

    public ListF<Long> getLayerIdsByUid(PassportUid uid) {
        return getLayerUsers(LayerUserFields.UID.eq(uid)).map(LayerUser::getLayerId);
    }

    public ListF<LayerUser> getLayerUsersIsNotifyChangesByLayerIds(ListF<Long> layerIds) {
        return getLayerUsers(LayerUserFields.IS_NOTIFY_CHANGES.eq(true)
                .and(LayerUserFields.LAYER_ID.column().inSet(layerIds)));
    }

    public ListF<LayerUser> getLayerUsersByUids(ListF<PassportUid> uids) {
        return getLayerUsers(LayerUserFields.UID.column().inSet(uids));
    }

    public ListF<LayerUser> getLayerUsersByUidAndLayerIds(PassportUid uid, ListF<Long> layerIds) {
        SqlCondition c = SqlCondition.all(LayerUserFields.UID.eq(uid), LayerUserFields.LAYER_ID.column().inSet(layerIds));

        ListF<LayerUser> layerUsers = layerUserDao.findLayerUsers(c);

        SetF<Long> missedLayerIds = layerIds.unique().minus(layerUsers.map(LayerUserFields.LAYER_ID.getF()));
        for (Long missedLayerId : missedLayerIds) {
            layerUserCache.putInCache(Tuple2.tuple(missedLayerId, uid), Option.empty());
        }

        putLayerUsersToCache(layerUsers);

        return layerUsers;
    }

    public long getOrCreateFirstUserLayer(PassportUid uid, Option<String> layerNameO) {
        if (layerDao.findUserLayerCount(uid) == 0) {
            return createDefaultUserLayer(uid, layerNameO);
        } else {
            Option<Long> res = getFirstUserLayerId(uid, layerNameO);
            return res.isPresent() ? res.get() : createUserLayer(uid, layerNameO);
        }
    }

    public long getOrCreateDefaultLayer(final PassportUid uid) {
        if (layerDao.findUserLayerCount(uid) == 0) {
            return createDefaultUserLayer(uid, Option.empty());
        } else {
            return getDefaultLayerId(uid).get();
        }
    }

    public MapF<PassportUid, Long> getOrCreateDefaultLayers(CollectionF<SettingsInfo> settings) {
        MapF<PassportUid, Option<Long>> firsts = layerDao.findFirstCreatedUserLayerIds(
                settings.filterMap(s -> Option.when(!s.getCommon().getLayerId().isPresent(), s.getUid()))).toMap();

        return settings.toMap(SettingsInfo::getUid, s -> s.getCommon().getLayerId()
                .orElse(() -> firsts.getOrThrow(s.getUid()))
                .getOrElse(() -> createDefaultUserLayer(s.getUid(), Option.empty())));
    }

    public List<Long> findOwnLayerIds(PassportUid uid) {
        return layerDao.findLayerIdsByUid(uid);
    }

    private long createDefaultUserLayer(final PassportUid uid, Option<String> layerName) {
        final Layer layer = new Layer();
        layer.setIsEventsClosedByDefault(!uid.isYandexTeamRu());

        if (layerName.isPresent()) {
            layer.setName(layerName.get());
        }
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.RW_M, new Function0<Long>() {
            public Long apply() {
                return createUserLayer(uid, notificationRoutines.getDefaultNotifications(),
                        layer, true, createDefaultLayerUserOverrides(uid.getDomain()));
            }
        });
    }

    public Option<Long> getDefaultLayerId(final PassportUid uid) {
        Option<Long> layerId = getDefaultLayerIdFromSettings(Option.of(uid));
        if (!layerId.isPresent()) {
            return getFirstUserLayerId(uid, Option.empty());
        } else {
            return layerId;
        }
    }

    public Option<Long> getDefaultLayerIdFromSettings(final Option<PassportUid> uidO) {
        if (uidO.isPresent()) {
            return Option.ofNullable(settingsRoutines.getSettingsField(uidO.get(), SettingsFields.LAYER_ID));
        } else {
            return Option.empty();
        }
    }

    public ListF<LayerUserWithRelations> getLayerUsersWithRelationsByUid(PassportUid layerUserUid, Option<Language> lang) {
        ListF<LayerUserWithRelations> layers = layerDao.findLayerUsersWithRelationsByLayerUserUid(layerUserUid)
                .map(createLayerUserWithRelationsF(lang));

        putLayerUsersWithRelationsToCache(layers);

        return layers.sorted(layerComparatorForUser(Option.of(layerUserUid)).compose(LayerUserWithRelations::getLayer));
    }

    public ListF<LayerUserWithRelations> getLayerUsersWithRelationsByUids(ListF<PassportUid> layerUserUids) {
        ListF<LayerUserWithRelations> layerUsers = layerDao.findLayerUsersWithRelationsByLayerUserUids(layerUserUids)
                .map(createLayerUserWithRelationsF(Option.empty()));

        putLayerUsersWithRelationsToCache(layerUsers);
        return layerUsers;
    }

    public Option<LayerUserWithRelations> getLayerUserWithRelations(PassportUid uid, long layerId, Option<Language> lang) {
        Option<LayerUserWithRelations> lu = layerDao.findLayerUserWithRelations(uid, layerId)
                .map(createLayerUserWithRelationsF(lang).asFunction());

        putLayerUsersWithRelationsToCache(lu);
        return lu;
    }

    public Function2<LayerUser, Layer, LayerUserWithRelations> createLayerUserWithRelationsF(Option<Language> lang) {
        return (layerUser, layer) -> new LayerUserWithRelations(
                layerUser, layer, evalLayerName(layer, lang), evalLayerHeadBgColorRgb(layer, layerUser));
    }

    public ListF<LayerUserWithRelations> getLayerUsersWithRelationsByLayerIds(ListF<Long> layerIds) {
        ListF<LayerUserWithRelations> layers = layerDao.findLayerUsersWithRelationsByLayerIds(layerIds)
                .map(createLayerUserWithRelationsF(Option.empty()));

        putLayerUsersWithRelationsToCache(layers);

        return layers.sorted(layerComparatorForUser(Option.empty()).compose(LayerUserWithRelations::getLayer));
    }

    public void updateLayer(Layer layerData, PassportUid uid, ActionInfo actionInfo) {
        Layer data = keepNullNameOnUpdate(layerData.copy(), uid);
        data.setLastUpdateTs(actionInfo.getNow());
        data.setCollLastUpdateTs(actionInfo.getNow());

        layerDao.updateLayer(data);
        layerByIdCache.removeFromCache(layerData.getId());
    }

    public void updateLayerUserVisibleInUi(PassportUid uid, long layerId, boolean isVisible) {
        layerUserDao.updateLayerUserSetVisibleInUiByLayerIdAndUid(layerId, uid, isVisible);
        layerUserCache.removeFromCache(Tuple2.tuple(layerId, uid));
    }

    public void updateLayerUser(long layerId, PassportUid uid, LayerUser layerUserData, ActionInfo actionInfo) {
        layerUserDao.updateLayerUserByUidAndLayerId(layerUserData, uid, layerId);
        lastUpdateManager.updateLayerTimestamps(Cf.list(layerId), actionInfo);

        layerUserCache.flush();
    }

    public DateTimeZone getLayerTimezone(Layer layer) {
        return dateTimeManager.getTimeZoneForUid(layer.getCreatorUid());
    }

    @Override
    protected Option<LayerUser> findOwnerBean(PassportUid uid, long id) {
        return layerUserDao.findLayerUserByLayerIdAndUid(id, uid);
    }


    public static boolean isService(Layer layer) {
        return layer.getType() == LayerType.SERVICE; // <=> sid set and != 31
    }

    public static Service getService(Layer layer) {
        return DbSvcRoutines.getService(layer.getSid());
    }

    public static Option<Service> getServiceO(Layer layer) {
        return isService(layer) ? Option.of(getService(layer)) : Option.empty();
    }

    private Layer keepNullNameOnUpdate(Layer layer, PassportUid uid) {
        Layer storedLayer = getLayerById(layer.getId());

        if (!storedLayer.getName().filter(Cf.String.notEmptyF()).isPresent()) {
            Option<Language> lang = userManager.getUserByUid(uid).filterMap(YandexUser::getLanguage);
            ListF<String> defaultNames =
                    LayerRoutines.DEFAULT_USER_LAYER_NAME.getAllNames().plus(evalLayerName(storedLayer, lang));

            if (layer.getName().exists(defaultNames::containsTs) ) {
                Layer layerWithNoName = layer.copy();
                layerWithNoName.setName(Option.empty());

                return layerWithNoName;
            }
        }
        return layer;
    }

    private String evalLayerName(final Layer layer, boolean isShort, Option<Language> lang, Option<Settings> settings) {
        if (isService(layer)) {
            Service service = getService(layer);
            return isShort ? service.getSName() : service.getName();
        } else {
            Option<String> layerNameO = layer.getName().filter(Cf.String.notEmptyF());
            Validate.isTrue(layer.getType() == LayerType.USER || layerNameO.isPresent());
            return layerNameO.getOrElse(new Function0<String>() {
                public String apply() {
                    Lazy<Settings> getSettings = Lazy.withSupplier(() ->
                            settings.getOrElse(() -> settingsRoutines.getSettingsByUid(layer.getCreatorUid()).getCommon()));

                    if (layer.getCreatorUid().isYandexTeamRu()) { // CAL-2441
                        Settings settings = getSettings.get();
                        return Cf.list(
                                settings.getUserName().getOrElse(""),
                                settings.getUserLogin().getOrElse(""),
                                settings.getEmail().getLocalPart(),
                                DEFAULT_USER_LAYER_NAME.getName(lang.getOrElse(settings::getLanguage)))
                                .find(Cf.String.emptyF().notF()).get();
                    } else {
                        return DEFAULT_USER_LAYER_NAME.getName(lang.getOrElse(() -> getSettings.get().getLanguage()));
                    }
                }
            });
        }
    }

    public String evalLayerName(Layer layer, Option<Language> lang) {
        return evalLayerName(layer, false, lang, Option.empty());
    }

    public String evalLayerNameBySettings(Layer layer, Option<Settings> settings) {
        return evalLayerName(layer, false, Option.empty(), settings);
    }

    public Color evalLayerHeadBgColorRgb(Layer layer, LayerUser layerUser) {
        if (layer.getType() != LayerType.SERVICE) {
            return Color.fromRgb(layerUser.getHeadBgColor().getOrThrow("Layer of type " + layer.getType() + " without color"));
        } else {
            return Color.fromRgb(DbSvcRoutines.getService(layer.getSid()).getLayerHeadBgColor());
        }
    }

    public String evalLayerSName(Layer layer) {
        return evalLayerName(layer, true, Option.empty(), Option.empty());
    }

    public static String evalCssClass(Layer layer, LayerUser layerUser) {
        return isService(layer) ? getService(layer).getLayerCssClass() : layerUser.getCssClass().getOrNull();
    }

    public static String evalServiceDomainName(Layer layer) {
        return isService(layer) ? getService(layer).getDomainName() : null;
    }

    public void updateDefaultLayer(PassportUid uid, Long id, boolean isDefault) {
        // Default layer for settings
        if (isDefault) {
            val layer = getLayerById(id);
            authorizer.ensureLayerProperties(LayerInfoForPermsCheck.fromLayer(layer), uid, LayerType.USER);
        }
        Settings s = settingsRoutines.getOrCreateSettingsByUid(uid).getCommon();
        // If default layer was this one AND is_default == false, update to null;
        // or if default layer was another one AND is_default == true, update to this layer id.
        if (ObjectUtils.equals(s.getLayerId().getOrNull(), id) != isDefault) {
            Long newLayerId = isDefault ? id : null;
            settingsRoutines.updateDefaultLayer(uid, newLayerId);
            log.debug("Settings default layer id updated to: {}", newLayerId);
        } // if
    }

    public Comparator<Layer> layerComparatorForUser(Option<PassportUid> uid) {
        // explicit default layer id from settings, or first by creation, or none
        Option<Long> defaultLayerIdO = uid.isPresent() ? getDefaultLayerId(uid.get()) : Option.empty();

        Comparator<Layer> r = Comparator.constEqualComparator();

        // 1. [default layer goes first]
        if (defaultLayerIdO.isPresent()) {
            val defaultLayerId = defaultLayerIdO.get().longValue();
            Function<Layer, Integer> f = a -> a.getId() == defaultLayerId ? -1 : 0;
            r = r.thenComparing(f.andThenNaturalComparator());
        }

        // 2. user layers, then all other (subscription) ones
        r = r.thenComparing((o1, o2) -> {
            val t1 = o1.getType();
            val t2 = o2.getType();
            return t1 == t2 ? 0 : t1 == LayerType.USER ? -1 : t2 == LayerType.USER ? 1 : 0;
        });

        // 3. [me, then other users]
        if (uid.isPresent()) {
            r = r.thenComparing(LayerFields.CREATOR_UID.getF().<Layer, PassportUid>uncheckedCast().andThen(Comparator.valueLowC(uid.get())));
        }

        // 4. sort by alphabet
        Function<Layer, String> evalLayerNameF = a -> evalLayerName(a, Option.empty());
        r = r.thenComparing(evalLayerNameF.andThenNaturalComparator());

        return r;
    }
}
