package ru.yandex.calendar.logic.layer;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import lombok.val;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.Tuple4;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.NullableOtherValue;
import ru.yandex.calendar.logic.beans.generated.LayerUser;
import ru.yandex.calendar.logic.beans.generated.LayerUserFields;
import ru.yandex.calendar.logic.sharing.perm.LayerActionClass;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.annotation.SampleValue;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlQueryUtils;

import static java.util.Collections.emptyList;

public class LayerUserDao extends CalendarJdbcDaoSupport {

    @Autowired
    private GenericBeanDao genericBeanDao;


    @RunWithRandomTest
    public ListF<LayerUser> findLayerUsers(SqlCondition... conditions) {
        SqlCondition c = SqlCondition.all(conditions);
        String q = "SELECT * FROM layer_user WHERE " + c.sql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, LayerUser.class, c.args());
    }

    @RunWithRandomTest
    public Option<LayerUser> findLayerUserByLayerIdAndUid(long layerId, PassportUid uid) {
        String q = "SELECT * FROM layer_user WHERE layer_id = ? AND uid = ?";
        return getJdbcTemplate().queryForOption(q, LayerUser.class, layerId, uid);
    }

    @RunWithRandomTest
    public ListF<LayerUser> findLayerUserByLayerId(long layerId) {
        String q = "SELECT * FROM layer_user WHERE layer_id = ?";
        return getJdbcTemplate().queryForList(q, LayerUser.class, layerId);
    }

    @RunWithRandomTest
    public int updateLayerUserPerm(long layerId, PassportUid uid, @SampleValue("admin") @Nullable LayerActionClass perm) {
        String q = "UPDATE layer_user SET perm = ? WHERE layer_id = ? AND uid = ?";
        return getJdbcTemplate().update(q, new NullableOtherValue<>(Option.ofNullable(perm)), layerId, uid);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateLayerUserSetVisibleInUiByLayerIdAndUid(long layerId, PassportUid uid, boolean visibleInUi) {
        String q = "UPDATE layer_user SET is_visible_in_ui = ? WHERE layer_id = ? AND uid = ? AND is_visible_in_ui != ?";
        getJdbcTemplate().update(q, visibleInUi, layerId, uid, visibleInUi);
    }

    @RunWithRandomTest
    public ListF<PassportUid> findNonInvitedLayerUsers(long layerId) {
        val q = "SELECT uid\n"
              + "FROM layer_user\n"
              + "WHERE perm = 'access' AND layer_id = ?";
        return getJdbcTemplate().queryForList(q, Long.class, layerId).map(PassportUid::cons);
    }

    @RunWithRandomTest
    public Map<PassportUid, LayerActionClass> getLayerPermissions(long layerId) {
        val q = "SELECT uid, perm\n"
              + "FROM layer_user\n"
              + "WHERE layer_id = ?";

        val data = getJdbcTemplate().queryForList2(q, Long.class, String.class, layerId);
        return StreamEx.of(data)
                .mapToEntry(Tuple2::get1, Tuple2::get2)
                .mapKeys(PassportUid::new)
                .mapValues(String::toUpperCase)
                .mapValues(LayerActionClass::valueOf)
                .distinctKeys()
                .toImmutableMap();
    }

    @RunWithRandomTest
    public ListF<Long> findLayerUserIdsByLayerIdAndUids(long layerId, ListF<PassportUid> uids) {
        String q = "SELECT id FROM layer_user WHERE layer_id = ? AND uid " + SqlQueryUtils.inSet(uids);

        if (skipQuery(uids, q, layerId)) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, layerId);
    }

    @RunWithRandomTest
    public ListF<Long> findLayerUserIdsByLayerIds(ListF<Long> layerIds) {
        String q = "SELECT id FROM layer_user WHERE layer_id " + SqlQueryUtils.inSet(layerIds);

        if (skipQuery(layerIds, q)) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class);
    }

    @RunWithRandomTest
    public ListF<Long> findLayerIdsByLayerUserUids(ListF<PassportUid> uids) {
        String q = "SELECT layer_id FROM layer_user WHERE uid " + SqlQueryUtils.inSet(uids);

        if (skipQuery(uids, q)) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class);
    }

    @RunWithRandomTest
    public Tuple2List<PassportUid, List<UserLayersSharing.LayerPermissionSettings>> findOwnAndInvitedLayerUserPerms(ListF<PassportUid> uids) {
        val q = "SELECT lu.uid, lu.layer_id, lu.perm, l.creator_uid\n"
              + "FROM layer_user lu\n"
              + "JOIN layer l ON l.id = lu.layer_id\n"
              + "WHERE lu.uid " + SqlQueryUtils.inSet(uids);

        if (skipQuery(uids, q)) {
            return Tuple2List.tuple2List();
        }

        final var result = getJdbcTemplate().query4(q,
                (rs, rn) -> PassportUid.cons(rs.getLong(1)),
                (rs, rn) -> rs.getLong(2),
                (rs, rn) -> LayerActionClass.valueOf(rs.getString(3).toUpperCase()),
                (rs, rn) -> PassportUid.cons(rs.getLong(4)));

        val grouped = StreamEx.of(result)
                .mapToEntry(Tuple4::get1,
                            r -> new UserLayersSharing.LayerPermissionSettings(r.get2(), r.get3().getActions(), r.get4()))
                .grouping();

        return uids.zipWith(uid -> grouped.getOrDefault(uid, emptyList()));
    }

    public void deleteLayerUsersByIds(ListF<Long> layerUserIds) {
        String q = "DELETE FROM layer_user WHERE id " + SqlQueryUtils.inSet(layerUserIds);

        if (skipQuery(layerUserIds, q)) return;

        getJdbcTemplate().update(q);
    }

    @RunWithRandomTest
    public boolean existsLayerUserForUserAndAnyOfLayers(PassportUid ownerUid, ListF<Long> layerIds) {
        SqlCondition layerIdsIn = LayerUserFields.LAYER_ID.column().inSet(layerIds.unique());
        String q = "SELECT COUNT(*) FROM layer_user WHERE uid = ? " + layerIdsIn.andSql();

        if (skipQuery(layerIdsIn, q, ownerUid, layerIdsIn.args())) return false;

        return getJdbcTemplate().queryForInt(q, ownerUid, layerIdsIn.args()) > 0;
    }

    public long saveLayerUser(LayerUser layerUser) {
        return genericBeanDao.insertBeanGetGeneratedKey(layerUser);
    }

    public void updateLayerUserByUidAndLayerId(LayerUser layerUserData, PassportUid uid, long layerId) {
        genericBeanDao.updateBeans(
                layerUserData,
                SqlCondition.condition("uid = ? AND layer_id = ?", uid, layerId));
    }

    public void updateLayerUser(LayerUser layerUser) {
        genericBeanDao.updateBean(layerUser);
    }

    @RunWithRandomTest
    public void updateLayerUsersSetUidByIds(ListF<Long> layerUserIds, PassportUid newUid) {
        String q = "UPDATE layer_user SET uid = ? WHERE id " + SqlQueryUtils.inSet(layerUserIds);

        if (skipQuery(layerUserIds, q, newUid)) return;

        getJdbcTemplate().update(q, newUid);
    }
}
