package ru.yandex.calendar.logic.log;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonUnwrapped;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import lombok.val;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.impl.ArrayListF;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.log.dao.EventsLogDao;
import ru.yandex.calendar.util.db.TransactionListener;
import ru.yandex.misc.ThreadLocalX;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.HostnameUtils;

public class EventsLogger {
    public static final String JSONNAME = "events-json";

    private static final Logger jsonLogger = LoggerFactory.getLogger(JSONNAME);

    private static final ThreadLocalX<ArrayListF<String>> jsonTransactional = new ThreadLocalX<>();
    public static final ObjectMapper mapper = new ObjectMapper()
            .registerModule(new Jdk8Module())
            .registerModule(new JodaModule())
            .configure(com.fasterxml.jackson.databind.SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false)
            .setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE)
            .setVisibility(PropertyAccessor.ALL, JsonAutoDetect.Visibility.NONE)
            .setVisibility(PropertyAccessor.FIELD, JsonAutoDetect.Visibility.ANY);

    @Autowired
    private EventsLogDao eventsLogDao;

    private static class ResultLogEvent {
        private String host;
        @JsonProperty("event_info")
        private LogEvent eventInfo;
        @JsonUnwrapped
        private ActionInfo actionInfo;

        ResultLogEvent(LogEvent event, ActionInfo actionInfo) {
            this.eventInfo = event;
            this.actionInfo = actionInfo;
            this.host = HostnameUtils.localHostnameShort();
        }
    }

    public static String genJsonLine(LogEvent event, ActionInfo actionInfo) {
        try {
            return mapper.writeValueAsString(new ResultLogEvent(event, actionInfo));
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    public void log(LogEvent event, ActionInfo actionInfo) {
        if (!event.isAccept(actionInfo))
            return;

        if (!event.isOmitDatabasing(actionInfo)) {
            // will be rolled back with transaction, if error
            eventsLogDao.logSafe(event, actionInfo);
        }

        val jsonLine = genJsonLine(event, actionInfo);

        if (event.isTransactional() && jsonTransactional.isSet()) {
            jsonTransactional.get().add(jsonLine);
        } else {
            jsonLogger.info(jsonLine);
        }
    }

    public static TransactionListener createTransactionListener() {
        return new TransactionListener() {
            @Override
            public void transactionStarted() {
                jsonTransactional.set(new ArrayListF<>());
            }

            @Override
            public void transactionCommitted() {
                jsonTransactional.get().forEach(jsonLogger::info);
            }

            @Override
            public void transactionEnded() {
                jsonTransactional.remove();
            }
        };
    }
}
