package ru.yandex.calendar.logic.log.change;

import java.util.Optional;

import lombok.val;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.event.ActorId;
import ru.yandex.calendar.logic.event.EventAttachedLayerId;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.log.EventIdLogDataJson;
import ru.yandex.calendar.logic.log.change.changes.EventFieldsChangesJson;
import ru.yandex.calendar.logic.log.change.changes.LayersChangesJson;
import ru.yandex.calendar.logic.log.change.changes.RepetitionChangesJson;
import ru.yandex.calendar.logic.log.change.changes.ResourcesChangesJson;
import ru.yandex.calendar.logic.log.change.changes.UserRelatedChangesJson;
import ru.yandex.calendar.logic.log.change.changes.UsersChangesJson;
import ru.yandex.calendar.logic.sharing.participant.EventParticipants;
import ru.yandex.misc.email.Email;

public class EventChangeLogEvents {

    public static EventChangeLogEventJson created(
            ActorId actor, EventWithRelations event, RepetitionInstanceInfo repetition) {
        return new EventChangeLogEventJson(actor, EventChangeType.CREATE, new EventIdLogDataJson(event),
                EventChangesJson.find(Optional.empty(), Optional.empty(), Optional.of(event), Optional.of(repetition)));
    }

    public static EventChangeLogEventJson updated(
            ActorId actor, EventWithRelations oldEvent, RepetitionInstanceInfo oldRepetition,
            EventWithRelations updatedEvent, RepetitionInstanceInfo updatedRepetition) {
        return new EventChangeLogEventJson(actor, EventChangeType.UPDATE, new EventIdLogDataJson(updatedEvent),
                EventChangesJson.find(Optional.of(oldEvent), Optional.of(oldRepetition),
                        Optional.of(updatedEvent), Optional.of(updatedRepetition)));
    }

    public static EventChangeLogEventJson recreated(
            ActorId actor, Option<Instant> recurrenceId,
            EventWithRelations oldEvent, RepetitionInstanceInfo oldRepetition,
            EventWithRelations updatedEvent, RepetitionInstanceInfo updatedRepetition) {
        EventIdLogDataJson idLogData = new EventIdLogDataJson(updatedEvent);
        idLogData = recurrenceId.map(idLogData::withRecurrenceId).getOrElse(idLogData);

        return new EventChangeLogEventJson(actor, EventChangeType.RECREATE, idLogData,
                EventChangesJson.find(Optional.of(oldEvent), Optional.of(oldRepetition),
                        Optional.of(updatedEvent), Optional.of(updatedRepetition)));
    }

    public static EventChangeLogEventJson tailed(
            ActorId actor, EventWithRelations oldEvent, RepetitionInstanceInfo oldRepetition,
            EventWithRelations updatedEvent, RepetitionInstanceInfo updatedRepetition) {
        return new EventChangeLogEventJson(actor, EventChangeType.TAIL, new EventIdLogDataJson(updatedEvent),
                EventChangesJson.find(Optional.of(oldEvent), Optional.of(oldRepetition),
                        Optional.of(updatedEvent), Optional.of(updatedRepetition)));
    }

    public static EventChangeLogEventJson deleted(
            ActorId actor, EventWithRelations event, RepetitionInstanceInfo repetition) {
        return deleted(actor, event.getExternalId(), new EventAndRepetition(event.getEvent(), repetition));
    }

    public static EventChangeLogEventJson deleted(ActorId actor, String externalId, EventAndRepetition event) {
        return deleted(actor, externalId, event, EventChangeType.DELETE);
    }

    public static EventChangeLogEventJson deletedSpam(String externalId, EventAndRepetition event) {
        return deleted(ActorId.yaCalendar(), externalId, event, EventChangeType.DELETE_SPAM);
    }

    private static EventChangeLogEventJson deleted(
            ActorId actor, String externalId, EventAndRepetition event, EventChangeType changeType
    ) {
        val changes = new EventChangesJson(
                EventFieldsChangesJson.of(Optional.empty(), event.getEvent()),
                RepetitionChangesJson.find(Optional.of(event.getRepetitionInfo()), Optional.empty()),
                ResourcesChangesJson.empty(), UsersChangesJson.empty(), LayersChangesJson.empty());

        return new EventChangeLogEventJson(actor, changeType,
                new EventIdLogDataJson(externalId, event.getEvent()), changes);
    }

    public static EventChangeLogEventJson updated(
            ActorId actor, EventIdLogDataJson eventId, EventParticipants oldParticipants,
            EventParticipants curParticipants, ListF<EventAttachedLayerId> layerIds) {
        return updated(actor, eventId, EventChangesJson.empty()
                .withResources(ResourcesChangesJson.find(Optional.of(oldParticipants), Optional.of(curParticipants)))
                .withUsers(UsersChangesJson.find(Optional.of(oldParticipants), Optional.of(curParticipants)))
                .withLayers(LayersChangesJson.of(layerIds)));
    }

    public static EventChangeLogEventJson updated(
            ActorId actor, EventIdLogDataJson eventId, EventAttachedLayerId layerId) {
        return updated(actor, eventId, EventChangesJson.empty()
                .withUsersAndLayers(new UserRelatedChangesJson(UsersChangesJson.empty(), layerId)));
    }

    public static EventChangeLogEventJson updated(
            ActorId actor, EventIdLogDataJson eventId, Email userEmail,
            Option<EventUser> oldEu, Option<EventUser> curEu, Option<EventAttachedLayerId> layerId) {
        return updated(actor, eventId, EventChangesJson.empty()
                .withUsersAndLayers(UserRelatedChangesJson.find(userEmail, oldEu.toOptional(), curEu.toOptional(), layerId.toOptional())));
    }

    public static EventChangeLogEventJson updated(ActorId actor, EventIdLogDataJson eventId, EventChangesJson changes) {
        return new EventChangeLogEventJson(actor, EventChangeType.UPDATE, eventId, changes);
    }
}
