package ru.yandex.calendar.logic.log.change;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonUnwrapped;
import lombok.val;

import ru.yandex.calendar.logic.event.EventLayerWithRelations;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.log.change.changes.EventFieldsChangesJson;
import ru.yandex.calendar.logic.log.change.changes.LayersChangesJson;
import ru.yandex.calendar.logic.log.change.changes.RepetitionChangesJson;
import ru.yandex.calendar.logic.log.change.changes.ResourcesChangesJson;
import ru.yandex.calendar.logic.log.change.changes.UserRelatedChangesJson;
import ru.yandex.calendar.logic.log.change.changes.UsersChangesJson;

@JsonInclude(JsonInclude.Include.NON_EMPTY)
public class EventChangesJson {
    @JsonUnwrapped
    public final EventFieldsChangesJson event;
    public final Optional<RepetitionChangesJson> repetition;
    public final Optional<ResourcesChangesJson> resources;
    public final Optional<UsersChangesJson> users;
    public final Optional<LayersChangesJson> layers;

    public EventChangesJson(
            EventFieldsChangesJson event, RepetitionChangesJson repetition,
            ResourcesChangesJson resources, UsersChangesJson users, LayersChangesJson layers) {
        this.event = event;
        this.repetition = repetition.toOptional();
        this.resources = resources.toOptional();
        this.users = users.toOptional();
        this.layers = layers.toOptional();
    }

    public EventChangesJson(
            EventFieldsChangesJson event, Optional<RepetitionChangesJson> repetition,
            Optional<ResourcesChangesJson> resources, Optional<UsersChangesJson> users, Optional<LayersChangesJson> layers) {
        this.event = event;
        this.repetition = repetition;
        this.resources = resources;
        this.users = users;
        this.layers = layers;
    }

    public static EventChangesJson empty() {
        return new EventChangesJson(
                EventFieldsChangesJson.empty(), RepetitionChangesJson.empty(),
                ResourcesChangesJson.empty(), UsersChangesJson.empty(), LayersChangesJson.empty());
    }

    public static EventChangesJson find(
            Optional<EventWithRelations> oldEvent, Optional<RepetitionInstanceInfo> oldRepetition,
            Optional<EventWithRelations> curEvent, Optional<RepetitionInstanceInfo> curRepetition) {
        val oldParticipants = oldEvent.map(EventWithRelations::getEventParticipants);
        val curParticipants = curEvent.map(EventWithRelations::getEventParticipants);

        val oldLayers = oldEvent.map(v -> (List<EventLayerWithRelations>) v.getEventLayersWithRelations()).orElse(new ArrayList<>());
        val curLayers = curEvent.map(v -> (List<EventLayerWithRelations>) v.getEventLayersWithRelations()).orElse(new ArrayList<>());

        return new EventChangesJson(
                EventFieldsChangesJson.find(oldEvent, curEvent),
                RepetitionChangesJson.find(oldRepetition, curRepetition),
                ResourcesChangesJson.find(oldParticipants, curParticipants),
                UsersChangesJson.find(oldParticipants, curParticipants),
                LayersChangesJson.find(oldLayers, curLayers));
    }

    public boolean isEmpty() {
        return event.isEmpty() && !repetition.isPresent() && !resources.isPresent() && !users.isPresent() && !layers.isPresent();
    }

    public EventChangesJson withEvent(EventFieldsChangesJson event) {
        return new EventChangesJson(event, repetition, resources, users, layers);
    }

    public EventChangesJson withRepetition(RepetitionChangesJson repetition) {
        return new EventChangesJson(event, repetition.toOptional(), resources, users, layers);
    }

    public EventChangesJson withResources(ResourcesChangesJson resources) {
        return new EventChangesJson(event, repetition, resources.toOptional(), users, layers);
    }

    public EventChangesJson withUsers(UsersChangesJson users) {
        return new EventChangesJson(event, repetition, resources, users.toOptional(), layers);
    }

    public EventChangesJson withLayers(LayersChangesJson layers) {
        return new EventChangesJson(event, repetition, resources, users, layers.toOptional());
    }

    public EventChangesJson withUsersAndLayers(UserRelatedChangesJson changes) {
        val layersChanges = changes.layerChanges.map(LayersChangesJson::of).orElse(LayersChangesJson.empty());
        return new EventChangesJson(event, repetition, resources,
                changes.userChanges.toOptional(), layersChanges.toOptional());
    }
}
