package ru.yandex.calendar.logic.log.change.changes;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.annotation.JsonInclude;
import lombok.val;

import ru.yandex.calendar.logic.event.EventAttachedLayerId;
import ru.yandex.calendar.logic.event.EventLayerWithRelations;
import ru.yandex.inside.passport.PassportUid;

@JsonInclude(JsonInclude.Include.NON_EMPTY)
public class LayersChangesJson {
    private List<LayerIdentJson> removed;
    private List<LayerIdentJson> added;

    public LayersChangesJson(List<LayerIdentJson> removed, List<LayerIdentJson> added) {
        this.removed = removed;
        this.added = added;
    }

    public static LayersChangesJson empty() {
        return new LayersChangesJson(new ArrayList<>(), new ArrayList<>());
    }

    public static LayersChangesJson of(EventAttachedLayerId layerId) {
        return of(Arrays.asList(layerId));
    }

    private static List<LayerIdentJson> convertLayerIdsToLayerIdents(List<EventAttachedLayerId> layerIds, Function<EventAttachedLayerId, Optional<Long>> extractor) {
        return layerIds.stream().flatMap(layerId -> extractor.apply(layerId).map(Stream::of).orElseGet(Stream::empty))
                .map(id -> new LayerIdentJson(id, Optional.empty())).collect(Collectors.toList());
    }

    public static LayersChangesJson of(List<EventAttachedLayerId> layerIds) {
        return new LayersChangesJson(convertLayerIdsToLayerIdents(layerIds, (layerId) -> layerId.getDetachedId().toOptional()),
                convertLayerIdsToLayerIdents(layerIds, (layerId) -> layerId.getAttachedId().toOptional()));
    }

    private static List<LayerIdentJson> filterAndSortLayerIdents(Map<Long, LayerIdentJson> res, Map<Long, LayerIdentJson> toFilter) {
        return res.values().stream().filter(i -> !toFilter.containsKey(i.getId())).sorted(Comparator.comparingLong(LayerIdentJson::getId)).collect(Collectors.toList());
    }

    public static LayersChangesJson find(List<EventLayerWithRelations> old, List<EventLayerWithRelations> cur) {
        val olds = old.stream().map(LayerIdentJson::cons).collect(Collectors.toMap(LayerIdentJson::getId, v -> v));
        val curs = cur.stream().map(LayerIdentJson::cons).collect(Collectors.toMap(LayerIdentJson::getId, v -> v));

        return new LayersChangesJson(filterAndSortLayerIdents(olds, curs), filterAndSortLayerIdents(curs, olds));
    }

    public Optional<LayersChangesJson> toOptional() {
        return isEmpty() ? Optional.empty() : Optional.of(this);
    }

    public boolean isEmpty() {
        return removed.isEmpty() && added.isEmpty();
    }

    public static class LayerIdentJson {
        public final long id;
        public final Optional<Long> uid;

        public LayerIdentJson(long id, Optional<PassportUid> uid) {
            this.id = id;
            this.uid = uid.map(v -> v.toUidOrZero().getUid());
        }

        public static LayerIdentJson cons(EventLayerWithRelations el) {
            return new LayerIdentJson(el.getLayerId(), Optional.of(el.getLayerCreatorUid()));
        }

        public long getId() {
            return id;
        }
    }
}
