package ru.yandex.calendar.logic.log.change.changes;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import lombok.val;

import ru.yandex.bolts.collection.impl.ArrayListF;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.sharing.participant.EventParticipants;
import ru.yandex.misc.email.Email;

@JsonInclude(JsonInclude.Include.NON_EMPTY)
public class ResourcesChangesJson {
    private List<ResourceIdentJson> removed;
    private List<ResourceIdentJson> added;

    public ResourcesChangesJson(List<ResourceIdentJson> removed, List<ResourceIdentJson> added) {
        this.removed = removed;
        this.added = added;
    }

    public static ResourcesChangesJson empty() {
        return new ResourcesChangesJson(new ArrayList<>(), new ArrayList<>());
    }

    private static Map<Email, ResourceIdentJson> convertEventParticipantsToMap(Optional<EventParticipants> ep) {
        return ep.map(EventParticipants::getResources)
                .orElse(new ArrayListF<>()).stream()
                .collect(Collectors.toMap(ResourceInfo::getEmail, ResourceIdentJson::cons));
    }

    private static List<ResourceIdentJson> getChanges(Map<Email, ResourceIdentJson> res, Map<Email, ResourceIdentJson> toFilter) {
        return res.values().stream().filter(i -> !toFilter.containsKey(i.email))
                .sorted(Comparator.comparing(ResourceIdentJson::getEmailValue)).collect(Collectors.toList());
    }

    public static ResourcesChangesJson find(Optional<EventParticipants> old, Optional<EventParticipants> cur) {
        val olds = convertEventParticipantsToMap(old);
        val curs = convertEventParticipantsToMap(cur);
        return new ResourcesChangesJson(getChanges(olds, curs), getChanges(curs, olds));
    }

    public Optional<ResourcesChangesJson> toOptional() {
        return isEmpty() ? Optional.empty() : Optional.of(this);
    }

    public boolean isEmpty() {
        return removed.isEmpty() && added.isEmpty();
    }

    public static class ResourceIdentJson {
        public final long id;
        @JsonIgnore
        public final Email email;

        public ResourceIdentJson(long id, Email email) {
            this.id = id;
            this.email = email;
        }

        public static ResourceIdentJson cons(ResourceInfo resource) {
            return new ResourceIdentJson(resource.getResourceId(), resource.getEmail());
        }

        @JsonGetter("email")
        public String getEmailValue() {
            return email.getEmail();
        }
    }
}
