package ru.yandex.calendar.logic.log.change.changes;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.annotation.JsonInclude;
import lombok.val;

import ru.yandex.calendar.logic.beans.generated.EventInvitation;
import ru.yandex.calendar.logic.event.EventUserWithRelations;
import ru.yandex.calendar.logic.sharing.participant.EventParticipants;

@JsonInclude(JsonInclude.Include.NON_EMPTY)
public class UsersChangesJson {
    private final List<UserChangesJson> removed;
    private final List<UserChangesJson> updated;
    private final List<UserChangesJson> added;

    public UsersChangesJson(
            List<UserChangesJson> removed,
            List<UserChangesJson> updated,
            List<UserChangesJson> added) {
        this.removed = removed;
        this.updated = updated;
        this.added = added;
    }

    public static UsersChangesJson empty() {
        return new UsersChangesJson(new ArrayList<>(), new ArrayList<>(), new ArrayList<>());
    }

    private static List<UserChangesJson> concatAndSortChanges(List<UserChangesJson> old, List<UserChangesJson> cur) {
        return Stream.concat(old.stream(), cur.stream()).sorted(UserChangesJson.comparator).collect(Collectors.toList());
    }

    public static UsersChangesJson find(Optional<EventParticipants> old, Optional<EventParticipants> cur) {
        UsersChangesJson eventUsers = find(old, cur,
                ep -> ep.getEventUsers().toMapMappingToKey(EventUserWithRelations::getUid), UserChangesJson::of);

        UsersChangesJson invitations = find(old, cur,
                ep -> ep.getInvitations().toMapMappingToKey(EventInvitation::getEmail), UserChangesJson::of);

        return new UsersChangesJson(concatAndSortChanges(eventUsers.removed, invitations.removed),
                concatAndSortChanges(eventUsers.updated, invitations.updated),
                concatAndSortChanges(eventUsers.added, invitations.added));
    }

    private static <K, V> List<UserChangesJson> collectUpdated(Map<K, V> oldMap, Map<K, V> newMap, BiFunction<Optional<V>, V, UserChangesJson> cons) {
        return oldMap.keySet().stream().filter(newMap::containsKey)
                .map(v -> cons.apply(Optional.of(oldMap.get(v)), newMap.get(v)))
                .filter(v -> !v.isEmpty())
                .collect(Collectors.toList());
    }

    private static <K, V> List<UserChangesJson> collectChanges(Map<K, V> map, Map<K, V> filterMap, BiFunction<Optional<V>, V, UserChangesJson> cons) {
        return map.keySet().stream()
                .filter(v -> !filterMap.containsKey(v))
                .map(v -> cons.apply(Optional.empty(), map.get(v)))
                .collect(Collectors.toList());
    }

    private static <S, K, V> UsersChangesJson find(
            Optional<S> old, Optional<S> cur, Function<S, Map<K, V>> extractor, BiFunction<Optional<V>, V, UserChangesJson> cons) {
        val oldMap = old.map(extractor).orElse(new HashMap<>());
        val curMap = cur.map(extractor).orElse(new HashMap<>());

        val updated = collectUpdated(oldMap, curMap, cons);
        val added = collectChanges(curMap, oldMap, cons);
        val removed = collectChanges(oldMap, curMap, cons);

        return new UsersChangesJson(removed, updated, added);
    }

    public Optional<UsersChangesJson> toOptional() {
        return isEmpty() ? Optional.empty() : Optional.of(this);
    }

    public boolean isEmpty() {
        return removed.isEmpty() && updated.isEmpty() && added.isEmpty();
    }
}
